package edu.unika.aifb.kaon.datalog.program;

/**
 * Represents a predicate in the program.
 */
public class Predicate {
    /** Set to <code>true</code> if this predicate is positive under classical negation. */
    protected boolean m_isPositive;
    /** The name of the predicate. */
    protected String m_name;
    /** The arity of this predicate. */
    protected int m_arity;
    /** The full name of the predicate which includes its positive/negative status and the arity. */
    protected String m_fullName;

    /**
     * Creates a predicate.
     *
     * @param isPositive                <code>true</code> if this predicate is positive under classical negation
     * @param name                      the name of the predicate (excludes the positive/negative status and the arity of the predicate)
     * @param arity                     the arity of the predicate
     */
    public Predicate(boolean isPositive,String name,int arity) {
        m_isPositive=isPositive;
        m_name=name;
        m_arity=arity;
        m_fullName=(m_isPositive ? "" : "-")+m_name+"/"+m_arity;
    }
    /**
     * Returns whether this predicate is classically positive.
     *
     * @return                          <code>true</code> if this predicate is classically positive
     */
    public boolean isPositive() {
        return m_isPositive;
    }
    /**
     * Returns the name of the predicate (excluding the arity of the predicate and its positive/negative status).
     *
     * @return                          the name of this predicate
     */
    public String getName() {
        return m_name;
    }
    /**
     * Returns the simple name of the predicate (excluding the arity of the predicate but including its positive/negative status).
     *
     * @return                          the simple name of this predicate
     */
    public String getSimpleName() {
        return (m_isPositive ? "" : "-")+m_name;
    }
    /**
     * Returns the full name of the predicate (including the arity of the predicate).
     *
     * @return                          the full name of the predicate
     */
    public String getFullName() {
        return m_fullName;
    }
    /**
     * Returns the arity of the predicate.
     *
     * @return                          the arity of the predicate
     */
    public int getArity() {
        return m_arity;
    }
    /**
     * Converts this predicate to string.
     *
     * @return                          the string name of this predicate
     */
    public String toString() {
        return m_fullName;
    }
    /**
     * Compares this object to another.
     *
     * @param that                      the other object
     * @return                          <code>true</code> if objects are equal
     */
    public boolean equals(Object that) {
        if (this==that)
            return true;
        if (!(that instanceof Predicate))
            return false;
        return m_fullName.equals(((Predicate)that).m_fullName);
    }
    /**
     * Returns the hash-code of this predicate.
     *
     * @return                          the hash-code of this object
     */
    public int hashCode() {
        return m_fullName.hashCode();
    }
}
