package edu.unika.aifb.kaon.datalog.program;

import java.util.Collection;
import java.util.List;
import java.util.ArrayList;
import java.util.EventListener;
import javax.swing.event.EventListenerList;

/**
 * Represents a mutable logic program consisting of a set of rules. This program can notify listeners about changes in the program.
 */
public class MutableProgram extends Program {
    /** The list of change listeners. */
    protected EventListenerList m_listenerList;

    /**
     * Creates an instance of this class.
     *
     * @param program                       the program from which this program is created
     */
    public MutableProgram(Program program) {
        this((Rule[])program.m_rules.clone());
    }
    /**
     * Creates an instance of this class.
     *
     * @param rules                         the array of rules in the program
     */
    public MutableProgram(Rule[] rules) {
        super(rules);
        m_listenerList=new EventListenerList();
    }
    /**
     * Adds a change listener to this program.
     *
     * @param listener                      the change listener being added
     */
    public void addProgramChangeListener(ProgramChangeListener listener) {
        m_listenerList.add(ProgramChangeListener.class,listener);
    }
    /**
     * Removes a change listener to this program.
     *
     * @param listener                      the change listener being removed
     */
    public void removeProgramChangeListener(ProgramChangeListener listener) {
        m_listenerList.remove(ProgramChangeListener.class,listener);
    }
    /**
     * Changes the program by adding and removing certain rules. The equality of rules is taked to mean instance equality.
     *
     * @param addedRules                    the collection of rules to be added
     * @param removedRules                  the collection of rules to be removed
     */
    public void changeProgram(Collection addedRules,Collection removedRules) {
        List newRules=new ArrayList(m_rules.length+addedRules.size()+removedRules.size());
        for (int i=0;i<m_rules.length;i++)
            newRules.add(m_rules[i]);
        newRules.addAll(addedRules);
        newRules.removeAll(removedRules);
        Rule[] newRulesArray=new Rule[newRules.size()];
        newRules.toArray(newRulesArray);
        m_rules=newRulesArray;
        fireProgramChanged(addedRules,removedRules);
    }
    /**
     * Notifies listeners that the rule was removed.
     *
     * @param addedRules                    the rules that were added
     * @param removedRules                  the rules that were removed
     */
    protected void fireProgramChanged(Collection addedRules,Collection removedRules) {
        Object[] listeners=m_listenerList.getListenerList();
        for (int i=listeners.length-2;i>=0;i-=2)
            if (listeners[i]==ProgramChangeListener.class)
                ((ProgramChangeListener)listeners[i+1]).programChanged(this,addedRules,removedRules);
    }

    /**
     * The listener receiving notifications about the change in the program.
     */
    public static interface ProgramChangeListener extends EventListener {
        /**
         * Called when a program is changed.
         *
         * @param program                   the program that was changed
         * @param addedRules                the rules that were added
         * @param removedRules              the rules that were removed
         */
        void programChanged(MutableProgram program,Collection addedRules,Collection removedRules);
    }
}
