package edu.unika.aifb.kaon.datalog.program;

/**
 * Represents a literal in the rule. A literal consists of a predicate, a pattern representing
 * the variable names or contstants and a flag determining whether the literal is negated.
 */
public class Literal {
    /** The predicate of this literal. */
    protected Predicate m_predicate;
    /** Set to <code>true</code> if this literal is positive. */
    protected boolean m_isPositive;
    /** The terms of the literal. */
    protected Term[] m_terms;
    /** The hash-code of the object. */
    protected int m_hashCode;

    /**
     * Creates a literal.
     *
     * @param predicate                         the predicate
     * @param isPositive                        <code>true</code> if the literal is positive
     * @param terms                             the terms of the literal
     */
    public Literal(Predicate predicate,boolean isPositive,Term[] terms) {
        this(predicate,isPositive,terms,false);
    }
    /**
     * Creates a literal with an additional variable for selecting the information about a disjunction.
     *
     * @param predicate                         the predicate
     * @param isPositive                        <code>true</code> if the literal is positive
     * @param terms                             the terms of the literal
     * @param hasDisjunctionVariable            <code>true</code> if the literal is disjunctive
     */
    public Literal(Predicate predicate,boolean isPositive,Term[] terms,boolean hasDisjunctionVariable) {
        m_predicate=predicate;
        m_isPositive=isPositive;
        m_terms=terms;
        if (m_terms.length!=m_predicate.getArity()+(hasDisjunctionVariable ? 1 : 0))
            throw new IllegalArgumentException("Incorrect number of terms in the literal.");
        m_hashCode=m_predicate.hashCode()*3+(m_isPositive ? 1 : 0);
        for (int i=0;i<m_terms.length;i++)
            m_hashCode=m_hashCode*7+m_terms[i].hashCode();
    }
    /**
     * Returns the predicate of this literal.
     *
     * @return                                  the predicate
     */
    public Predicate getPredicate() {
        return m_predicate;
    }
    /**
     * Returns whether this literal is positive.
     *
     * @return                                  <code>true</code> is this literal is positive
     */
    public boolean isPositive() {
        return m_isPositive;
    }
    /**
     * Returns the arity of the literal.
     *
     * @return                                  the arity of the literal
     */
    public int getArity() {
        return m_terms.length;
    }
    /**
     * Reutrns <code>true</code> if given argument is bound to a variable.
     *
     * @param argumentIndex                     the index of the argument
     * @return                                  <code>true</code> if the argument is bound to a variable
     */
    public boolean isArgumentBoundToVariable(int argumentIndex) {
        return m_terms[argumentIndex] instanceof Variable;
    }
    /**
     * Reutrns <code>true</code> if given argument is bound to a constant.
     *
     * @param argumentIndex                     the index of the argument
     * @return                                  <code>true</code> if the argument is bound to a constant
     */
    public boolean isArgumentBoundToConstant(int argumentIndex) {
        return m_terms[argumentIndex] instanceof Constant;
    }
    /**
     * Reutrns <code>true</code> if given argument is bound to a unary function call.
     *
     * @param argumentIndex                     the index of the argument
     * @return                                  <code>true</code> if the argument is bound to a unary function call
     */
    public boolean isArgumentBoundToUnaryFunctionCall(int argumentIndex) {
        return m_terms[argumentIndex] instanceof UnaryFunctionCall;
    }
    /**
     * Returns the term at given position.
     *
     * @param argumentIndex                     the index of the argument
     * @return                                  the term at given position
     */
    public Term getTerm(int argumentIndex) {
        return m_terms[argumentIndex];
    }
    /**
     * Returns the variable at given position.
     *
     * @param argumentIndex                     the index of the argument
     * @return                                  the variable at given position (or <code>null</code> if given position is not bound to a variable)
     */
    public Variable getArgumentVariable(int argumentIndex) {
        Term term=m_terms[argumentIndex];
        if (term instanceof Variable)
            return (Variable)term;
        else
            return null;
    }
    /**
     * Returns the constant of the given argument.
     *
     * @param argumentIndex                     the index of the argument
     * @return                                  the constant (of <code>null</code> if the argument is not bound to a constant)
     */
    public Constant getArgumentConstant(int argumentIndex) {
        Term term=m_terms[argumentIndex];
        if (term instanceof Constant)
            return (Constant)term;
        else
            return null;
    }
    /**
     * Returns the unary function call at given position.
     *
     * @param argumentIndex                     the index of the argument
     * @return                                  the the unary function at given position (or <code>null</code> if given position is not bound to a unary function call)
     */
    public UnaryFunctionCall getArgumentUnaryFunctionCall(int argumentIndex) {
        Term term=m_terms[argumentIndex];
        if (term instanceof UnaryFunctionCall)
            return (UnaryFunctionCall)term;
        else
            return null;
    }
    /**
     * Returns the array of terms of this literal.
     *
     * @return                                  the array of terms of the literal
     */
    public Term[] getTerms() {
        return m_terms;
    }
    /**
     * Returns <code>true</code> if this literal contains supplied variable.
     *
     * @param variable                          the variable
     * @return                                  <code>true</code> if this literal contains supplied variable
     */
    public boolean containsVariable(Variable variable) {
        for (int i=m_terms.length-1;i>=0;--i)
            if (m_terms[i].equals(variable))
                return true;
        return false;
    }
    /**
     * Converts this literal to a string.
     *
     * @return                                  the string representation of the literal
     */
    public String toString() {
        StringBuffer buffer=new StringBuffer();
        if (!m_isPositive)
            buffer.append("~");
        buffer.append(m_predicate.getSimpleName());
        int arity=m_predicate.getArity();
        if (arity>0) {
            buffer.append("(");
            for (int i=0;i<arity;i++) {
                if (i>0)
                    buffer.append(",");
                buffer.append(m_terms[i].toString());
            }
            buffer.append(")");
        }
        return buffer.toString();
    }
    /**
     * Compares this object to another.
     *
     * @param that                              the other object
     * @return                                  <code>true</code> if the objects are equal
     */
    public boolean equals(Object that) {
        if (this==that)
            return true;
        if (!(that instanceof Literal))
            return false;
        Literal thatLiteral=(Literal)that;
        if (!m_predicate.equals(thatLiteral.getPredicate()) || m_isPositive!=thatLiteral.m_isPositive || m_terms.length!=thatLiteral.m_terms.length)
            return false;
        for (int i=m_terms.length-1;i>=0;--i)
            if (!m_terms[i].equals(thatLiteral.m_terms[i]))
                return false;
        return true;
    }
    /**
     * Returns the hash-code of this object.
     *
     * @return                                  the hash-code of the object
     */
    public int hashCode() {
        return m_hashCode;
    }
}
