package edu.unika.aifb.kaon.datalog.parser;

import java.util.List;
import java.util.LinkedList;
import java.util.Set;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Collections;

import edu.unika.aifb.kaon.datalog.*;
import edu.unika.aifb.kaon.datalog.program.*;
import edu.unika.aifb.kaon.datalog.evaluation.*;

/**
 * Default consumer that tracks all facts in the database.
 */
public class DefaultDatalogParserConsumer implements DatalogParserConsumer {
    /** The predicate factory. */
    protected PredicateFactory m_predicateFactory;
    /** The extensional manager for receiving facts. */
    protected ExtensionalManager m_extensionalManager;
    /** The name of the database for facts. */
    protected String m_newFactsPredicatesDatabaseName;
    /** The list of rules. */
    protected List m_rules;
    /** The list of facts. */
    protected List m_facts;
    /** The set of predicates encountered in facts. */
    protected Set m_predicatesInFacts;
    /** The set of predicates encountered in disjunctions. */
    protected Set m_predicatesInDisjunctiveFacts;

    /**
     * Creates an instsance of this class.
     *
     * @param predicateFactory                  the predicate factory
     * @param extensionalManager                the extensional manager (can be <code>null</code> - facts are then only collected internally)
     * @param newFactsPredicatesDatabaseName    the name of the database receiving new predicates encountered in facts
     */
    public DefaultDatalogParserConsumer(PredicateFactory predicateFactory,ExtensionalManager extensionalManager,String newFactsPredicatesDatabaseName) {
        m_predicateFactory=predicateFactory;
        m_extensionalManager=extensionalManager;
        m_newFactsPredicatesDatabaseName=newFactsPredicatesDatabaseName;
    }
    /**
     * Returns the predicate for given paramters.
     *
     * @param isPositive                        <code>true</code> if predicate is classically positive
     * @param name                              the name of the predicate (without positive/negative status and arity)
     * @param arity                             the arity of the predicate
     * @return                                  the predicate
     * @throws ParseException                   thrown if parsing should be stopped
     */
    public Predicate getPredicate(boolean isPositive,String name,int arity) {
        return m_predicateFactory.getPredicate(isPositive,name,arity);
    }
    /**
     * Called when parsing is started.
     *
     * @throws ParseException                   thrown if parsing should be stopped
     */
    public void startParsing() {
        m_rules=new LinkedList();
        m_facts=new LinkedList();
        m_predicatesInFacts=new HashSet();
        m_predicatesInDisjunctiveFacts=new HashSet();
    }
    /**
     * Called when parsing is done.
     *
     * @throws ParseException                   thrown if parsing should be stopped
     */
    public void doneParsing() throws ParseException {
        try {
            if (m_extensionalManager!=null) {
                Iterator iterator=m_predicatesInFacts.iterator();
                while (iterator.hasNext()) {
                    Predicate predicate=(Predicate)iterator.next();
                    ExtensionalDatabase extensionalDatabase=m_extensionalManager.getExtensionalDatabase(predicate);
                    if (extensionalDatabase==null)
                        extensionalDatabase=m_extensionalManager.getExtensionalDatabase(m_newFactsPredicatesDatabaseName);
                    if (!(extensionalDatabase instanceof MemoryExtensionalDatabase))
                        throw new ParseException("Database for predicate '"+predicate.getFullName()+"' is not a memory database.");
                    MemoryExtensionalDatabase memoryExtensionalDatabase=(MemoryExtensionalDatabase)extensionalDatabase;
                    MemoryPredicateExtension memoryPredicateExtension=memoryExtensionalDatabase.createPredicateExtension(predicate);
                    memoryPredicateExtension.indexAll();
                    memoryPredicateExtension.setContainsDisjunctionInfo(m_predicatesInDisjunctiveFacts.contains(predicate));
                }
                m_extensionalManager.bulkChange(m_facts,Collections.EMPTY_LIST);
            }
        }
        catch (DatalogException e) {
            ParseException pe=new ParseException(e.getMessage());
            pe.initCause(e);
            throw pe;
        }
    }
    /**
     * Called when a rule has been parsed.
     *
     * @param rule                              the rule
     * @throws ParseException                   thrown if parsing should be stopped
     */
    public void consumeRule(Rule rule) {
        m_rules.add(rule);
    }
    /**
     * Called when a fact has been parsed.
     *
     * @param disjunction                       the disjunction
     * @throws ParseException                   thrown if parsing should be stopped
     */
    public void consumeFact(LiteralValue[] disjunction) {
        m_facts.add(disjunction);
        if (disjunction.length>1) {
            for (int i=0;i<disjunction.length;i++) {
                m_predicatesInFacts.add(disjunction[i].getPredicate());
                m_predicatesInDisjunctiveFacts.add(disjunction[i].getPredicate());
            }
        }
        else {
            for (int i=0;i<disjunction.length;i++)
                m_predicatesInFacts.add(disjunction[i].getPredicate());
        }
    }
    /**
     * Returns the parsed program.
     *
     * @return                                  the parsed program
     */
    public Program getProgram() {
        Rule[] rulesArray=new Rule[m_rules.size()];
        m_rules.toArray(rulesArray);
        return new Program(rulesArray);
    }
    /**
     * Returns the parsed facts.
     *
     * @return                                  the parsed facts
     */
    public LiteralValue[][] getFacts() {
        LiteralValue[][] factsArray=new LiteralValue[m_facts.size()][];
        m_facts.toArray(factsArray);
        return factsArray;
    }
    /**
     * Returns the list or rules
     *
     * @return                                  the list of rules
     */
    public List getRules() {
        return m_rules;
    }
}
