package edu.unika.aifb.kaon.datalog.materialization;

import java.util.Set;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Collection;
import java.util.Collections;

import edu.unika.aifb.kaon.datalog.*;
import edu.unika.aifb.kaon.datalog.program.*;
import edu.unika.aifb.kaon.datalog.evaluation.*;

/**
 * A manager for materialized predicates.
 */
public class MaterializationManager {
    /** The extensional manager managing extensions of predicates. */
    protected ExtensionalManager m_extensionalManager;
    /** The listener for the extensional manager. */
    protected ExtensionalManager.ExensionChangeListener m_extensionChangeListener;
    /** The key under which auxiliary database has been registered. */
    protected String m_auxiliaryPredicatesDatabaseKey;
    /** The memory extensional database containing auxiliary prediactes (all but materialization predicates). */
    protected MemoryExtensionalDatabase m_auxiliaryPredicatesDatabase;
    /** The manager of the maintenance program. */
    protected MaintenanceProgramManager m_maintenanceProgramManager;
    /** The mutable program that this class is managing - may be <code>null</code>. */
    protected MutableProgram m_materializedProgram;
    /** The listener to the program. */
    protected MutableProgram.ProgramChangeListener m_programChangeListener;

    /**
     * Creates an instance of this class with no program to manage. Use changeRules() to add or remove
     * maintained rules.
     *
     * @param extensionalManager                        the extensional manager
     * @param predicateFactory                          the predicate factory
     * @throws DatalogException                         thrown if there is an error
     */
    public MaterializationManager(ExtensionalManager extensionalManager,PredicateFactory predicateFactory) throws DatalogException {
        this(extensionalManager,predicateFactory,null);
    }
    /**
     * Creates an instance of this class.
     *
     * @param extensionalManager                        the extensional manager
     * @param predicateFactory                          the predicate factory
     * @param materializedProgram                       the materialized program
     * @throws DatalogException                         thrown if there is an error
     */
    public MaterializationManager(ExtensionalManager extensionalManager,PredicateFactory predicateFactory,MutableProgram materializedProgram) throws DatalogException {
        m_extensionalManager=extensionalManager;
        m_maintenanceProgramManager=new MaintenanceProgramManager(predicateFactory);
        m_auxiliaryPredicatesDatabaseKey="materialization_auxiliary_predicates:"+System.currentTimeMillis();
        m_auxiliaryPredicatesDatabase=new MemoryExtensionalDatabase();
        m_extensionalManager.registerExtensionalDatabase(m_auxiliaryPredicatesDatabaseKey,m_auxiliaryPredicatesDatabase);
        m_extensionChangeListener=new ExtensionalManager.ExensionChangeListener() {
            public void extensionsWillChange(ExtensionalManager extensionalManager,Collection changeInfos) {
                try {
                    updateMaterialization(changeInfos);
                }
                catch (DatalogException ignored) {
                }
            }
            public void extensionsChanged(ExtensionalManager extensionalManager,Collection changeInfos) {
            }
        };
        m_extensionalManager.addExensionChangeListener(m_extensionChangeListener);
        m_materializedProgram=materializedProgram;
        if (m_materializedProgram!=null) {
            m_programChangeListener=new MutableProgram.ProgramChangeListener() {
                public void programChanged(MutableProgram program,Collection addedRules,Collection removedRules) {
                    try {
                        changeRules(addedRules,removedRules);
                    }
                    catch (DatalogException ignored) {
                    }
                }
            };
            m_materializedProgram.addProgramChangeListener(m_programChangeListener);
            addProgramRules(m_materializedProgram);
        }
    }
    /**
     * Deletes this manager.
     */
    public void dispose() {
        if (m_materializedProgram!=null) {
            m_materializedProgram.removeProgramChangeListener(m_programChangeListener);
            m_materializedProgram=null;
            m_programChangeListener=null;
        }
        m_extensionalManager.registerExtensionalDatabase(m_auxiliaryPredicatesDatabaseKey,null);
        m_extensionalManager.removeExensionChangeListener(m_extensionChangeListener);
        m_extensionalManager=null;
        m_maintenanceProgramManager=null;
        m_auxiliaryPredicatesDatabaseKey=null;
        m_auxiliaryPredicatesDatabase=null;
    }
    /**
     * Notifies the maintenance manager that some extensions will change and that materialization should be updated.
     *
     * @param changeInfos                               contains information about the changed extensions
     * @throws DatalogException                         thrown if there is an error
     */
    public void updateMaterialization(Collection changeInfos) throws DatalogException {
        try {
            Iterator iterator=changeInfos.iterator();
            while (iterator.hasNext()) {
                ExtensionalManager.ChangeInfo element=(ExtensionalManager.ChangeInfo)iterator.next();
                if (element.m_disjunctionRestInfo!=null)
                    throw new DatalogException("Materialization is not supported for disjunctive programs.");
                Object tuple[]=element.m_minimalTuple;
                Predicate predicate=element.m_minimalPredicateExtension.getPredicate();
                Predicate updatePredicate=element.m_isAddition ? m_maintenanceProgramManager.getInsPredicate(predicate) : m_maintenanceProgramManager.getDelPredicate(predicate);
                MemoryPredicateExtension extensionToChange=m_auxiliaryPredicatesDatabase.getMemoryPredicateExtension(updatePredicate);
                if (extensionToChange!=null)
                    extensionToChange.addTuple(tuple);
            }
            executeProgramUpdateMaterialization(m_maintenanceProgramManager.getMaintenanceProgram());
        }
        finally {
            m_auxiliaryPredicatesDatabase.clear();
        }
    }
    /**
     * Notifies the maintenance manager that some rules will change ange and that materialization should be updated.
     *
     * @param addedRules                                the collection of rules to add
     * @param removedRules                              the collection of rules to remove
     * @throws DatalogException                         thrown if there is an error
     */
    public void changeRules(Collection addedRules,Collection removedRules) throws DatalogException {
        Set changedPredicates=new HashSet();
        Iterator iterator=addedRules.iterator();
        while (iterator.hasNext()) {
            Rule rule=(Rule)iterator.next();
            if (!rule.isHorn())
                throw new DatalogException("Non-horn rules are not supported.");
            changedPredicates.add(rule.getHeadLiteral().getPredicate());
        }
        iterator=removedRules.iterator();
        while (iterator.hasNext()) {
            Rule rule=(Rule)iterator.next();
            if (!rule.isHorn())
                throw new DatalogException("Non-horn rules are not supported.");
            changedPredicates.add(rule.getHeadLiteral().getPredicate());
        }
        iterator=addedRules.iterator();
        while (iterator.hasNext()) {
            Rule rule=(Rule)iterator.next();
            Iterator addedPredicates=m_maintenanceProgramManager.addRule(rule).iterator();
            while (addedPredicates.hasNext()) {
                Predicate addedPredicate=(Predicate)addedPredicates.next();
                m_auxiliaryPredicatesDatabase.createPredicateExtension(addedPredicate);
            }
        }
        iterator=removedRules.iterator();
        while (iterator.hasNext()) {
            Rule rule=(Rule)iterator.next();
            m_maintenanceProgramManager.removeRule(rule);
        }
        Program program=m_maintenanceProgramManager.getMaintenanceProgram(changedPredicates);
        executeProgramUpdateMaterialization(program);
        iterator=m_maintenanceProgramManager.purge().iterator();
        while (iterator.hasNext()) {
            Predicate predicate=(Predicate)iterator.next();
            m_auxiliaryPredicatesDatabase.removePredicateExtension(predicate);
        }
    }
    /**
     * Executes the given maintenance program and upadtes materializations.
     *
     * @param maintenanceProgram                        the program to execute
     * @throws DatalogException                         thrown if there is an error
     */
    protected void executeProgramUpdateMaterialization(Program maintenanceProgram) throws DatalogException {
        try {
            ProgramCompiler programCompiler=new ProgramCompiler(m_extensionalManager,maintenanceProgram,Collections.EMPTY_MAP);
            Evaluator evaluator=programCompiler.getEvaluator();
            evaluator.execute();
            Iterator iterator=m_maintenanceProgramManager.getPredicates();
            while (iterator.hasNext()) {
                Predicate predicate=(Predicate)iterator.next();
                ExtensionalDatabase predicateDatabase=m_extensionalManager.getExtensionalDatabase(predicate);
                if (predicateDatabase instanceof MemoryExtensionalDatabase) {
                    MemoryPredicateExtension materializedPredicateExtension=((MemoryExtensionalDatabase)predicateDatabase).getMemoryPredicateExtension(predicate);
                    MemoryPredicateExtension plusExtension=m_auxiliaryPredicatesDatabase.getMemoryPredicateExtension(m_maintenanceProgramManager.getPlusPredicate(predicate));
                    Iterator tuples=plusExtension.selectTuples(new Object[0],new int[0]);
                    while (tuples.hasNext()) {
                        Object[] tuple=(Object[])tuples.next();
                        if (!materializedPredicateExtension.addTuple(tuple))
                            throw new DatalogException("Internal error: plus tuple is already in the materialization.");
                    }
                    MemoryPredicateExtension minusExtension=m_auxiliaryPredicatesDatabase.getMemoryPredicateExtension(m_maintenanceProgramManager.getMinusPredicate(predicate));
                    tuples=minusExtension.selectTuples(new Object[0],new int[0]);
                    while (tuples.hasNext()) {
                        Object[] tuple=(Object[])tuples.next();
                        if (materializedPredicateExtension.removeTuple(tuple)==null)
                            throw new DatalogException("Internal error: minus tuple not in the materialization.");
                    }
                }
            }
        }
        finally {
            m_auxiliaryPredicatesDatabase.clear();
        }
    }
    /**
     * Adds all rules from given program to the materialization.
     *
     * @param program                                   the program for which the rules are added
     * @throws DatalogException                         thrown if there is an error
     */
    public void addProgramRules(Program program) throws DatalogException {
        if (!program.isHorn())
            throw new DatalogException("Materialization of non-Horn programs is not supported.");
        Set addedRules=new HashSet();
        for (int i=0;i<program.getNumberOfRules();i++) {
            Rule rule=program.getRule(i);
            addedRules.add(rule);
        }
        changeRules(addedRules,Collections.EMPTY_SET);
    }
}
