package edu.unika.aifb.kaon.datalog.materialization;

import java.util.Map;
import java.util.HashMap;
import java.util.Set;
import java.util.HashSet;
import java.util.List;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;

import edu.unika.aifb.kaon.datalog.program.*;

/**
 * A manager for mainenance program.
 */
public class MaintenanceProgramManager {
    /** The predicate factory to use for generating predicates. */
    protected PredicateFactory m_predicateFactory;
    /** The map of PredicateInfo objects indexed by the predicate. */
    protected Map m_predicateInfosByPredicate;
    /** The last maintenance program, or <code>null</code> if maintenance program hasn't been created. */
    protected Program m_maintenanceProgram;

    /**
     * Creates an instance of this class.
     *
     * @param predicateFactory                          the predicate factory
     */
    public MaintenanceProgramManager(PredicateFactory predicateFactory) {
        m_predicateFactory=predicateFactory;
        m_predicateInfosByPredicate=new HashMap();
    }
    /**
     * Returns the iterator of all predicates.
     *
     * @return                                          the iterator of all predicates
     */
    public Iterator getPredicates() {
        return m_predicateInfosByPredicate.keySet().iterator();
    }
    /**
     * Returns PredicateInfo for given predicate.
     *
     * @param predicate                                 the predicate
     * @return                                          PredicateInfo object with information about given predicate
     */
    protected PredicateInfo getPredicateInfo(Predicate predicate) {
        PredicateInfo predicateInfo=(PredicateInfo)m_predicateInfosByPredicate.get(predicate);
        if (predicateInfo==null) {
            predicateInfo=new PredicateInfo(predicate);
            m_predicateInfosByPredicate.put(predicate,predicateInfo);
            predicateInfo.initialize();
        }
        return predicateInfo;
    }
    /**
     * Returns a p^Del predicate for given predicate.
     *
     * @param predicate                                 the predicate
     * @return                                          the p^Del predicate for given predicate
     */
    public Predicate getDelPredicate(Predicate predicate) {
        return getPredicateInfo(predicate).m_delPredicate;
    }
    /**
     * Returns a p^Ins predicate for given predicate.
     *
     * @param predicate                                 the predicate
     * @return                                          the p^Del predicate for given predicate
     */
    public Predicate getInsPredicate(Predicate predicate) {
        return getPredicateInfo(predicate).m_insPredicate;
    }
    /**
     * Returns a p^Red predicate for given predicate.
     *
     * @param predicate                                 the predicate
     * @return                                          the p^Del predicate for given predicate
     */
    public Predicate getRedPredicate(Predicate predicate) {
        return getPredicateInfo(predicate).m_redPredicate;
    }
    /**
     * Returns a p^New predicate for given predicate.
     *
     * @param predicate                                 the predicate
     * @return                                          the p^Del predicate for given predicate
     */
    public Predicate getNewPredicate(Predicate predicate) {
        return getPredicateInfo(predicate).m_newPredicate;
    }
    /**
     * Returns a p^Plus predicate for given predicate.
     *
     * @param predicate                                 the predicate
     * @return                                          the p^Del predicate for given predicate
     */
    public Predicate getPlusPredicate(Predicate predicate) {
        return getPredicateInfo(predicate).m_plusPredicate;
    }
    /**
     * Returns a p^Minus predicate for given predicate.
     *
     * @param predicate                                 the predicate
     * @return                                          the p^Del predicate for given predicate
     */
    public Predicate getMinusPredicate(Predicate predicate) {
        return getPredicateInfo(predicate).m_minusPredicate;
    }
    /**
     * Adds a rule to this manager.
     *
     * @param rule                                      the rule to be added
     * @return                                          the set of predicates that were added to the manager as the consequence of this addition
     */
    public Set addRule(Rule rule) {
        m_maintenanceProgram=null;
        PredicateInfo predicateInfo=getPredicateInfo(rule.getHeadLiteral().getPredicate());
        predicateInfo.addRule(rule);
        // now update the supports for the predicates
        Set addedPredicates=new HashSet();
        if (predicateInfo.m_predicateSupport.isEmpty())
            predicateInfo.dumpModifiedPredicates(addedPredicates);
        predicateInfo.m_predicateSupport.add(rule);
        for (int i=0;i<rule.getBodyLength();i++) {
            PredicateInfo bodyPredicateInfo=getPredicateInfo(rule.getBodyLiteral(i).getPredicate());
            if (bodyPredicateInfo.m_predicateSupport.isEmpty())
                bodyPredicateInfo.dumpModifiedPredicates(addedPredicates);
            bodyPredicateInfo.m_predicateSupport.add(rule);
        }
        return addedPredicates;
    }
    /**
     * Removes a rule from this manager.
     *
     * @param rule                                      the rule to be removed
     */
    public void removeRule(Rule rule) {
        m_maintenanceProgram=null;
        PredicateInfo predicateInfo=getPredicateInfo(rule.getHeadLiteral().getPredicate());
        predicateInfo.removeRule(rule);
        // now update the supports for the predicates
        predicateInfo.m_predicateSupport.remove(rule);
        for (int i=0;i<rule.getBodyLength();i++) {
            PredicateInfo bodyPredicateInfo=getPredicateInfo(rule.getBodyLiteral(i).getPredicate());
            bodyPredicateInfo.m_predicateSupport.remove(rule);
        }
    }
    /**
     * Removes auxiliary predicates which aren't mentioned in any of the rules.
     *
     * @return                                          the set of predicates that were purged
     */
    public Set purge() {
        Set removedPredicates=new HashSet();
        Iterator iterator=m_predicateInfosByPredicate.values().iterator();
        while (iterator.hasNext()) {
            PredicateInfo predicateInfo=(PredicateInfo)iterator.next();
            if (predicateInfo.m_predicateSupport.isEmpty()) {
                predicateInfo.dumpModifiedPredicates(removedPredicates);
                iterator.remove();
            }
        }
        return removedPredicates;
    }
    /**
     * Returns the maintenance program based on the current state of the materialization manager.
     *
     * @return                                          the maintenance program
     */
    public Program getMaintenanceProgram() {
        if (m_maintenanceProgram==null) {
            List programRules=new ArrayList();
            Iterator iterator=m_predicateInfosByPredicate.values().iterator();
            while (iterator.hasNext()) {
                PredicateInfo predicateInfo=(PredicateInfo)iterator.next();
                predicateInfo.dumpMaintenanceRules(programRules);
            }
            Rule[] rulesArray=new Rule[programRules.size()];
            programRules.toArray(rulesArray);
            m_maintenanceProgram=new Program(rulesArray);
        }
        return m_maintenanceProgram;
    }
    /**
     * Returns the maintenance program in which specified predicates are computed from scratch.
     *
     * @param recomputedPredicates                      the set of predicates to recompute from scratch
     * @return                                          the maintenance program
     */
    public Program getMaintenanceProgram(Set recomputedPredicates) {
        List programRules=new ArrayList();
        Iterator iterator=m_predicateInfosByPredicate.values().iterator();
        while (iterator.hasNext()) {
            PredicateInfo predicateInfo=(PredicateInfo)iterator.next();
            if (recomputedPredicates.contains(predicateInfo.m_predicate))
                predicateInfo.dumpRecomputePredicates(programRules);
            else
                predicateInfo.dumpMaintenanceRules(programRules);
        }
        Rule[] rulesArray=new Rule[programRules.size()];
        programRules.toArray(rulesArray);
        return new Program(rulesArray);
    }

    /**
     * Encapsulates all relevant information about a predicate.
     */
    protected class PredicateInfo {
        /** The predicate for which this class contains information. */
        public Predicate m_predicate;
        /** The p^Del predicate. */
        public Predicate m_delPredicate;
        /** The p^Ins predicate. */
        public Predicate m_insPredicate;
        /** The p^Red predicate. */
        public Predicate m_redPredicate;
        /** The p^New predicate. */
        public Predicate m_newPredicate;
        /** The p^Plus predicate. */
        public Predicate m_plusPredicate;
        /** The p^Minus predicate. */
        public Predicate m_minusPredicate;
        /** The map of sets of p^Del rules indexed by the original rule. */
        public Map m_delRulesByOriginalRule;
        /** The map of p^Red rules indexed by the original rule. */
        public Map m_redRuleByOriginalRule;
        /** The map of sets of p^Ins rules indexed by the original rule. */
        public Map m_insRulesByOriginalRule;
        /** The array of exactly three p^New rules for this predicate. */
        public Rule[] m_newRules;
        /** The p^Plus rule for this predicate. */
        public Rule m_plusRule;
        /** The p^Minus rule for this predicate. */
        public Rule m_minusRule;
        /** The set of rules refering to this predicate, either in the head or in the body. */
        public Set m_predicateSupport;

        /**
         * Creates an instance of this class.
         *
         * @param predicate                             the predicate
         */
        public PredicateInfo(Predicate predicate) {
            m_predicate=predicate;
            m_delPredicate=m_predicateFactory.getPredicate(m_predicate.isPositive(),m_predicate.getSimpleName()+"_del",m_predicate.getArity());
            m_insPredicate=m_predicateFactory.getPredicate(m_predicate.isPositive(),m_predicate.getSimpleName()+"_ins",m_predicate.getArity());
            m_redPredicate=m_predicateFactory.getPredicate(m_predicate.isPositive(),m_predicate.getSimpleName()+"_red",m_predicate.getArity());
            m_newPredicate=m_predicateFactory.getPredicate(m_predicate.isPositive(),m_predicate.getSimpleName()+"_new",m_predicate.getArity());
            m_plusPredicate=m_predicateFactory.getPredicate(m_predicate.isPositive(),m_predicate.getSimpleName()+"_plus",m_predicate.getArity());
            m_minusPredicate=m_predicateFactory.getPredicate(m_predicate.isPositive(),m_predicate.getSimpleName()+"_minus",m_predicate.getArity());
        }
        /**
         * Initializes this object. This must be done outside constructor, since otwerwise we might get into an infinite loop.
         */
        protected void initialize() {
            m_delRulesByOriginalRule=new HashMap();
            m_redRuleByOriginalRule=new HashMap();
            m_insRulesByOriginalRule=new HashMap();
            m_newRules=generateNewRules(m_predicate);
            m_plusRule=generatePlusRule(m_predicate);
            m_minusRule=generateMinusRule(m_predicate);
            m_predicateSupport=new HashSet();
        }
        /**
         * Dumps all modified predicates related with this predicate.
         *
         * @param collection                            the collection receiving the modified predicates
         */
        public void dumpModifiedPredicates(Collection collection) {
            collection.add(m_delPredicate);
            collection.add(m_insPredicate);
            collection.add(m_redPredicate);
            collection.add(m_newPredicate);
            collection.add(m_plusPredicate);
            collection.add(m_minusPredicate);
        }
        /**
         * Dumps maintenance rules to given collection.
         *
         * @param maintenanceRules                      the collection receiving the rules
         */
        public void dumpMaintenanceRules(Collection maintenanceRules) {
            Iterator iterator=m_delRulesByOriginalRule.values().iterator();
            while (iterator.hasNext()) {
                Set set=(Set)iterator.next();
                maintenanceRules.addAll(set);
            }
            maintenanceRules.addAll(m_redRuleByOriginalRule.values());
            iterator=m_insRulesByOriginalRule.values().iterator();
            while (iterator.hasNext()) {
                Set set=(Set)iterator.next();
                maintenanceRules.addAll(set);
            }
            for (int i=0;i<m_newRules.length;i++)
                maintenanceRules.add(m_newRules[i]);
            maintenanceRules.add(m_plusRule);
            maintenanceRules.add(m_minusRule);
        }
        /**
         * Creates rules that completely recompute the extension of this predicate. This is
         * useful when some of the rules defining this predicate has been added to the program
         *
         * @param maintenanceRules                      the collection receiving rules
         *
         */
        public void dumpRecomputePredicates(Collection maintenanceRules) {
            // one can iterate over any collection
            Iterator iterator=m_delRulesByOriginalRule.keySet().iterator();
            while (iterator.hasNext()) {
                Rule rule=(Rule)iterator.next();
                Literal headLiteral=rule.getHeadLiteral();
                Literal[] headLiterals=new Literal[] { new Literal(getNewPredicate(headLiteral.getPredicate()),true,headLiteral.getTerms()) };
                Literal[] bodyLiterals=new Literal[rule.getBodyLength()];
                for (int i=0;i<rule.getBodyLength();i++) {
                    Literal bodyLiteral=rule.getBodyLiteral(i);
                    bodyLiterals[i]=new Literal(getNewPredicate(bodyLiteral.getPredicate()),bodyLiteral.isPositive(),bodyLiteral.getTerms());
                }
                Rule newRule=new Rule(rule.getLabel()+"_recompute_new",headLiterals,bodyLiterals);
                maintenanceRules.add(newRule);
            }
            Term[] terms=new Term[m_predicate.getArity()];
            for (int i=0;i<m_predicate.getArity();i++)
                terms[i]=new Variable("X"+i);
            Rule insRule=new Rule(m_predicate.getFullName()+"_recompute_ins",new Literal[] { new Literal(m_insPredicate,true,terms) },new Literal[] {
                new Literal(m_newPredicate,true,terms),
                new Literal(m_predicate,false,terms),
            });
            maintenanceRules.add(insRule);
            Rule delRule=new Rule(m_predicate.getFullName()+"_recompute_del",new Literal[] { new Literal(m_delPredicate,true,terms) },new Literal[] {
                new Literal(m_predicate,true,terms),
                new Literal(m_newPredicate,false,terms),
            });
            maintenanceRules.add(delRule);
            maintenanceRules.add(m_plusRule);
            maintenanceRules.add(m_minusRule);
        }
        /**
         * Adds a rule that defines this predicate to this object.
         *
         * @param rule                                  the rule added
         */
        public void addRule(Rule rule) {
            m_delRulesByOriginalRule.put(rule,generateDelRules(rule));
            m_redRuleByOriginalRule.put(rule,generateRedRule(rule));
            m_insRulesByOriginalRule.put(rule,generateInsRules(rule));
        }
        /**
         * Removes a rule that defines this predicate to this object.
         *
         * @param rule                                  the rule added
         */
        public void removeRule(Rule rule) {
            m_delRulesByOriginalRule.remove(rule);
            m_redRuleByOriginalRule.remove(rule);
            m_insRulesByOriginalRule.remove(rule);
        }
        /**
         * Generates a p^Del rules.
         *
         * @param rule                                  the rule for which p^Del rules are generated
         * @return                                      the set of rules
         */
        protected Set generateDelRules(Rule rule) {
            Set ruleSet=new HashSet();
            Literal headLiteral=rule.getHeadLiteral();
            Literal[] headLiterals=new Literal[] { new Literal(getDelPredicate(headLiteral.getPredicate()),true,headLiteral.getTerms()) };
            for (int i=0;i<rule.getBodyLength();i++) {
                Literal bodyLiteral=rule.getBodyLiteral(i);
                Literal[] bodyLiterals=(Literal[])rule.getBodyLiterals().clone();
                bodyLiterals[i]=new Literal(getDelPredicate(bodyLiteral.getPredicate()),bodyLiteral.isPositive(),bodyLiteral.getTerms());
                ruleSet.add(new Rule(rule.getLabel()+"_del"+i,headLiterals,bodyLiterals));
            }
            return ruleSet;
        }
        /**
         * Generates p^Ins rules for given rule.
         *
         * @param rule                                  the rule for which p^Ins rules are generated
         * @return                                      the set of rules
         */
        protected Set generateInsRules(Rule rule) {
            Set ruleSet=new HashSet();
            Literal headLiteral=rule.getHeadLiteral();
            Literal[] headLiterals=new Literal[] { new Literal(getInsPredicate(headLiteral.getPredicate()),true,headLiteral.getTerms()) };
            for (int i=0;i<rule.getBodyLength();i++) {
                Literal[] bodyLiterals=new Literal[rule.getBodyLength()];
                for (int j=0;j<rule.getBodyLength();j++) {
                    Literal bodyLiteral=rule.getBodyLiteral(j);
                    Predicate predicate;
                    if (i==j)
                        predicate=getInsPredicate(bodyLiteral.getPredicate());
                    else
                        predicate=getNewPredicate(bodyLiteral.getPredicate());
                    bodyLiterals[j]=new Literal(predicate,bodyLiteral.isPositive(),bodyLiteral.getTerms());
                }
                ruleSet.add(new Rule(rule.getLabel()+"_ins"+i,headLiterals,bodyLiterals));
            }
            return ruleSet;
        }
        /**
         * Generates the p^Red rule for given rule.
         *
         * @param rule                                  the rule for which p^Red rule is generated
         * @return                                      the p^Red rule
         */
        protected Rule generateRedRule(Rule rule) {
            Literal headLiteral=rule.getHeadLiteral();
            Literal[] headLiterals=new Literal[] { new Literal(getRedPredicate(headLiteral.getPredicate()),true,headLiteral.getTerms()) };
            Literal[] bodyLiterals=new Literal[rule.getBodyLength()+1];
            bodyLiterals[0]=new Literal(getDelPredicate(headLiteral.getPredicate()),true,headLiteral.getTerms());
            for (int i=0;i<rule.getBodyLength();i++) {
                Literal bodyLiteral=rule.getBodyLiteral(i);
                bodyLiterals[i+1]=new Literal(getNewPredicate(bodyLiteral.getPredicate()),bodyLiteral.isPositive(),bodyLiteral.getTerms());
            }
            return new Rule(rule.getLabel()+"_red",headLiterals,bodyLiterals);
        }
        /**
         * Generates the p^New rules for given rule.
         *
         * @param predicate                             the predicate for which p^New rules are generated
         * @return                                      the array of rules
         */
        protected Rule[] generateNewRules(Predicate predicate) {
            Rule[] ruleArray=new Rule[3];
            Term[] terms=new Term[predicate.getArity()];
            for (int i=0;i<predicate.getArity();i++)
                terms[i]=new Variable("X"+i);
            Literal[] headLiterals=new Literal[] { new Literal(getNewPredicate(predicate),true,terms) };
            ruleArray[0]=new Rule(predicate.getFullName()+"_new0",headLiterals,new Literal[] {
                new Literal(predicate,true,terms),
                new Literal(getDelPredicate(predicate),false,terms),
            });
            ruleArray[1]=new Rule(predicate.getFullName()+"_new1",headLiterals,new Literal[] { new Literal(getRedPredicate(predicate),true,terms) });
            ruleArray[2]=new Rule(predicate.getFullName()+"_new2",headLiterals,new Literal[] { new Literal(getInsPredicate(predicate),true,terms) });
            return ruleArray;
        }
        /**
         * Generates the p^Plus rule for given rule.
         *
         * @param predicate                             the predicate for which p^Plus rules are generated
         * @return                                      the p^Plus rule
         */
        protected Rule generatePlusRule(Predicate predicate) {
            Term[] terms=new Term[predicate.getArity()];
            for (int i=0;i<predicate.getArity();i++)
                terms[i]=new Variable("X"+i);
            return new Rule(predicate.getFullName()+"_plus",new Literal[] { new Literal(getPlusPredicate(predicate),true,terms) },new Literal[] {
                new Literal(getInsPredicate(predicate),true,terms),
                new Literal(predicate,false,terms),
            });
        }
        /**
         * Generates the p^Minus rule for given rule.
         *
         * @param predicate                             the predicate for which p^Minus rules are generated
         * @return                                      the p^Minus rule
         */
        protected Rule generateMinusRule(Predicate predicate) {
            Term[] terms=new Term[predicate.getArity()];
            for (int i=0;i<predicate.getArity();i++)
                terms[i]=new Variable("X"+i);
            return new Rule(predicate.getFullName()+"_minus",new Literal[] { new Literal(getMinusPredicate(predicate),true,terms) },new Literal[] {
                new Literal(getDelPredicate(predicate),true,terms),
                new Literal(getInsPredicate(predicate),false,terms),
                new Literal(getRedPredicate(predicate),false,terms),
            });
        }
    }
}
