package edu.unika.aifb.kaon.datalog.magic;

import java.util.Set;
import java.util.HashSet;
import java.util.Map;
import java.util.HashMap;
import java.util.List;
import java.util.LinkedList;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Collections;

import edu.unika.aifb.kaon.datalog.*;
import edu.unika.aifb.kaon.datalog.program.*;

/**
 * This class implements the standard, maximal left-to-right SIPS.
 */
public class StandardSIPS implements SIPS {
    /** An instance of this class. */
    public static final SIPS INSTANCE=new StandardSIPS();

    /**
     * Examines a rule and produces SIPS information for it.
     *
     * @param rule                                  the rule to be examined
     * @param idbPredicates                         the set of IDB predicates (that participate in adornment)
     * @param boundHeadVariables                    the array specifying which variables from the head are bound
     * @param predicateBindingPatterns              determines allowed binding patterns of a predicate
     * @param allFreeOverrides                      the set of predicates that occur all free
     * @return                                      the array containing information about adornation of literals
     * @throws DatalogException                     thrown if there is no valid rewriting of a program
     */
    protected AdornationInfo[] processRule(Rule rule,Set idbPredicates,boolean[] boundHeadVariables,PredicateBindingPatterns predicateBindingPatterns,Set allFreeOverrides) throws DatalogException {
        Set boundVariablesSet=new HashSet();
        for (int i=0;i<rule.getHeadLiteral().getArity();i++) {
            Variable variable=rule.getHeadLiteral().getArgumentVariable(i);
            if (variable!=null && boundHeadVariables[i])
                boundVariablesSet.add(variable);
        }
        List nonResolvedLiterals=new ArrayList(rule.getBodyLength());
        for (int i=0;i<rule.getBodyLength();i++)
            nonResolvedLiterals.add(rule.getBodyLiteral(i));
        AdornationInfo[] result=new AdornationInfo[rule.getBodyLength()];
        int index=0;
        boolean inIDBRun=false;
        // Always pick a non-resolved the literal with most bound variables. Try to group IDB and non-IDB literals.
        while (!nonResolvedLiterals.isEmpty()) {
            Literal bestSoFar=null;
            int bindingGoodnessInBestSoFar=Integer.MIN_VALUE;
            int bestSoFarIndex=0;
            boolean bestSoFarIsIDB=false;
            boolean[] bestSoFarBoundVariables=null;
            for (int nonResolvedLiteralIndex=0;nonResolvedLiteralIndex<nonResolvedLiterals.size();nonResolvedLiteralIndex++) {
                Literal literal=(Literal)nonResolvedLiterals.get(nonResolvedLiteralIndex);
                boolean literalIsIDB=idbPredicates.contains(literal.getPredicate());
                boolean[] boundVariables=new boolean[literal.getArity()];
                for (int i=0;i<literal.getArity();i++) {
                    Variable variable=literal.getArgumentVariable(i);
                    if (variable==null || boundVariablesSet.contains(variable))
                        boundVariables[i]=true;
                }
                if (predicateBindingPatterns.isBindingPatternAllowed(literal.getPredicate(),boundVariables)) {
                    int bindingGoodness=getBindingGoodness(literal,boundVariables);
                    if (bestSoFar==null || bindingGoodness>bindingGoodnessInBestSoFar || (bindingGoodness==bindingGoodnessInBestSoFar && literalIsIDB==inIDBRun && bestSoFarIsIDB!=inIDBRun)) {
                        bestSoFar=literal;
                        bestSoFarIndex=nonResolvedLiteralIndex;
                        bindingGoodnessInBestSoFar=bindingGoodness;
                        bestSoFarIsIDB=literalIsIDB;
                        bestSoFarBoundVariables=boundVariables;
                    }
                }
            }
            if (bestSoFar==null)
                throw new DatalogException("Cannot order literals according to allowed binding patterns.");
            for (int i=0;i<bestSoFar.getArity();i++) {
                Variable variable=bestSoFar.getArgumentVariable(i);
                if (variable!=null)
                    boundVariablesSet.add(variable);
            }
            if (allFreeOverrides.contains(bestSoFar.getPredicate()))
                for (int i=0;i<bestSoFarBoundVariables.length;i++)
                    bestSoFarBoundVariables[i]=false;
            result[index++]=new AdornationInfo(bestSoFar,bestSoFarIsIDB ? bestSoFarBoundVariables : null,true);
            inIDBRun=bestSoFarIsIDB;
            nonResolvedLiterals.remove(bestSoFarIndex);
        }
        return result;
    }
    /**
     * Creates an adorned program from given program.
     *
     * @param program                               the program
     * @param rulesByHeadPredicate                  map of sets of rules indexed by the predicate defining them
     * @param queryPredicate                        the query predicate
     * @param boundVariables                        the bound variables
     * @param additionalFreePredicates              the set of predicates that are additionally invoked as free
     * @param idbPredicates                         the set of IDB predicates in the program
     * @param predicateBindingPatterns              determines allowed binding patterns of a predicate
     * @param allFreeOverrides                      the set of predicates that occur all free
     * @return                                      the adorned program
     * @throws DatalogException                     thrown if there is no valid rewriting of a program
     */
    protected AdornedRuleInfo[] adornProgram(Program program,Map rulesByHeadPredicate,Predicate queryPredicate,boolean[] boundVariables,Set additionalFreePredicates,Set idbPredicates,PredicateBindingPatterns predicateBindingPatterns,Set allFreeOverrides) throws DatalogException {
        Set processedHeadAdornations=new HashSet();
        List result=new LinkedList();
        List adornedPredicatesToProcess=new LinkedList();
        PredicateAdornationInfo predicateAdornationInfo=new PredicateAdornationInfo(queryPredicate,boundVariables);
        adornedPredicatesToProcess.add(predicateAdornationInfo);
        if (predicateAdornationInfo.isAllFree())
            allFreeOverrides.add(predicateAdornationInfo.m_predicate);
        Iterator additionalPredicatesIterator=additionalFreePredicates.iterator();
        while (additionalPredicatesIterator.hasNext()) {
            Predicate predicate=(Predicate)additionalPredicatesIterator.next();
            predicateAdornationInfo=new PredicateAdornationInfo(predicate,new boolean[predicate.getArity()]);
            adornedPredicatesToProcess.add(predicateAdornationInfo);
            if (predicateAdornationInfo.isAllFree())
                allFreeOverrides.add(predicateAdornationInfo.m_predicate);
        }
        while (!adornedPredicatesToProcess.isEmpty()) {
            PredicateAdornationInfo headPredicateAdornationInfo=(PredicateAdornationInfo)adornedPredicatesToProcess.remove(0);
            Predicate unadornedHeadPredicate=headPredicateAdornationInfo.m_predicate;
            if (idbPredicates.contains(unadornedHeadPredicate)) {
                processedHeadAdornations.add(headPredicateAdornationInfo);
                Set rulesDefiningPredicate=(Set)rulesByHeadPredicate.get(unadornedHeadPredicate);
                if (rulesDefiningPredicate==null)
                    rulesDefiningPredicate=Collections.EMPTY_SET;
                Iterator iterator=rulesDefiningPredicate.iterator();
                while (iterator.hasNext()) {
                    Rule rule=(Rule)iterator.next();
                    SIPS.AdornationInfo[] adornationInfos=processRule(rule,idbPredicates,headPredicateAdornationInfo.m_boundVariables,predicateBindingPatterns,allFreeOverrides);
                    result.add(new AdornedRuleInfo(rule,new AdornationInfo(rule.getHeadLiteral(),headPredicateAdornationInfo.m_boundVariables,false),adornationInfos));
                    for (int i=0;i<adornationInfos.length;i++) {
                        AdornationInfo bodyAdornationInfo=adornationInfos[i];
                        if (bodyAdornationInfo.m_boundVariables!=null) {
                            PredicateAdornationInfo bodyPredicateAdornationInfo=new PredicateAdornationInfo(bodyAdornationInfo.m_literal.getPredicate(),bodyAdornationInfo.m_boundVariables);
                            if (bodyPredicateAdornationInfo.isAllFree())
                                allFreeOverrides.add(bodyPredicateAdornationInfo.m_predicate);
                            if (!processedHeadAdornations.contains(bodyPredicateAdornationInfo))
                                adornedPredicatesToProcess.add(0,bodyPredicateAdornationInfo);
                        }
                    }
                }
            }
        }
        AdornedRuleInfo[] resultArray=new AdornedRuleInfo[result.size()];
        result.toArray(resultArray);
        return resultArray;
    }
    /**
     * Creates an adorned program from given program.
     *
     * @param program                               the program
     * @param queryPredicate                        the query predicate
     * @param boundVariables                        the bound variables
     * @param additionalFreePredicates              the set of predicates that are additionally invoked as free
     * @param idbPredicates                         the set of IDB predicates in the program
     * @param predicateBindingPatterns              determines allowed binding patterns of a predicate
     * @return                                      the adorned program
     * @throws DatalogException                     thrown if there is no valid rewriting of a program
     */
    public AdornedRuleInfo[] adornProgram(Program program,Predicate queryPredicate,boolean[] boundVariables,Set additionalFreePredicates,Set idbPredicates,PredicateBindingPatterns predicateBindingPatterns) throws DatalogException {
        Map rulesByHeadPredicate=new HashMap();
        for (int ruleIndex=0;ruleIndex<program.getNumberOfRules();ruleIndex++) {
            Rule rule=program.getRule(ruleIndex);
            for (int headIndex=0;headIndex<rule.getHeadLength();headIndex++) {
                Predicate headPredicate=rule.getHeadLiteral(headIndex).getPredicate();
                Set set=(Set)rulesByHeadPredicate.get(headPredicate);
                if (set==null) {
                    set=new HashSet();
                    rulesByHeadPredicate.put(headPredicate,set);
                }
                set.add(rule);
            }
        }
        Set allFreeOverrides=new HashSet();
        AdornedRuleInfo[] result=adornProgram(program,rulesByHeadPredicate,queryPredicate,boundVariables,additionalFreePredicates,idbPredicates,predicateBindingPatterns,allFreeOverrides);
        if (!allFreeOverrides.isEmpty())
            result=adornProgram(program,rulesByHeadPredicate,queryPredicate,boundVariables,additionalFreePredicates,idbPredicates,predicateBindingPatterns,allFreeOverrides);
        return result;
    }

    /**
     * Holds the information about the adornation of a predicate.
     */
    public static class PredicateAdornationInfo {
        /** The predicate being adorned. */
        public Predicate m_predicate;
        /** The array defining the bindings of the variables. It is <code>null</code> if the predicate is an EDB predicate or if this literal doesn't preticipate in SIPS. Only positions in literals that have variables should be marked as bound. */
        public boolean[] m_boundVariables;
        /** The hash-code. */
        protected int m_hashCode;

        public PredicateAdornationInfo(Predicate predicate,boolean[] boundVariables) {
            m_predicate=predicate;
            m_boundVariables=boundVariables;
            m_hashCode=m_predicate.hashCode();
            for (int i=0;i<m_boundVariables.length;i++)
                m_hashCode=m_hashCode*7+(m_boundVariables[i] ? 1 : 0);
        }
        public boolean isAllFree() {
            for (int i=0;i<m_boundVariables.length;i++)
                if (m_boundVariables[i])
                    return false;
            return true;
        }
        public boolean equals(Object other) {
            if (this==other)
                return true;
            if (!(other instanceof PredicateAdornationInfo))
                return false;
            PredicateAdornationInfo otherPredicateAdornationInfo=(PredicateAdornationInfo)other;
            if (!m_predicate.equals(otherPredicateAdornationInfo.m_predicate))
                return false;
            for (int i=0;i<m_boundVariables.length;i++)
                if (m_boundVariables[i]!=otherPredicateAdornationInfo.m_boundVariables[i])
                    return false;
            return true;
        }
        public int hashCode() {
            return m_hashCode;
        }
    }
    /**
     * Compute the goodness of the bindings in a literal.
     *
     * @param literal                               the literal for which the goodness is computed
     * @param boundVariables                        the variables bound in the literal
     * @return                                      the goodness of the binding
     */
    protected int getBindingGoodness(Literal literal,boolean[] boundVariables) {
        int goodness;
        if (literal.isPositive()) {
            goodness=0;
            for (int i=0;i<boundVariables.length;i++)
                if (boundVariables[i])
                    goodness++;
        }
        else
            goodness=-10000;
        return goodness;
    }
}
