package edu.unika.aifb.kaon.datalog.magic;

import java.util.Set;
import java.util.HashSet;
import java.util.Map;
import java.util.Collections;

import edu.unika.aifb.kaon.datalog.*;
import edu.unika.aifb.kaon.datalog.program.*;
import edu.unika.aifb.kaon.datalog.evaluation.*;

/**
 * This class represents a magic set rewriting of the starting program with respect to given query.
 */
public class MagicSetRewriting {
    /** The original program. */
    protected Program m_program;
    /** The query predicate. */
    protected Predicate m_queryPredicate;
    /** The array specifying which variables are bound and which are not. */
    protected boolean[] m_boundVariables;
    /** The rewritten magic program. */
    protected Program m_magicProgram;
    /** The seed predicate. */
    protected Predicate m_seedPredicate;
    /** The predicate containing the answer. */
    protected Predicate m_answerPredicate;
    /** The set of all predicates that were created during processing. */
    protected Set m_addedPredicates;
    /** The map indexed the predicates containing the set of adorned predicates for this predicate. */
    protected Map m_adornedPredicatesByPredicate;

    /**
     * Creates an instance of this class.
     *
     * @param program                           the original program
     * @param queryPredicate                    the query predicate
     * @param boundVariables                    the array specifying which variables are bound
     * @param sips                              the sideways information passing strategy
     * @param predicateBindingPatterns          determines allowed binding patterns of a predicate
     * @throws DatalogException                 thrown if rewriting can't be created
     */
    public MagicSetRewriting(Program program,Predicate queryPredicate,boolean[] boundVariables,SIPS sips,PredicateBindingPatterns predicateBindingPatterns) throws DatalogException {
        MagicSetRewriting realRewriting=null;
        if (program.isHorn())
            realRewriting=new HornMagicSetRewriting(program,queryPredicate,boundVariables,Collections.EMPTY_SET,sips,predicateBindingPatterns,Collections.EMPTY_SET);
        else
            realRewriting=new DisjunctiveMagicSetRewriting(program,queryPredicate,boundVariables,sips,predicateBindingPatterns);
        m_program=realRewriting.getOriginalProgram();
        m_queryPredicate=realRewriting.getQueryPredicate();
        m_boundVariables=realRewriting.getBoundVariables();
        m_magicProgram=realRewriting.getMagicProgram();
        m_seedPredicate=realRewriting.getSeedPredicate();
        m_answerPredicate=realRewriting.getAnswerPredicate();
        m_addedPredicates=realRewriting.getAddedPredicates();
        m_adornedPredicatesByPredicate=realRewriting.getAdornedPredicatesByPredicate();
    }
    /**
     * The constructor for the derived classes that can be used to circumvent the public constructor.
     */
    protected MagicSetRewriting() {
    }
    /**
     * Returns the original program.
     *
     * @return                                  the original program
     */
    public Program getOriginalProgram() {
        return m_program;
    }
    /**
     * Returns the query predicate.
     *
     * @return                                  the query predicate
     */
    public Predicate getQueryPredicate() {
        return m_queryPredicate;
    }
    /**
     * Returns the array specifying which variables are bound.
     *
     * @return                                  the array specifying which variables are bound
     */
    public boolean[] getBoundVariables() {
        return m_boundVariables;
    }
    /**
     * Returns the magic rewriting of the program.
     *
     * @return                                  the magic rewriting of the program
     */
    public Program getMagicProgram() {
        return m_magicProgram;
    }
    /**
     * Returns the seed predicate that is filled with the bound query values.
     *
     * @return                                  the seed predicate that is filled with bound query values
     */
    public Predicate getSeedPredicate() {
        return m_seedPredicate;
    }
    /**
     * Returns the predicate containing the answer to the query.
     *
     * @return                                  the predicate containing the answer to the query
     */
    public Predicate getAnswerPredicate() {
        return m_answerPredicate;
    }
    /**
     * Returns the set of predicates added during the process of the rewriting.
     *
     * @return                                  the set of predicates added during the process of the rewriting
     */
    public Set getAddedPredicates() {
        return m_addedPredicates;
    }
    /**
     * Returns the map indexed by the predicate containing the set of adorned predicates.
     *
     * @return                                  the map indexed by the predicate containing the set of adorned predicates
     */
    public Map getAdornedPredicatesByPredicate() {
        return m_adornedPredicatesByPredicate;
    }
    /**
     * Given bound values are stored in the database and a query operator is created that can be used to read the result.
     *
     * @param extensionalManager                the manager of extensional databases
     * @param bindings                          the bindings for the query predicate
     * @return                                  the query operator for reading the result
     * @throws DatalogException                 thrown if there is an error
     */
    public QueryOperator getResultForBoundValues(ExtensionalManager extensionalManager,Constant[] bindings) throws DatalogException {
        if (m_seedPredicate!=null) {
            MemoryPredicateExtension seedExtension=extensionalManager.getMemoryPredicateExtension(m_seedPredicate);
            Object[] seed=new Object[m_seedPredicate.getArity()+(seedExtension!=null && seedExtension.getContainsDisjunctionInfo() ? 1 : 0)];
            int[] seedIndices=new int[m_seedPredicate.getArity()];
            int[][] indicesToCopy=new int[m_answerPredicate.getArity()][2];
            int index=0;
            for (int i=0;i<bindings.length;i++) {
                indicesToCopy[i][0]=i;
                indicesToCopy[i][1]=i;
                if (bindings[i]!=null) {
                    seed[index]=bindings[i].getValue();
                    seedIndices[index]=i;
                    index++;
                }
            }
            if (seedExtension==null) {
                if (m_seedPredicate.getArity()!=0)
                    throw new DatalogException("Cannot locate seed predicate '"+m_seedPredicate.getFullName()+"' in extensional databases.");
            }
            else {
                if (seedExtension.getContainsDisjunctionInfo())
                    seed[seed.length-1]=new DisjunctionRestInfo(null);
                seedExtension.addTuple(seed);
            }
            MemoryPredicateExtension answerExtension=extensionalManager.getMemoryPredicateExtension(m_answerPredicate);
            if (answerExtension==null)
                throw new DatalogException("Cannot locate answer predicate '"+m_answerPredicate.getFullName()+"' in extensional databases.");
            return new PredicateExtensionLookup(answerExtension,Filters.ALWAYS_TRUE_TUPLE_FILTER,indicesToCopy,seed,seedIndices);
        }
        else {
            ExtensionalDatabase extensionalDatabase=extensionalManager.getExtensionalDatabase(m_answerPredicate);
            if (extensionalDatabase==null)
                throw new DatalogException("Cannot locate answer predicate '"+m_answerPredicate.getFullName()+"' in extensional databases.");
            int answerArity=m_answerPredicate.getArity();
            int numberOfBoundValues=0;
            for (int i=0;i<answerArity;i++)
                if (bindings[i]!=null)
                    numberOfBoundValues++;
            int index=0;
            Variable[] variables=new Variable[answerArity];
            Object[] boundValues=new Object[numberOfBoundValues];
            Variable[] boundVariables=new Variable[numberOfBoundValues];
            for (int i=0;i<answerArity;i++) {
                variables[i]=new Variable("X"+i);
                if (bindings[i]!=null) {
                    boundValues[index]=bindings[i].getValue();
                    boundVariables[index]=variables[i];
                    index++;
                }
            }
            if (numberOfBoundValues==0)
                return extensionalDatabase.createQueryOperator(new Literal[] { new Literal(m_answerPredicate,true,variables) },variables);
            else
                return extensionalDatabase.createQueryOperator(new Literal[] { new Literal(m_answerPredicate,true,variables) },new SingletonQueryOperator(boundValues),boundVariables,variables);
        }
    }
    /**
     * Returns the set of predicates occuring in rule heads (that is, the IDB predicates).
     *
     * @param program                           the program
     * @return                                  the set of predicates in rule heads
     */
    protected Set getPredicatesInRuleHeads(Program program) {
        Set result=new HashSet();
        for (int i=0;i<program.getNumberOfRules();i++) {
            Rule rule=program.getRule(i);
            for (int j=0;j<rule.getHeadLength();j++)
                result.add(rule.getHeadLiteral(j).getPredicate());
        }
        return result;
    }
}
