package edu.unika.aifb.kaon.datalog.magic;

import java.util.List;
import java.util.LinkedList;
import java.util.ArrayList;
import java.util.Set;
import java.util.HashSet;
import java.util.HashMap;

import edu.unika.aifb.kaon.datalog.*;
import edu.unika.aifb.kaon.datalog.program.*;

/**
 * This class represents a magic set rewriting of a horn program.
 */
public class HornMagicSetRewriting extends MagicSetRewriting {

    /**
     * Creates an instance of this class.
     *
     * @param program                           the original program
     * @param queryPredicate                    the query predicate
     * @param boundVariables                    the array specifying which variables are bound
     * @param additionalFreePredicates          the set of predicates that are additionally invoked as free
     * @param sips                              the sideways information passing strategy
     * @param predicateBindingPatterns          determines allowed binding patterns of a predicate
     * @param rulesWithNoModifiedRules          the set of rules for which modified rules should not be generated
     * @throws DatalogException                 thrown if rewriting can't be created
     */
    public HornMagicSetRewriting(Program program,Predicate queryPredicate,boolean[] boundVariables,Set additionalFreePredicates,SIPS sips,PredicateBindingPatterns predicateBindingPatterns,Set rulesWithNoModifiedRules) throws DatalogException {
        if (queryPredicate.getArity()!=boundVariables.length)
            throw new IllegalArgumentException("The arity of the query predicate and the signature of the bound variables don't match.");
        if (!program.isHorn())
            throw new DatalogException("This class can rewrite only Horn programs.");
        m_program=program;
        m_queryPredicate=queryPredicate;
        m_boundVariables=boundVariables;
        new Rewriter(additionalFreePredicates,sips,predicateBindingPatterns,rulesWithNoModifiedRules);
    }

    /**
     * Encapsulates the algorithm for rewriting a program into a magic program.
     */
    protected class Rewriter {
        /** Determines the allowed binding patterns. */
        protected PredicateBindingPatterns m_predicateBindingPatterns;
        /** Determines the rules for which modified rules should not be generated. */
        protected Set m_rulesWithNoModifiedRules;
        /** The list of rules of the magic program. */
        protected List m_magicProgramRules;
        /** The set of IDB predicates. */
        protected Set m_idbPredicates;
        /** The factory for the predicates. */
        protected PredicateFactory m_predicateFactory;
        /** The index of the rule. */
        protected int m_ruleIndex;
        /** The index of rules within one outer rule. */
        protected int m_withinRuleIndex;
        /** The index of magic rules within one outer rule. */
        protected int m_magicWithinRuleIndex;

        /**
         * Creates an instance of this class.
         *
         * @param additionalFreePredicates      the set of predicates that are additionally invoked as free
         * @param sips                          the SIPS to use
         * @param predicateBindingPatterns      determines allowed binding patterns of a predicate
         * @param rulesWithNoModifiedRules      the set of rules for which modified rules should not be generated
         * @throws DatalogException             thrown if rewriting can't be created
         */
        public Rewriter(Set additionalFreePredicates,SIPS sips,PredicateBindingPatterns predicateBindingPatterns,Set rulesWithNoModifiedRules) throws DatalogException {
            m_predicateBindingPatterns=predicateBindingPatterns;
            m_rulesWithNoModifiedRules=rulesWithNoModifiedRules;
            m_magicProgramRules=new LinkedList();
            m_idbPredicates=getPredicatesInRuleHeads(m_program);
            m_predicateFactory=new PredicateFactory();
            m_adornedPredicatesByPredicate=new HashMap();
            SIPS.AdornedRuleInfo[] adornedRuleInfos=sips.adornProgram(m_program,m_queryPredicate,m_boundVariables,additionalFreePredicates,m_idbPredicates,m_predicateBindingPatterns);
            for (int i=0;i<adornedRuleInfos.length;i++)
                processAdornedRule(adornedRuleInfos[i]);
            Rule[] magicProgramRulesArray=new Rule[m_magicProgramRules.size()];
            m_magicProgramRules.toArray(magicProgramRulesArray);
            m_magicProgram=new Program(magicProgramRulesArray);
            if (m_idbPredicates.contains(m_queryPredicate)) {
                m_answerPredicate=getAdornedPredicate(m_queryPredicate,m_boundVariables);
                m_seedPredicate=getMagicPredicate(m_queryPredicate,m_boundVariables);
            }
            else
                m_answerPredicate=m_queryPredicate;
            m_addedPredicates=m_predicateFactory.getAllPredicates();
        }
        /**
         * Processes an adorned rule.
         *
         * @param adornedRuleInfo               the information about an adorned rule
         */
        protected void processAdornedRule(SIPS.AdornedRuleInfo adornedRuleInfo) {
            m_withinRuleIndex=0;
            m_magicWithinRuleIndex=0;
            String ruleLabelPrefix=getRuleLabelPrefix(adornedRuleInfo);
            Rule rule=adornedRuleInfo.m_rule;
            SIPS.AdornationInfo[] bodyAdornationInfos=adornedRuleInfo.m_bodyAdornationInfos;
            boolean[] boundHeadVariables=adornedRuleInfo.m_headAdornationInfo.m_boundVariables;
            Predicate adornedHeadPredicate=getAdornedPredicate(rule.getHeadLiteral().getPredicate(),boundHeadVariables);
            Literal headLiteral=new Literal(adornedHeadPredicate,true,rule.getHeadLiteral().getTerms());
            // the first supmagic literal is the magic predicate of the head with bound variables
            Literal previousSupMagicLiteral=createMagicLiteral(rule.getHeadLiteral(),boundHeadVariables);
            // locate the last literal that participates in SIPS
            int lastLiteralParticipatingInSIPSIndex=bodyAdornationInfos.length-1;
            while (lastLiteralParticipatingInSIPSIndex>0 && !bodyAdornationInfos[lastLiteralParticipatingInSIPSIndex].m_hasIncomingArc)
                lastLiteralParticipatingInSIPSIndex--;
            // now process all adornation infos until the last one that is participating in SIPS
            int nextLiteralToAddFromIndex=0;
            for (int i=0;i<lastLiteralParticipatingInSIPSIndex;i++) {
                SIPS.AdornationInfo adornationInfo=bodyAdornationInfos[i];
                if (adornationInfo.m_boundVariables!=null) {
                    // first process all EDB literals that might have appeared in between
                    if (i>nextLiteralToAddFromIndex)
                        previousSupMagicLiteral=createSupMagicLiteralAndRule(ruleLabelPrefix,previousSupMagicLiteral,headLiteral,bodyAdornationInfos,nextLiteralToAddFromIndex,i-1);
                    if (adornationInfo.m_boundVariables!=null && adornationInfo.m_hasIncomingArc)
                        createMagicRule(ruleLabelPrefix,adornationInfo,previousSupMagicLiteral);
                    previousSupMagicLiteral=createSupMagicLiteralAndRule(ruleLabelPrefix,previousSupMagicLiteral,headLiteral,bodyAdornationInfos,i,i);
                    nextLiteralToAddFromIndex=i+1;
                }
            }
            // if the SIPS ended in EBD predicates before the last participaiting IDB predicate, then output those predicates
            if (nextLiteralToAddFromIndex<lastLiteralParticipatingInSIPSIndex && bodyAdornationInfos[lastLiteralParticipatingInSIPSIndex].m_boundVariables!=null) {
                previousSupMagicLiteral=createSupMagicLiteralAndRule(ruleLabelPrefix,previousSupMagicLiteral,headLiteral,bodyAdornationInfos,nextLiteralToAddFromIndex,lastLiteralParticipatingInSIPSIndex-1);
                nextLiteralToAddFromIndex=lastLiteralParticipatingInSIPSIndex;
            }
            // now create the modified rule for the rest of the literals
            int index=0;
            Literal[] ruleBody;
            if (previousSupMagicLiteral.getArity()!=0) {
                ruleBody=new Literal[1+bodyAdornationInfos.length-nextLiteralToAddFromIndex];
                ruleBody[index++]=previousSupMagicLiteral;
            }
            else
                ruleBody=new Literal[bodyAdornationInfos.length-nextLiteralToAddFromIndex];
            for (int i=nextLiteralToAddFromIndex;i<bodyAdornationInfos.length;i++) {
                SIPS.AdornationInfo adornationInfo=bodyAdornationInfos[i];
                Literal currentLiteral=adornationInfo.m_literal;
                if (adornationInfo.m_boundVariables!=null) {
                    Predicate adornedPredicate=getAdornedPredicate(currentLiteral.getPredicate(),adornationInfo.m_boundVariables);
                    ruleBody[index++]=new Literal(adornedPredicate,currentLiteral.isPositive(),currentLiteral.getTerms());
                    if (adornationInfo.m_hasIncomingArc)
                        createMagicRule(ruleLabelPrefix,adornationInfo,previousSupMagicLiteral);
                }
                else
                    ruleBody[index++]=currentLiteral;
            }
            if (!m_rulesWithNoModifiedRules.contains(rule)) {
                Rule modifiedRule=new Rule(ruleLabelPrefix+(m_withinRuleIndex++)+"(m)",headLiteral,ruleBody);
                m_magicProgramRules.add(modifiedRule);
            }
            m_ruleIndex++;
        }
        /**
         * Creates the magic rule.
         *
         * @param ruleLabelPrefix               the prefix for the rule label
         * @param adornationInfo                the information about the adornment for which the rule is created
         * @param supMagicLiteral               the supmagic literal for which the magic rule is generated
         */
        protected void createMagicRule(String ruleLabelPrefix,SIPS.AdornationInfo adornationInfo,Literal supMagicLiteral) {
            Literal magicLiteral=createMagicLiteral(adornationInfo.m_literal,adornationInfo.m_boundVariables);
            if (!magicLiteral.getPredicate().equals(supMagicLiteral.getPredicate()) && magicLiteral.getArity()!=0) {
                Rule magicRule=new Rule(ruleLabelPrefix+(m_magicWithinRuleIndex++)+"(mag)",magicLiteral,supMagicLiteral.getArity()==0 ? new Literal[0] : new Literal[] { supMagicLiteral });
                m_magicProgramRules.add(magicRule);
            }
        }
        /**
         * Creates a supplementary magic literal for given literal.
         *
         * @param ruleLabelPrefix               the prefix for the rule label
         * @param previousSupMagicLiteral       the previous supmagic literal
         * @param headLiteral                   the literal of the rule head (needed to preserve variables that appear in the head)
         * @param bodyAdornationInfos           the list of body adornation infos
         * @param fromLiteralIndex              the index from which the literals are processed
         * @param toLiteralIndex                the index to which the literals are processed
         * @return                              the supplemenray magic literal for given literal
         */
        protected Literal createSupMagicLiteralAndRule(String ruleLabelPrefix,Literal previousSupMagicLiteral,Literal headLiteral,SIPS.AdornationInfo[] bodyAdornationInfos,int fromLiteralIndex,int toLiteralIndex) {
            // holds the set of the variables, which is equal to all variables in the previous literal, the head literal or the body literals
            List variables=new ArrayList();
            // holds the elements of the supmagic rule's body
            int index=0;
            Literal[] ruleBody;
            if (previousSupMagicLiteral.getArity()!=0) {
                ruleBody=new Literal[toLiteralIndex-fromLiteralIndex+2];
                ruleBody[index++]=previousSupMagicLiteral;
            }
            else
                ruleBody=new Literal[toLiteralIndex-fromLiteralIndex+1];
            // add all variables from the previous supmagic literal
            processVariables(previousSupMagicLiteral,variables,headLiteral,bodyAdornationInfos,toLiteralIndex);
            // now process all literals between from and to
            for (int i=fromLiteralIndex;i<=toLiteralIndex;i++) {
                Literal currentLiteral=bodyAdornationInfos[i].m_literal;
                processVariables(currentLiteral,variables,headLiteral,bodyAdornationInfos,toLiteralIndex);
                if (bodyAdornationInfos[i].m_boundVariables!=null) {
                    Predicate adornedPredicate=getAdornedPredicate(currentLiteral.getPredicate(),bodyAdornationInfos[i].m_boundVariables);
                    ruleBody[index++]=new Literal(adornedPredicate,currentLiteral.isPositive(),currentLiteral.getTerms());
                }
                else
                    ruleBody[index++]=currentLiteral;
            }
            String predicateName="supmagic_"+m_ruleIndex+"_"+toLiteralIndex;
            Variable[] variablesArray=new Variable[variables.size()];
            variables.toArray(variablesArray);
            Literal supMagicLiteral=new Literal(m_predicateFactory.getPredicate(predicateName,variables.size()),true,variablesArray);
            if (supMagicLiteral.getArity()!=0) {
                Rule supMagicRule=new Rule(ruleLabelPrefix+(m_withinRuleIndex++)+"(sm)",supMagicLiteral,ruleBody);
                m_magicProgramRules.add(supMagicRule);
            }
            return supMagicLiteral;
        }
        /**
         * Adds the variables of the literal to the array of variables.
         *
         * @param literal                       the literal to be processed
         * @param variables                     the array of variables to which the literal's variables are added
         * @param headLiteral                   the literal of the rule head (needed to preserve variables that appear in the head)
         * @param bodyAdornationInfos           the list of body adornation infos
         * @param toLiteralIndex                the index to which the literals are processed
         */
        protected void processVariables(Literal literal,List variables,Literal headLiteral,SIPS.AdornationInfo[] bodyAdornationInfos,int toLiteralIndex) {
            nextVariable: for (int i=0;i<literal.getArity();i++) {
                Term term=literal.getTerm(i);
                if ((term instanceof Variable) && !variables.contains(term)) {
                    Variable variable=(Variable)term;
                    if (headLiteral.containsVariable(variable))
                        variables.add(term);
                    else {
                        for (int j=toLiteralIndex+1;j<bodyAdornationInfos.length;j++)
                            if (bodyAdornationInfos[j].m_literal.containsVariable(variable)) {
                                variables.add(variable);
                                continue nextVariable;
                            }
                    }
                }
            }
        }
        /**
         * Creates the magic literal from the literal.
         *
         * @param literal                       the literal from which the magic literal is created
         * @param boundVariables                the indices of the bound variables
         * @return                              the magic literal
         */
        protected Literal createMagicLiteral(Literal literal,boolean[] boundVariables) {
            Predicate magicPredicate=getMagicPredicate(literal.getPredicate(),boundVariables);
            Term[] terms=new Term[magicPredicate.getArity()];
            int index=0;
            for (int i=0;i<literal.getArity();i++)
                if (boundVariables[i])
                    terms[index++]=literal.getTerm(i);
            return new Literal(magicPredicate,true,terms);
        }
        /**
         * Returns the magic predicate for given predicate and bindings.
         *
         * @param predicate                     the predicate for which the magic predicate is required
         * @param boundVariables                the bindings of variables
         * @return                              the of the magic predicate
         */
        protected Predicate getMagicPredicate(Predicate predicate,boolean[] boundVariables) {
            int magicPredicateArity=0;
            for (int i=0;i<predicate.getArity();i++)
                if (boundVariables[i])
                    magicPredicateArity++;
            String magicPredicateName="magic_"+getAdornedPredicateName(predicate,boundVariables);
            return m_predicateFactory.getPredicate(magicPredicateName,magicPredicateArity);
        }
        /**
         * Returns the name of the adorned predicate.
         *
         * @param predicate                     the predicate
         * @param boundVariables                the bindings of variables
         * @return                              the name of the adorned predicate
         */
        protected String getAdornedPredicateName(Predicate predicate,boolean[] boundVariables) {
            StringBuffer buffer=new StringBuffer(predicate.getSimpleName());
            buffer.append('_');
            for (int i=0;i<boundVariables.length;i++)
                buffer.append(boundVariables[i] ? 'b' : 'f');
            return buffer.toString();
        }
        /**
         * Generates the adorned predicate.
         *
         * @param predicate                     the predicate
         * @param boundVariables                the bindings of variables
         * @return                              the adorned predicate
         */
        protected Predicate getAdornedPredicate(Predicate predicate,boolean[] boundVariables) {
            Predicate adornedPredicate=m_predicateFactory.getPredicate(getAdornedPredicateName(predicate,boundVariables),predicate.getArity());
            Set set=(Set)m_adornedPredicatesByPredicate.get(predicate);
            if (set==null) {
                set=new HashSet();
                m_adornedPredicatesByPredicate.put(predicate,set);
            }
            set.add(adornedPredicate);
            return adornedPredicate;
        }
        /**
         * Retruns the prefix for the labels of the generated rules.
         *
         * @param adornedRuleInfo               the information about the rule adornments
         * @return                              the prefix for the label of the generated rules
         */
        protected String getRuleLabelPrefix(SIPS.AdornedRuleInfo adornedRuleInfo) {
            StringBuffer buffer=new StringBuffer();
            if (adornedRuleInfo.m_rule.getLabel()!=null)
                buffer.append(adornedRuleInfo.m_rule.getLabel());
            buffer.append('(');
            boolean[] boundVariables=adornedRuleInfo.m_headAdornationInfo.m_boundVariables;
            for (int i=0;i<boundVariables.length;i++)
                buffer.append(boundVariables[i] ? 'b' : 'f');
            buffer.append(")-");
            return buffer.toString();
        }
    }
}
