package edu.unika.aifb.kaon.datalog.jdbc;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Types;

/**
 * The manager taking care of the type names and type conversions.
 */
public class TypeManager {
    /** The instance of this type manager. */
    public static final TypeManager INSTANCE=new TypeManager();
    /** The type of regular expression. */
    public static final int TYPE_REGULAR_EXPRESSION=10000;
    /** The type of LIKE column. */
    public static final int TYPE_LIKE=10001;

    /**
     * Sets the value of the object to the prepared statement.
     *
     * @param type                              the type
     * @param object                            the object being set
     * @param preparedStatement                 the prepared statement being set
     * @param index                             the index
     * @throws SQLException                     thrown if there is a problem
     */
    public void setObjectToStatement(int type,Object object,PreparedStatement preparedStatement,int index) throws SQLException {
        if (object==null)
            preparedStatement.setNull(index,type);
        else
            switch (type) {
            case Types.INTEGER:
                preparedStatement.setInt(index,((Integer)object).intValue());
                break;
            case Types.BOOLEAN:
                preparedStatement.setBoolean(index,((Boolean)object).booleanValue());
                break;
            case Types.VARCHAR:
            case TYPE_REGULAR_EXPRESSION:
            case TYPE_LIKE:
            default:
                preparedStatement.setString(index,object.toString());
                break;
            }
    }
    /**
     * Returns the object from the result set.
     *
     * @param type                              the type
     * @param resultSet                         the result set
     * @param index                             the index
     * @return                                  the object
     * @throws SQLException                     thrown if there is a problem
     */
    public Object getObjectFromResultSet(int type,ResultSet resultSet,int index) throws SQLException {
        switch (type) {
        case Types.BOOLEAN:
            return resultSet.getBoolean(index) ? Boolean.TRUE : Boolean.FALSE;
        case Types.INTEGER:
            return new Integer(resultSet.getInt(index));
        case Types.VARCHAR:
        case TYPE_REGULAR_EXPRESSION:
        case TYPE_LIKE:
        default:
            return resultSet.getString(index);
        }
    }
    /**
     * Returns the character representation of the type.
     *
     * @param type                              the type
     * @return                                  the character representation of the type
     */
    public String typeToString(int type) {
        switch (type) {
        case Types.INTEGER:
            return "i";
        case Types.BOOLEAN:
            return "b";
        default:
        case Types.VARCHAR:
        case TYPE_REGULAR_EXPRESSION:
        case TYPE_LIKE:
            return "t";
        }
    }
    /**
     * Returns the SQL name of the type.
     *
     * @param type                              the type
     * @return                                  the SQL name of the type
     */
    public String typeToSQLName(int type) {
        switch (type) {
        case Types.INTEGER:
            return "int";
        case Types.BOOLEAN:
            return "smallint";
        default:
        case Types.VARCHAR:
        case TYPE_REGULAR_EXPRESSION:
        case TYPE_LIKE:
            return "varchar(255)";
        }
    }
    /**
     * Formats the given constant for SQL.
     *
     * @param constant                          the constant to be formatted
     * @return                                  the formatted constant
     */
    public String constantAsSQL(Object constant) {
        if (constant==null)
            return "NULL";
        else if (constant instanceof Boolean)
            return ((Boolean)constant).booleanValue() ? "1" : "0";
        else if (constant instanceof Integer)
            return constant.toString();
        else
            return "'"+encodeApostrophes(constant.toString())+"'";
    }
    /**
     * Encodes the apostrophes in a literal according to the SQL requirements.
     *
     * @param value                             the value to be encoded
     * @return                                  the encoded value
     */
    public static String encodeApostrophes(String value) {
        int apostrophePosition=value.indexOf('\'');
        while (apostrophePosition!=-1) {
            value=value.substring(0,apostrophePosition)+"'"+value.substring(apostrophePosition);
            apostrophePosition=value.indexOf('\'',apostrophePosition+2);
        }
        return value;
    }
}