package edu.unika.aifb.kaon.datalog.jdbc;

import java.util.Iterator;

import edu.unika.aifb.kaon.datalog.*;
import edu.unika.aifb.kaon.datalog.program.*;

/**
 * The generator for the queries on SQL2-compatible databases.
 */
public class QueryGeneratorSQL2 extends AbstractQueryGenerator {
    /**
     * Creates an instance of this class.
     *
     * @param extensionalDatabase                   the extensional database
     * @param literals                              the array of literals
     * @param boundVariables                        the bound variables
     * @param returnedVariables                     the needed variables
     * @throws DatalogException                     thrown if there is an error
     */
    public QueryGeneratorSQL2(JDBCExtensionalDatabase extensionalDatabase,Literal[] literals,Variable[] boundVariables,Variable[] returnedVariables) throws DatalogException {
        super(extensionalDatabase,literals,boundVariables,returnedVariables);
    }
    /**
     * Generates the query.
     *
     * @param buffer                            the buffer receiving the query
     */
    protected void generateQuery(StringBuffer buffer) {
        buffer.append("SELECT DISTINCT ");
        for (int i=0;i<m_probeSQLNames.length;i++) {
            if (i!=0)
                buffer.append(',');
            buffer.append(m_probeSQLNames[i]);
        }
        boolean firstTable=true;
        if (m_buildJoinTypes.length!=0) {
            buffer.append(" FROM ");
            buffer.append(getBindingsTableName());
            buffer.append(" bindings");
            firstTable=false;
        }
        Iterator iterator=m_positiveLiteralInfos.iterator();
        while (iterator.hasNext()) {
            LiteralInfo literalInfo=(LiteralInfo)iterator.next();
            if (firstTable)
                buffer.append(" FROM ");
            else
                buffer.append(',');
            buffer.append(literalInfo.getTableName());
            buffer.append(' ');
            buffer.append(literalInfo.getAliasName());
            firstTable=false;
        }
        boolean firstLiteral=true;
        iterator=m_positiveLiteralInfos.iterator();
        while (iterator.hasNext()) {
            LiteralInfo literalInfo=(LiteralInfo)iterator.next();
            if (addConditions(literalInfo,buffer,firstLiteral ? " WHERE " : " AND "))
                firstLiteral=false;
        }
        iterator=m_negativeLiteralInfos.iterator();
        while (iterator.hasNext()) {
            LiteralInfo literalInfo=(LiteralInfo)iterator.next();
            if (firstLiteral)
                buffer.append(" WHERE ");
            else
                buffer.append(" AND ");
            processNegativeLiteralInfo(literalInfo,buffer);
            firstLiteral=false;
        }
    }
    /**
     * Processes the information about a negative literal.
     *
     * @param literalInfo                       the information about the literal
     * @param buffer                            the buffer
     */
    protected void processNegativeLiteralInfo(LiteralInfo literalInfo,StringBuffer buffer) {
        buffer.append("NOT EXISTS (SELECT * FROM ");
        buffer.append(literalInfo.getTableName());
        buffer.append(' ');
        buffer.append(literalInfo.getAliasName());
        addConditions(literalInfo,buffer," WHERE ");
        buffer.append(")");
    }
    /**
     * Appends the conditions to the buffer.
     *
     * @param literalInfo                       the information about the literal
     * @param buffer                            the buffer
     * @param firstConjunction                  the element printed instead of the first conjunction
     * @return                                  <code>true</code> if at least one condition was appended
     */
    protected boolean addConditions(LiteralInfo literalInfo,StringBuffer buffer,String firstConjunction) {
        boolean firstCondition=true;
        Iterator iterator=literalInfo.getJoinConditions();
        while (iterator.hasNext()) {
            Object[] joinCondition=(Object[])iterator.next();
            int variableIndex=((Integer)joinCondition[0]).intValue();
            if (firstCondition)
                buffer.append(firstConjunction);
            else
                buffer.append(" AND ");
            buffer.append(literalInfo.getColumnName(variableIndex));
            buffer.append((String)joinCondition[1]);
            buffer.append((String)joinCondition[2]);
            firstCondition=false;
        }
        iterator=literalInfo.getConstantConditions();
        while (iterator.hasNext()) {
            String condition=(String)iterator.next();
            if (firstCondition)
                buffer.append(firstConjunction);
            else
                buffer.append(" AND ");
            buffer.append(condition);
            firstCondition=false;
        }
        return !firstCondition;
    }
}
