package edu.unika.aifb.kaon.datalog.jdbc;

import java.sql.Connection;
import java.sql.Statement;
import java.sql.ResultSet;
import java.sql.SQLException;

import edu.unika.aifb.kaon.datalog.*;
import edu.unika.aifb.kaon.datalog.evaluation.*;

/**
 * The operator that evaluates an expression in the engineering server database without any bound values.
 */
public class NoBoundValuesExpressionOperator implements QueryOperator {
    /** The database that created this operator. */
    protected JDBCExtensionalDatabase m_extensionalDatabase;
    /** The current tuple. */
    protected Object[] m_currentTuple;
    /** The current connection. */
    protected Connection m_connection;
    /** The current statement. */
    protected Statement m_statement;
    /** The current result set. */
    protected ResultSet m_resultSet;
    /** The query producing the result. */
    protected String m_query;
    /** The types of the query result. */
    protected int[] m_queryResultTypes;

    /**
     * Creates an instance of this class.
     *
     * @param extensionalDatabase               the database
     * @param query                             the query producing the result
     * @param queryResultTypes                  the types of the query result
     */
    public NoBoundValuesExpressionOperator(JDBCExtensionalDatabase extensionalDatabase,String query,int[] queryResultTypes) {
        m_extensionalDatabase=extensionalDatabase;
        m_query=query;
        m_queryResultTypes=queryResultTypes;
    }
    /**
     * Starts the processing of the query. The cursor is positioned on the first tuple, or at end.
     *
     * @throws DatalogException         thrown if there is an error in evaluation
     */
    public void start() throws DatalogException {
        m_connection=getConnection();
        try {
            m_extensionalDatabase.notifyWillExecuteQuery(m_query);
            long startTime=System.currentTimeMillis();
            m_statement=m_connection.createStatement();
            m_resultSet=m_statement.executeQuery(m_query);
            next();
            long queryTime=System.currentTimeMillis()-startTime;
            m_extensionalDatabase.notifyQueryExecutionTime(queryTime);
        }
        catch (SQLException error) {
            try {
                closeQuery();
            }
            catch (DatalogException ignored) {
            }
            throw new DatalogException("SQL error",error);
        }
        catch (DatalogException error) {
            closeQuery();
            throw error;
        }
    }
    /**
     * Stops the processing of the query.
     *
     * @throws DatalogException         thrown if there is an error in evaluation
     */
    public void stop() throws DatalogException {
        try {
            closeQuery();
        }
        finally {
            m_connection=null;
        }
    }
    /**
     * Returns the current tuple of the operator. If the tuple stream is at the end, <code>null</code> is returned.
     *
     * @return                          the current tuple (or <code>null</code> if the tuple stream is at the end)
     */
    public Object[] tuple() {
        return m_currentTuple;
    }
    /**
     * Returns <code>true</code> if the current stream is after the last tuple.
     *
     * @return                          <code>true</code> if the stream is after the last tuple
     */
    public boolean afterLast() {
        return m_currentTuple==null;
    }
    /**
     * Moves the cursor to the next position. If the tuple stream is at the end this method call has no effect.
     *
     * @throws DatalogException         thrown if there is an error in evaluation
     */
    public void next() throws DatalogException {
        try {
            if (m_resultSet!=null && m_resultSet.next()) {
                TypeManager typeManager=m_extensionalDatabase.getDatabaseManager().getTypeManager();
                m_currentTuple=new Object[m_queryResultTypes.length];
                for (int i=0;i<m_queryResultTypes.length;i++)
                    m_currentTuple[i]=typeManager.getObjectFromResultSet(m_queryResultTypes[i],m_resultSet,i+1);
            }
            else
                m_currentTuple=null;
        }
        catch (SQLException error) {
            try {
                closeQuery();
            }
            catch (DatalogException ignored) {
            }
            throw new DatalogException("SQL error",error);
        }
    }
    /**
     * Returns the database connection.
     *
     * @return                          the database connection
     */
    protected Connection getConnection() {
        return m_extensionalDatabase.getConnection();
    }
    /**
     * Closes the current query.
     *
     * @throws DatalogException         thrown if there is an error
     */
    protected void closeQuery() throws DatalogException {
        m_currentTuple=null;
        ResultSet resultSet=m_resultSet;
        m_resultSet=null;
        Statement statement=m_statement;
        m_statement=null;
        try {
            try {
                if (resultSet!=null)
                    resultSet.close();
            }
            finally {
                if (statement!=null)
                    statement.close();
            }
        }
        catch (SQLException error) {
            throw new DatalogException("SQL error",error);
        }
    }
}
