package edu.unika.aifb.kaon.datalog.jdbc;

import java.util.List;
import java.util.LinkedList;
import java.util.Map;
import java.util.Iterator;
import java.sql.Statement;
import java.sql.PreparedStatement;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.SQLException;

import edu.unika.aifb.kaon.datalog.*;
import edu.unika.aifb.kaon.datalog.program.*;
import edu.unika.aifb.kaon.datalog.evaluation.*;

/**
 * The extensional database which wraps a JDBC database to datalog.
 */
public class JDBCExtensionalDatabase implements ExtensionalDatabase {
    /** The map of predicateDBinfo objects indexed by the predicate. */
    protected Map m_predicateDBInfos;
    /** The connection on which the database is working. */
    protected Connection m_connection;
    /** The database manager. */
    protected DatabaseManager m_databaseManager;
    /** The list of temporary tables. */
    protected List m_temporaryTables;

    /**
     * Creates an instance of this class.
     *
     * @param predicateDBInfos                  the map of predicateDBInfo objects for this database
     */
    public JDBCExtensionalDatabase(Map predicateDBInfos) {
        m_predicateDBInfos=predicateDBInfos;
        m_temporaryTables=new LinkedList();
    }
    /**
     * Returns the connection.
     *
     * @return                                  the connection (may be <code>null</code> if no connection is set)
     */
    public Connection getConnection() {
        return m_connection;
    }
    /**
     * Sets the connection.
     *
     * @param connection                        the connection (may be <code>null</code> if no connection is set)
     * @param databaseManager                   the database manager (may be <code>null</code> if no connection is set)
     * @throws DatalogException                 thrown if there is an error
     */
    public void setConnection(Connection connection,DatabaseManager databaseManager) throws DatalogException {
        if (m_connection!=null) {
            Iterator iterator=m_temporaryTables.iterator();
            while (iterator.hasNext()) {
                TemporaryTable temporaryTable=(TemporaryTable)iterator.next();
                temporaryTable.destroy();
                temporaryTable.close();
            }
            m_temporaryTables.clear();
        }
        m_connection=connection;
        if (m_connection!=null)
            m_databaseManager=databaseManager;
        else
            m_databaseManager=null;
    }
    /**
     * Returns <code>true</code> if supplied predicate can be evaluated in this database.
     *
     * @param predicate                         the predicate
     * @return                                  <code>true</code> if supplied predicate can be evaluated in this database
     */
    public boolean canEvaluatePredicate(Predicate predicate) {
        return m_predicateDBInfos.containsKey(predicate.getFullName());
    }
    /**
     * Returns <code>true</code> if supplied predicate in this database contains disjunction information.
     *
     * @param predicate                         the predicate
     * @return                                  <code>true</code> if supplied predicate in this database contains disjunction information
     */
    public boolean getContainsDisjunctionInfo(Predicate predicate) {
        return false;
    }
    /**
     * Creates an operator that evaluates specified literals using the values from the operator passed in.
     *
     * @param literals                          the literals to be evaluated
     * @param bindingsOperator                  the operator producing the bindings
     * @param boundVariables                    the variables of the bindings
     * @param returnedVariables                 the variables that the operator should return
     * @return                                  the query operator evaluating specified literals
     * @throws DatalogException                 thrown if there is an error
     */
    public QueryOperator createQueryOperator(Literal[] literals,QueryOperator bindingsOperator,Variable[] boundVariables,Variable[] returnedVariables) throws DatalogException {
        AbstractQueryGenerator queryGenerator=getQueryGenerator(literals,boundVariables,returnedVariables);
        return queryGenerator.getQueryOperator(bindingsOperator);
    }
    /**
     * Creates an operator that evaluatesspecified literals with no variables passed in.
     *
     * @param literals                          the literals to be evaluated
     * @param returnedVariables                 the variables that the operator should return
     * @return                                  the query operator evaluating specified literals
     * @throws DatalogException                 thrown if there is an error
     */
    public QueryOperator createQueryOperator(Literal[] literals,Variable[] returnedVariables) throws DatalogException {
        AbstractQueryGenerator queryGenerator=getQueryGenerator(literals,new Variable[0],returnedVariables);
        return queryGenerator.getQueryOperator(null);
    }
    /**
     * Returns the query generator.
     *
     * @param literals                          the literals
     * @param boundVariables                    the bound variables
     * @param returnedVariables                 the returned variables
     * @return                                  the query generator
     * @throws DatalogException                 thrown if there is an error
     */
    public AbstractQueryGenerator getQueryGenerator(Literal[] literals,Variable[] boundVariables,Variable[] returnedVariables) throws DatalogException {
        return new QueryGeneratorSQL2(this,literals,boundVariables,returnedVariables);
    }
    /**
     * Returns the database manager for the current connection.
     *
     * @return                                  the database manager
     */
    public DatabaseManager getDatabaseManager() {
        return m_databaseManager;
    }
    /**
     * Returns the temporary table of given arity.
     *
     * @param columnTypes                       the types of the columns
     * @return                                  the temporary table with given arity
     * @throws DatalogException                 thrown if there is an error
     */
    public TemporaryTable getTemporaryTable(int[] columnTypes) throws DatalogException {
        Iterator iterator=m_temporaryTables.iterator();
        nextTable: while (iterator.hasNext()) {
            TemporaryTable temporaryTable=(TemporaryTable)iterator.next();
            int[] tableColumnTypes=temporaryTable.getColumnTypes();
            if (columnTypes.length==tableColumnTypes.length) {
                for (int i=0;i<columnTypes.length;i++)
                    if (columnTypes[i]!=tableColumnTypes[i])
                        continue nextTable;
                return temporaryTable;
            }
        }
        TemporaryTable result=new TemporaryTable(columnTypes);
        m_temporaryTables.add(result);
        return result;
    }
    /**
     * Notifies this object that a query will be executed.
     *
     * @param queryString                       the query string
     */
    public void notifyWillExecuteQuery(String queryString) {
    }
    /**
     * Notifies this object that a query was finished in specified time.
     *
     * @param queryTime                         the time of the query
     */
    public void notifyQueryExecutionTime(long queryTime) {
    }
    /**
     * Notifies this object that insertion of bound values took specified amount of time.
     *
     * @param numberOfBoundValues               the number of bound values to be inserted
     * @param insertionTime                     the time it took to insert specified number of bound values
     */
    public void notifyInsertBoundValuesTime(int numberOfBoundValues,long insertionTime) {
    }
    /**
     * Initializes this literal information with initial conditions. Subclasses can override this to provide additional conditions.
     *
     * @param literalInfo                       information about the literal
     */
    public void addInitialConditions(AbstractQueryGenerator.LiteralInfo literalInfo) {
    }
    /**
     * Returns the information about database details for given predicate.
     *
     * @param predicate                         the predicate
     * @return                                  the database information for given predicate
     */
    public PredicateDBInfo getPredicateDBInfo(Predicate predicate) {
        return (PredicateDBInfo)m_predicateDBInfos.get(predicate.getFullName());
    }
    /**
     * Registers information about the predicate.
     *
     * @param predicateDBInfos                  the map of predicateDBInfo objects
     * @param isPositive                        set to <code>true</code> if the predicate is positive
     * @param predicateName                     the simple name of the predicate
     * @param tableName                         the name of the table in the database
     * @param columns                           the names of the columns
     * @param columnTypes                       specifies the types of the columns
     */
    public static void registerPredicateDBInfo(Map predicateDBInfos,boolean isPositive,String predicateName,String tableName,String[] columns,int[] columnTypes) {
        PredicateDBInfo predicateDBInfo=new PredicateDBInfo(isPositive,predicateName,tableName,columns,columnTypes);
        predicateDBInfos.put(predicateDBInfo.m_predicateFullName,predicateDBInfo);
    }
    /**
     * Examines the connection and returns the appropriate database manager.
     *
     * @param connection                    the connection
     * @return                              the database manager approrpate for the connection
     * @throws DatalogException             thrown if there is an error
     */
    public static DatabaseManager createDatabaseManager(Connection connection) throws DatalogException {
        try {
            DatabaseMetaData databaseMetaData=connection.getMetaData();
            String productName=databaseMetaData.getDatabaseProductName();
            if ("Microsoft SQL Server".equals(productName))
                return new DatabaseManagerLocalTables() {
                    protected String getTemporaryTableUnadornedName() {
                        return "#TemporaryTable";
                    }
                    protected String getCreateTemporaryTablePrefix() {
                        return "CREATE TABLE ";
                    }
                };
            else if ("PostgreSQL".equals(productName))
                return new DatabaseManagerLocalTables() {
                    protected void startCast(StringBuffer buffer,int columnType) {
                        buffer.append("CAST(");
                    }
                    protected void endCast(StringBuffer buffer,int columnType) {
                        buffer.append(" AS ");
                        buffer.append(m_typeManager.typeToSQLName(columnType));
                        buffer.append(")");
                    }
                };
            else if (productName.startsWith("DB2"))
                return new DatabaseManagerLocalTables() {
                    protected String getTemporaryTableUnadornedName() {
                        return "session.TemporaryTable";
                    }
                    protected String getCreateTemporaryTablePrefix() {
                        return "DECLARE GLOBAL TEMPORARY TABLE ";
                    }
                    protected String getCreateTemporaryTableSuffix() {
                        return " ON COMMIT DELETE ROWS NOT LOGGED";
                    }
                    public boolean passBindingsThroughTemporaryTable(int numberOfBindings,int[] columnTypes) {
                        return true;
                    }
                };
            else if ("Oracle".equals(productName))
                return new DatabaseManager() {
                    public boolean modelIDAsInnerSelect() {
                        return true;
                    }
                };
            else
                throw new DatalogException("Unsupported database '"+productName+"'");
        }
        catch (SQLException e) {
            throw new DatalogException("SQL error",e);
        }
    }

    /**
     * The information about the predicates in the database.
     */
    public static class PredicateDBInfo {
        /** The full name of the predicate. */
        protected String m_predicateFullName;
        /** The name of the table. */
        protected String m_tableName;
        /** The array of names of columns bound to specified positions in the predicate. */
        protected String[] m_columns;
        /** Specifies the types of the columns. */
        protected int[] m_columnTypes;

        public PredicateDBInfo(boolean isPositive,String predicateName,String tableName,String[] columns,int[] columnTypes) {
            m_columns=columns;
            m_columnTypes=columnTypes;
            m_predicateFullName=(isPositive ? "" : "-")+predicateName+"/"+m_columns.length;
            m_tableName=tableName;
        }
        public String getTableName() {
            return m_tableName;
        }
        public String getColumnName(int columnIndex) {
            return m_columns[columnIndex];
        }
        public int getColumnType(int columnIndex) {
            return m_columnTypes[columnIndex];
        }
    }

    /**
     * Contains information about a temporary table.
     */
    public class TemporaryTable {
        /** The types of columns of the table. */
        protected int[] m_columnTypes;
        /** The name of the temporary table. */
        protected String m_tableName;
        /** The statement for inserting elements into the table. */
        protected PreparedStatement m_insertElementsStatement;
        /** The statement for deleting elements from the table. */
        protected PreparedStatement m_emptyTableStatement;

        /**
         * Creates a temporary table of given arity.
         *
         * @param columnTypes                   the types of the columns of the table
         * @throws DatalogException             thrown if there is an error
         */
        public TemporaryTable(int[] columnTypes) throws DatalogException {
            m_columnTypes=columnTypes;
            m_tableName=m_databaseManager.getTemporaryTableName(m_columnTypes);
            StringBuffer buffer=new StringBuffer();
            buffer.append("INSERT INTO ");
            buffer.append(m_tableName);
            buffer.append(" (");
            for (int i=0;i<m_columnTypes.length;i++) {
                if (i!=0)
                    buffer.append(',');
                buffer.append(m_databaseManager.getTypeManager().typeToString(m_columnTypes[i]));
                buffer.append(i);
            }
            buffer.append(") VALUES (");
            for (int i=0;i<m_columnTypes.length;i++) {
                if (i!=0)
                    buffer.append(',');
                buffer.append('?');
            }
            buffer.append(')');
            try {
                String createTableSQL=m_databaseManager.getCreateTempoararyTableSQL(m_columnTypes);
                if (createTableSQL!=null) {
                    Statement statement=m_connection.createStatement();
                    try {
                        statement.executeUpdate(createTableSQL);
                    }
                    finally {
                        statement.close();
                    }
                }
                m_insertElementsStatement=m_connection.prepareStatement(buffer.toString());
                m_emptyTableStatement=m_connection.prepareStatement(m_databaseManager.getEmptyTemporaryTableSQL(m_columnTypes));
            }
            catch (SQLException error) {
                close();
                throw new DatalogException("SQL error",error);
            }
        }
        /**
         * Returns the types of the columns of the table.
         *
         * @return                              the types of columns of the table
         */
        public int[] getColumnTypes() {
            return m_columnTypes;
        }
        /**
         * Returns the name of the table.
         *
         * @return                              the name of the table
         */
        public String getTableName() {
            return m_tableName;
        }
        /**
         * Returns the statement for inserting elements.
         *
         * @return                              the statement for inserting elements
         */
        public PreparedStatement getInsertElementsStatement() {
            return m_insertElementsStatement;
        }
        /**
         * Deletes the table.
         *
         * @throws DatalogException             thrown if there is an error
         */
        public void delete() throws DatalogException {
            try {
                m_emptyTableStatement.execute();
            }
            catch (SQLException error) {
                throw new DatalogException("SQL error",error);

            }
        }
        /**
         * Closes the table and prepares the object for disposal.
         */
        public void close() {
            if (m_insertElementsStatement!=null) {
                try {
                    m_insertElementsStatement.close();
                }
                catch (SQLException ignored) {
                }
                m_insertElementsStatement=null;
            }
            if (m_emptyTableStatement!=null) {
                try {
                    m_emptyTableStatement.close();
                }
                catch (SQLException ignored) {
                }
                m_emptyTableStatement=null;
            }
        }
        /**
         * Destroys this table.
         */
        public void destroy() {
            String dropTableSQL=m_databaseManager.getDropTableSQL(m_columnTypes);
            if (dropTableSQL!=null) {
                try {
                    Statement statement=m_connection.createStatement();
                    try {
                        statement.executeUpdate(dropTableSQL);
                    }
                    finally {
                        statement.close();
                    }
                }
                catch (SQLException ignored) {
                }
            }
       }
    }
}
