package edu.unika.aifb.kaon.datalog.jdbc;

import java.util.Iterator;

/**
 * The database manager for databases using local temporary tables (MS SQL and PostgreSQL).
 */
public class DatabaseManagerLocalTables extends DatabaseManager {
    /**
     * Returns the name of the temporary table with given types of columns.
     *
     * @param columnTypes                   the types of the columns
     * @return                              the name of the table with given types of columns
     */
    public String getTemporaryTableName(int[] columnTypes) {
        StringBuffer buffer=new StringBuffer();
        buffer.append(getTemporaryTableUnadornedName());
        buffer.append('_');
        for (int i=0;i<columnTypes.length;i++)
            buffer.append(m_typeManager.typeToString(columnTypes[i]));
        return buffer.toString();
    }
    /**
     * Returns the SQL statement for creating the temporary table with given types of columns.
     *
     * @param columnTypes                   the types of the columns
     * @return                              the statement for creating a temporary table with given types of columns (or <code>null</code> if table is part of the schema)
     */
    public String getCreateTempoararyTableSQL(int[] columnTypes) {
        StringBuffer buffer=new StringBuffer();
        buffer.append(getCreateTemporaryTablePrefix());
        buffer.append(getTemporaryTableName(columnTypes));
        buffer.append(" (");
        for (int i=0;i<columnTypes.length;i++) {
            if (i!=0)
                buffer.append(',');
            buffer.append(m_typeManager.typeToString(columnTypes[i]));
            buffer.append(i);
            buffer.append(' ');
            buffer.append(m_typeManager.typeToSQLName(columnTypes[i]));
        }
        buffer.append(")");
        buffer.append(getCreateTemporaryTableSuffix());
        return buffer.toString();
    }
    /**
     * Returns the SQL statement for destroying the temporary table with given types of columns.
     *
     * @param columnTypes                   the types of the columns
     * @return                              the SQL statement for deleting the table
     */
    public String getDropTableSQL(int[] columnTypes) {
        return "DROP TABLE "+getTemporaryTableName(columnTypes);
    }
    /**
     * Checks whether this database should use the temporary table for passing bindings.
     *
     * @param numberOfBindings              the number of bindings
     * @param columnTypes                   the types of columns
     * @return                              <code>true</code> if temporary table should be used to pass bindings
     */
    public boolean passBindingsThroughTemporaryTable(int numberOfBindings,int[] columnTypes) {
        return numberOfBindings>200;
    }
    /**
     * If temporary table is not used, this method should modify the query with literals
     * that specify the bindings;
     *
     * @param columnTypes                   the types of columns
     * @param buildJoinIndices              the join indices of the build stream
     * @param boundValues                   the iterator specifying the bound values
     * @param originalQuery                 the original query
     * @return                              the modified query
     */
    public String modifyQueryWithBindings(int[] columnTypes,int[] buildJoinIndices,Iterator boundValues,String originalQuery) {
        String temporaryTableName=getTemporaryTableName(columnTypes);
        int startIndex=originalQuery.indexOf(temporaryTableName);
        if (startIndex==-1)
            return originalQuery;
        else {
            StringBuffer buffer=new StringBuffer();
            buffer.append(originalQuery.substring(0,startIndex));
            buffer.append('(');
            while (boundValues.hasNext()) {
                Object[] tuple=(Object[])boundValues.next();
                buffer.append("SELECT ");
                for (int i=0;i<buildJoinIndices.length;i++) {
                    int buildJoinIndex=buildJoinIndices[i];
                    if (i!=0)
                        buffer.append(',');
                    Object object=tuple[buildJoinIndex];
                    startCast(buffer,columnTypes[i]);
                    String constantAsSQL=m_typeManager.constantAsSQL(object);
                    buffer.append(constantAsSQL);
                    endCast(buffer,columnTypes[i]);
                    buffer.append(" AS ");
                    buffer.append(m_typeManager.typeToString(columnTypes[i]));
                    buffer.append(i);
                }
                if (boundValues.hasNext())
                    buffer.append(" UNION ALL ");
            }
            buffer.append(')');
            buffer.append(originalQuery.substring(startIndex+temporaryTableName.length()));
            return buffer.toString();
        }
    }
    /**
     * Returns the unadorned name of the temporary table.
     *
     * @return                              the unadorned name of the temporary table
     */
    protected String getTemporaryTableUnadornedName() {
        return "TemporaryTable";
    }
    /**
     * Returns the prefix of the string for creating the temporary table.
     *
     * @return                              the prefix string for creating temporary table
     */
    protected String getCreateTemporaryTablePrefix() {
        return "CREATE LOCAL TEMPORARY TABLE ";
    }
    /**
     * Returns the suffix of the string for creating the temporary table.
     *
     * @return                              the suffix string for creating temporary table
     */
    protected String getCreateTemporaryTableSuffix() {
        return "";
    }
    /**
     * Appends the start of the cast if necessary.
     *
     * @param buffer                        the buffer receiving the cast
     * @param columnType                    the type of the column
     */
    protected void startCast(StringBuffer buffer,int columnType) {
    }
    /**
     * Ends the cast if necessary.
     *
     * @param buffer                        the buffer receiving the cast
     * @param columnType                    the type of the column
     */
    protected void endCast(StringBuffer buffer,int columnType) {
    }
}
