package edu.unika.aifb.kaon.datalog.jdbc;

import java.util.Iterator;
import java.sql.Connection;
import java.sql.Statement;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

import edu.unika.aifb.kaon.datalog.*;
import edu.unika.aifb.kaon.datalog.evaluation.*;

/**
 * The operator that evaluates an expression in the engineering server database by using bound values.
 */
public class BoundValuesExpressionOperator extends AbstractHashJoin {
    /** The database that created this operator. */
    protected JDBCExtensionalDatabase m_extensionalDatabase;
    /** The current probe tuple. */
    protected Object[] m_probeTuple;
    /** The current connection. */
    protected Connection m_connection;
    /** The current statement. */
    protected Statement m_statement;
    /** The current result set. */
    protected ResultSet m_resultSet;
    /** The query producing the result. */
    protected String m_query;
    /** The types of the query result. */
    protected int[] m_queryResultTypes;
    /** The types of the columns of the build stream participating in the join. */
    protected int[] m_buildJoinTypes;
    /** The bindings table. */
    protected JDBCExtensionalDatabase.TemporaryTable m_bindingsTable;

    /**
     * Creates an instance of this class.
     *
     * @param extensionalDatabase               the database
     * @param bindingsOperator                  the operator producing bindings
     * @param query                             the query
     * @param queryResultTypes                  the types of the query result
     * @param buildJoinTypes                    the types of columns of the build stream participating in the join
     * @param buildJoinIndices                  the indices of the build participating in the join
     * @param buildNonJoinIndices               the indices of the build not participating in the join
     * @param buildIndicesToCopy                the pairs of indices to copy from the build stream
     * @param probeJoinIndices                  the indices of the probe (query) participating in join
     * @param probeNonJoinIndices               the indices of the probe (query) not participating in join
     * @param probeIndicesToCopy                the pairs of indices to copy from the probe (query) stream
     */
    public BoundValuesExpressionOperator(JDBCExtensionalDatabase extensionalDatabase,QueryOperator bindingsOperator,String query,int[] queryResultTypes,int[] buildJoinTypes,int[] buildJoinIndices,int[] buildNonJoinIndices,int[][] buildIndicesToCopy,int[] probeJoinIndices,int[] probeNonJoinIndices,int[][] probeIndicesToCopy) {
        super(bindingsOperator,buildJoinIndices,buildNonJoinIndices,probeJoinIndices,probeNonJoinIndices,buildIndicesToCopy,probeIndicesToCopy,Filters.ALWAYS_TRUE_JOIN_TUPLE_FILTER);
        m_extensionalDatabase=extensionalDatabase;
        m_query=query;
        m_queryResultTypes=queryResultTypes;
        m_buildJoinTypes=buildJoinTypes;
    }
    /**
     * Stops the processing of the query.
     *
     * @throws DatalogException         thrown if there is an error in evaluation
     */
    public void stop() throws DatalogException {
        try {
            super.stop();
        }
        finally {
            try {
                closeQuery();
            }
            finally {
                m_connection=null;
            }
        }
    }
    /**
     * Returns the database connection.
     *
     * @return                          the database connection
     */
    protected Connection getConnection() {
        return m_extensionalDatabase.getConnection();
    }
    /**
     * Closes the current query.
     *
     * @throws DatalogException         thrown if there is an error
     */
    protected void closeQuery() throws DatalogException {
        m_probeTuple=null;
        ResultSet resultSet=m_resultSet;
        m_resultSet=null;
        Statement statement=m_statement;
        m_statement=null;
        try {
            try {
                if (m_bindingsTable!=null)
                    m_bindingsTable.delete();
            }
            finally {
                m_bindingsTable=null;
                try {
                    if (resultSet!=null)
                        resultSet.close();
                }
                finally {
                    if (statement!=null)
                        statement.close();
                }
            }
        }
        catch (SQLException error) {
            throw new DatalogException("SQL error",error);
        }
    }
    /**
     * Called after the build stream is processed. Probe is generated at this point from the keys of the join map.
     *
     * @throws DatalogException                 thrown if there is an error in evaluation
     */
    protected void buildProcessed() throws DatalogException {
        if (m_joinMap.size()==0)
            m_probeTuple=null;
        else {
            m_connection=getConnection();
            try {
                String query=null;
                if (!m_extensionalDatabase.getDatabaseManager().passBindingsThroughTemporaryTable(m_joinMap.size(),m_buildJoinTypes))
                    query=m_extensionalDatabase.getDatabaseManager().modifyQueryWithBindings(m_buildJoinTypes,m_buildJoinIndices,m_joinMap.keyIterator(),m_query);
                if (query==null) {
                    insertBoundValues();
                    query=m_query;
                }
                m_extensionalDatabase.notifyWillExecuteQuery(query);
                long startTime=System.currentTimeMillis();
                m_statement=m_connection.createStatement();
                m_resultSet=m_statement.executeQuery(query);
                probeNext();
                long queryTime=System.currentTimeMillis()-startTime;
                m_extensionalDatabase.notifyQueryExecutionTime(queryTime);
            }
            catch (SQLException error) {
                try {
                    closeQuery();
                }
                catch (DatalogException ignored) {
                }
                throw new DatalogException("SQL error",error);
            }
            catch (DatalogException error) {
                closeQuery();
                throw error;
            }
        }
    }
    /**
     * Returns the current tuple of the probe stream. If the stream is at the end, <code>null</code> is returned.
     *
     * @return                                  the current tuple of the probe stream (or <code>null</code> if the stream is at the end)
     */
    protected Object[] probeTuple() {
        return m_probeTuple;
    }
    /**
     * Moves the probe stream to the next element. If the probe stream is at the end, this method has no effects.
     *
     * @throws DatalogException                 thrown if there is an error in evaluation
     */
    protected void probeNext() throws DatalogException {
        try {
            if (m_resultSet!=null && m_resultSet.next()) {
                TypeManager typeManager=m_extensionalDatabase.getDatabaseManager().getTypeManager();
                m_probeTuple=new Object[m_queryResultTypes.length];
                for (int i=0;i<m_queryResultTypes.length;i++)
                    m_probeTuple[i]=typeManager.getObjectFromResultSet(m_queryResultTypes[i],m_resultSet,i+1);
            }
            else
                m_probeTuple=null;
        }
        catch (SQLException error) {
            try {
                closeQuery();
            }
            catch (DatalogException ignored) {
            }
            throw new DatalogException("SQL error",error);
        }
    }
    /**
     * Returns <code>true</code> if the probe stream has completely been used up.
     *
     * @return                                  <code>true</code> if the probe stream has been used up
     */
    protected boolean probeAfterLast() {
        return m_probeTuple==null;
    }
    /**
     * Inserts the bound values into the join table.
     *
     * @throws DatalogException         thrown if there is an error with inserting the bound values
     */
    protected void insertBoundValues() throws DatalogException {
        if (m_buildJoinIndices.length!=0) {
            long startTime=System.currentTimeMillis();
            m_bindingsTable=m_extensionalDatabase.getTemporaryTable(m_buildJoinTypes);
            try {
                TypeManager typeManager=m_extensionalDatabase.getDatabaseManager().getTypeManager();
                PreparedStatement insertJoinValues=m_bindingsTable.getInsertElementsStatement();
                insertJoinValues.clearBatch();
                Iterator iterator=m_joinMap.keyIterator();
                while (iterator.hasNext()) {
                    Object[] keyTuple=(Object[])iterator.next();
                    for (int i=0;i<m_buildJoinIndices.length;i++) {
                        int buildJoinIndex=m_buildJoinIndices[i];
                        Object object=keyTuple[buildJoinIndex];
                        typeManager.setObjectToStatement(m_buildJoinTypes[i],object,insertJoinValues,i+1);
                    }
                    insertJoinValues.addBatch();
                }
                insertJoinValues.executeBatch();
                long insertionTime=System.currentTimeMillis()-startTime;
                m_extensionalDatabase.notifyInsertBoundValuesTime(m_joinMap.size(),insertionTime);
            }
            catch (SQLException error) {
                throw new DatalogException("SQL error",error);
            }
        }
    }
}
