package edu.unika.aifb.kaon.datalog.jdbc;

import java.util.List;
import java.util.LinkedList;
import java.util.ArrayList;
import java.util.Map;
import java.util.HashMap;
import java.util.Set;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.Collection;
import java.util.Iterator;
import java.util.Collections;
import java.util.StringTokenizer;
import java.util.regex.Pattern;

import edu.unika.aifb.kaon.datalog.*;
import edu.unika.aifb.kaon.datalog.evaluation.*;
import edu.unika.aifb.kaon.datalog.program.*;

/**
 * The generator for the query that is used to evaluate the expression.
 */
public abstract class AbstractQueryGenerator {
    /** The extensional database for which the generator has been instantiated. */
    protected JDBCExtensionalDatabase m_extensionalDatabase;
    /** The set of join variables. */
    protected Set m_join;
    /** The ordering of the bindings. */
    protected List m_buildJoinOrdering;
    /** The set of floundering variables. */
    protected Set m_flounderingVariables;
    /** The list of regular expression variables. */
    protected List m_regularExpressionVariables;
    /** The list of variables used for filtering by regular exressions. */
    protected List m_regularExpressionsFilterOnVariables;
    /** The indices used for filtering by regular expressions. */
    protected int[] m_regularExpressionsFilterOnIndices;
    /** The list of regular expressions. */
    protected List m_regularExpressions;
    /** The map associating variables with their SQL names. */
    protected Map m_sqlNamesByVariables;
    /** The map associating variable types with their names. */
    protected Map m_typesByVariables;
    /** The list containing all positive literal infos. */
    protected List m_positiveLiteralInfos;
    /** The list containing all negative literals infos. */
    protected List m_negativeLiteralInfos;
    /** The types of the join columns in the build stream. */
    protected int[] m_buildJoinTypes;
    /** The indices of the join columns in the build stream. */
    protected int[] m_buildJoinIndices;
    /** The the indices of the non-join columns in the build stream. */
    protected int[] m_buildNonJoinIndices;
    /** The indices of the build stream to copy to result. */
    protected int[][] m_buildIndicesToCopy;
    /** The indices of the probe occuring in the join. */
    protected int[] m_probeJoinIndices;
    /** The indices of the probe not occuring in the join. */
    protected int[] m_probeNonJoinIndices;
    /** The indices of the probe to copy. */
    protected int[][] m_probeIndicesToCopy;
    /** The array of SQL names for the probe. */
    protected String[] m_probeSQLNames;
    /** The array of types of probe columns. */
    protected int[] m_probeTypes;

    /**
     * Creates an instance of this class and creates all necessary joins.
     *
     * @param extensionalDatabase               the extensional database
     * @param literals                          the array of literals
     * @param boundVariables                    the bound variables
     * @param returnedVariables                 the needed variables
     * @throws DatalogException                 thrown if there is a problem with compiling a query
     */
    public AbstractQueryGenerator(JDBCExtensionalDatabase extensionalDatabase,Literal[] literals,Variable[] boundVariables,Variable[] returnedVariables) throws DatalogException {
        m_extensionalDatabase=extensionalDatabase;
        m_flounderingVariables=new HashSet();
        m_sqlNamesByVariables=new HashMap();
        m_typesByVariables=new HashMap();
        m_positiveLiteralInfos=new LinkedList();
        m_negativeLiteralInfos=new LinkedList();
        m_regularExpressionVariables=new ArrayList();
        m_regularExpressionsFilterOnVariables=new ArrayList();
        m_regularExpressions=new ArrayList();
        // compute some elementary sets
        Set bindings=getVariableSet(boundVariables);
        Set returned=getVariableSet(returnedVariables);
        // body := { v | v \in L_i }
        Set body=buildBodyVariables(literals);
        // join := bindings \cap body
        m_join=new LinkedHashSet(bindings);
        m_join.retainAll(body);
        // probe := (returned \cap body) \cup join
        // NOTE: it must be returned \cap body, since in this way we guarrantee that all returned variables occur in the order requested
        // This has consequences on NoBoundValiesExpressionOperator!
        Set probe=new LinkedHashSet(returned);
        probe.retainAll(body);
        probe.addAll(m_join);
        // this is the ordering for the variables from the build stream participating in the join
        m_buildJoinOrdering=getVariableList(boundVariables);
        m_buildJoinOrdering.retainAll(m_join);
        // now process the literals
        buildLiteralInfos(literals);
        // create the resulting ordering
        List returnedOrdering=getVariableList(returnedVariables);
        returnedOrdering.addAll(m_regularExpressionVariables);
        // now process the build stream
        List buildOrdering=getVariableList(boundVariables);
        m_buildJoinIndices=getVariableIndices(buildOrdering,m_join);
        Set buildNonJoin=new LinkedHashSet(bindings);
        buildNonJoin.removeAll(m_join);
        m_buildNonJoinIndices=getVariableIndices(buildOrdering,buildNonJoin);
        m_buildJoinTypes=getVariableTypes(m_buildJoinOrdering);
        Set variablesToReturn=new LinkedHashSet(returned);
        variablesToReturn.retainAll(bindings);
        m_buildIndicesToCopy=getIndicesToCopy(buildOrdering,returnedOrdering,variablesToReturn);
        // now process the probe stream
        List probeOrdering=new ArrayList(probe);
        probeOrdering.addAll(m_regularExpressionVariables);
        m_probeJoinIndices=getVariableIndices(probeOrdering,m_join);
        Set probeNonJoin=new LinkedHashSet(probe);
        probeNonJoin.removeAll(m_join);
        probeNonJoin.addAll(m_regularExpressionVariables);
        m_probeNonJoinIndices=getVariableIndices(probeOrdering,probeNonJoin);
        variablesToReturn=new LinkedHashSet(returned);
        variablesToReturn.removeAll(bindings);
        variablesToReturn.addAll(m_regularExpressionVariables);
        m_probeIndicesToCopy=getIndicesToCopy(probeOrdering,returnedOrdering,variablesToReturn);
        // now create the probe stream information
        buildProbe(probeOrdering);
        // now create the indices of regular expressions variables
        m_regularExpressionsFilterOnIndices=new int[m_regularExpressionsFilterOnVariables.size()];
        for (int i=0;i<m_regularExpressionsFilterOnVariables.size();i++) {
            Variable variable=(Variable)m_regularExpressionsFilterOnVariables.get(i);
            m_regularExpressionsFilterOnIndices[i]=returnedOrdering.indexOf(variable);
        }
    }
    /**
     * Builds the set of all body variables.
     *
     * @param literals                          the array of literals
     * @return                                  the set of body variables
     */
    protected Set buildBodyVariables(Literal[] literals) {
        Set bodyVariables=new LinkedHashSet();
        for (int literalIndex=0;literalIndex<literals.length;literalIndex++) {
            Literal literal=literals[literalIndex];
            for (int variableIndex=0;variableIndex<literal.getArity();variableIndex++) {
                Variable variable=literal.getArgumentVariable(variableIndex);
                if (variable!=null)
                    bodyVariables.add(variable);
            }
        }
        return bodyVariables;
    }
    /**
     * Converts the array of variables to set.
     *
     * @param variables                         the array of variables
     * @return                                  the set of variables
     */
    protected Set getVariableSet(Variable[] variables) {
        Set result=new LinkedHashSet(variables.length);
        for (int i=0;i<variables.length;i++)
            result.add(variables[i]);
        return result;
    }
    /**
     * Converts the array of variables to list.
     *
     * @param variables                         the array of variables
     * @return                                  the list of variables
     */
    protected List getVariableList(Variable[] variables) {
        List result=new ArrayList(variables.length);
        for (int i=0;i<variables.length;i++)
            result.add(variables[i]);
        return result;
    }
    /**
     * Returns the list of indices of some variables.
     *
     * @param variableOrdering                  the list of variables defining the ordering
     * @param requiredVariables                 the variables whose indices are required
     * @return                                  the array of indices of variables
     * @throws DatalogException                 thrown if there is an error
     */
    protected int[] getVariableIndices(List variableOrdering,Collection requiredVariables) throws DatalogException {
        if (!variableOrdering.containsAll(requiredVariables))
            throw new DatalogException("Internal error: some required variable doesn't appear in the ordering.");
        int[] result=new int[requiredVariables.size()];
        int counter=0;
        int index=0;
        Iterator iterator=variableOrdering.iterator();
        while (iterator.hasNext()) {
            Variable variable=(Variable)iterator.next();
            if (requiredVariables.contains(variable))
                result[index++]=counter;
            counter++;
        }
        return result;
    }
    /**
     * Returns the types of the variables.
     *
     * @param variables                         the list of variables for which types should be returned
     * @return                                  the array of types of required variables
     * @throws DatalogException                 thrown if there is an error
     */
    protected int[] getVariableTypes(List variables) throws DatalogException {
        int[] result=new int[variables.size()];
        int index=0;
        Iterator iterator=variables.iterator();
        while (iterator.hasNext()) {
            Variable variable=(Variable)iterator.next();
            result[index]=getVariableType(variable);
            if (result[index]==-1)
                throw new DatalogException("Variable '"+variable+"' has an undefined type.");
            index++;
        }
        return result;
    }
    /**
     * Returns the pairs of indices to copy.
     *
     * @param sourceVariableOrdering            the list of variables defining the source ordering
     * @param targetVariableOrdering            the list of variables defining the target ordering
     * @param requiredVariables                 the variables whose types are required
     * @return                                  pairs of indices to copy
     * @throws DatalogException                 thrown if there is an error
     */
    protected int[][] getIndicesToCopy(List sourceVariableOrdering,List targetVariableOrdering,Collection requiredVariables) throws DatalogException {
        if (!sourceVariableOrdering.containsAll(requiredVariables))
            throw new DatalogException("Internal error: some required variable doesn't appear in the source ordering.");
        if (!targetVariableOrdering.containsAll(requiredVariables))
            throw new DatalogException("Internal error: some required variable doesn't appear in the target ordering.");
        int[][] result=new int[requiredVariables.size()][2];
        int counter=0;
        int index=0;
        Iterator iterator=sourceVariableOrdering.iterator();
        while (iterator.hasNext()) {
            Variable variable=(Variable)iterator.next();
            if (requiredVariables.contains(variable)) {
                result[index][0]=counter;
                result[index][1]=targetVariableOrdering.indexOf(variable);
                index++;
            }
            counter++;
        }
        return result;
    }
    /**
     * Processes a regular expression into a condition.
     *
     * @param literalInfo                       the information about the literal
     * @param variableIndex                     the index of the variable
     * @param regularExpression                 the regular expression
     */
    protected void processRegularExpression(LiteralInfo literalInfo,int variableIndex,String regularExpression) {
        Variable regularExpressionVariable=new Variable("regexp$internal"+m_regularExpressionVariables.size());
        m_regularExpressionVariables.add(regularExpressionVariable);
        String columnName=literalInfo.getColumnName(variableIndex);
        m_sqlNamesByVariables.put(regularExpressionVariable,columnName);
        m_typesByVariables.put(regularExpressionVariable,new Integer(TypeManager.TYPE_REGULAR_EXPRESSION));
        StringBuffer buffer=new StringBuffer();
        StringTokenizer tokenizer=new StringTokenizer(regularExpression);
        int numberOfConditions=0;
        while (tokenizer.hasMoreTokens()) {
            String token=tokenizer.nextToken();
            String dbToken=token.replaceAll("\\*","");
            if (numberOfConditions!=0)
                buffer.append(" OR ");
            buffer.append("upper(");
            buffer.append(columnName);
            buffer.append(") LIKE '%");
            String encodedToken=TypeManager.encodeApostrophes(dbToken.toUpperCase());
            buffer.append(encodedToken);
            buffer.append("%'");
            numberOfConditions++;
            StringBuffer filterExpression=new StringBuffer();
            filterExpression.append("((.*\\s+)|^)");
            int lastStarIndex=0;
            int starIndex;
            do {
                starIndex=token.indexOf('*',lastStarIndex);
                String subString;
                if (starIndex!=-1)
                    subString=token.substring(lastStarIndex,starIndex);
                else
                    subString=token.substring(lastStarIndex);
                if (subString.length()!=0) {
                    filterExpression.append("\\Q");
                    filterExpression.append(subString);
                    filterExpression.append("\\E");
                }
                if (starIndex!=-1) {
                    filterExpression.append(".*");
                    lastStarIndex=starIndex+1;
                }
            } while (starIndex!=-1);
            filterExpression.append("((\\s+.*)|$)");
            String expression=filterExpression.toString();
            m_regularExpressionsFilterOnVariables.add(regularExpressionVariable);
            m_regularExpressions.add(expression);
        }
        if (numberOfConditions!=0) {
            if (numberOfConditions>1) {
                buffer.insert(0,'(');
                buffer.append(')');
            }
            literalInfo.addConstantCondition(buffer.toString());
        }
    }
    /**
     * Builds the literal infos from the body.
     *
     * @param literals                          the array of literals
     * @throws DatalogException                 thrown if there is an error
     */
    protected void buildLiteralInfos(Literal[] literals) throws DatalogException {
        for (int literalIndex=0;literalIndex<literals.length;literalIndex++) {
            Literal literal=literals[literalIndex];
            LiteralInfo literalInfo=new LiteralInfo(m_extensionalDatabase.getPredicateDBInfo(literal.getPredicate()),"literal"+literalIndex);
            m_extensionalDatabase.addInitialConditions(literalInfo);
            for (int variableIndex=0;variableIndex<literal.getArity();variableIndex++) {
                Variable variable=literal.getArgumentVariable(variableIndex);
                if (variable!=null) {
                    if (!generateSQLNameForVariable(literal,literalInfo,variableIndex)) {
                        String sqlName=getSQLNameForVariable(variable);
                        literalInfo.addJoinCondition(variableIndex,"=",sqlName);
                    }
                }
                else if (literal.isArgumentBoundToConstant(variableIndex)) {
                    Object constant=literal.getArgumentConstant(variableIndex).getValue();
                    if (constant==null)
                        literalInfo.addConstantCondition(variableIndex," IS ",null);
                    else {
                        int columnType=literalInfo.getColumnType(variableIndex);
                        if (columnType==TypeManager.TYPE_LIKE) {
                            String sqlConstant=m_extensionalDatabase.getDatabaseManager().getTypeManager().constantAsSQL(constant);
                            literalInfo.addConstantCondition(variableIndex," LIKE ",sqlConstant);
                        }
                        else if (columnType==TypeManager.TYPE_REGULAR_EXPRESSION) {
                            if (!literal.isPositive())
                                throw new DatalogException("Regular expression arguments can't be bound in negative literals.");
                            processRegularExpression(literalInfo,variableIndex,constant.toString());
                        }
                        else {
                            String sqlConstant=m_extensionalDatabase.getDatabaseManager().getTypeManager().constantAsSQL(constant);
                            literalInfo.addConstantCondition(variableIndex,"=",sqlConstant);
                        }
                    }
                }
            }
            if (literal.isPositive())
                m_positiveLiteralInfos.add(literalInfo);
            else
                m_negativeLiteralInfos.add(literalInfo);
        }
    }

    /**
     * Builds the information about the probe.
     *
     * @param probeOrdering                     the list of probe variables
     * @throws DatalogException                 thrown if there is an error
     */
    protected void buildProbe(List probeOrdering) throws DatalogException {
        m_probeSQLNames=new String[probeOrdering.size()];
        m_probeTypes=new int[probeOrdering.size()];
        int index=0;
        Iterator iterator=probeOrdering.iterator();
        while (iterator.hasNext()) {
            Variable variable=(Variable)iterator.next();
            if (m_flounderingVariables.contains(variable))
                throw new DatalogException("Variable '"+variable+"' flounders and can't be returned.");
            m_probeSQLNames[index]=getSQLNameForVariable(variable);
            if (m_probeSQLNames[index]==null)
                throw new DatalogException("No SQL name for variable '"+variable+"'.");
            m_probeTypes[index]=getVariableType(variable);
            if (m_probeTypes[index]==-1)
                throw new DatalogException("No type for variable '"+variable+"'.");
            index++;
        }
    }
    /**
     * Records information about a variable at given position.
     *
     * @param variable                          the variable
     * @param literalInfo                       the information about the literal
     * @param columnIndex                       the index of the column
     */
    protected void generateVariable(Variable variable,LiteralInfo literalInfo,int columnIndex) {
        String columnName=literalInfo.getColumnName(columnIndex);
        if (!m_sqlNamesByVariables.containsKey(variable)) {
            m_sqlNamesByVariables.put(variable,columnName);
            int variableType=literalInfo.getColumnType(columnIndex);
            m_typesByVariables.put(variable,new Integer(variableType));
        }
    }
    /**
     * Checks whether given variable is among the bound variables and its SQL name hasn't been generated. If so, then the external SQL name is generated.
     *
     * @param literal                           the literal
     * @param literalInfo                       the information about the literal
     * @param variableIndex                     the index of the variable
     * @return                                  <code>true</code> if the variable was passed to this literal
     * @throws DatalogException                 thrown if there is an error
     */
    protected boolean generateSQLNameForVariable(Literal literal,LiteralInfo literalInfo,int variableIndex) throws DatalogException {
        Variable variable=literal.getArgumentVariable(variableIndex);
        int variableType=literalInfo.getColumnType(variableIndex);
        String sqlName=(String)m_sqlNamesByVariables.get(variable);
        if (sqlName==null) {
            boolean result;
            if (m_join.contains(variable)) {
                sqlName="bindings."+m_extensionalDatabase.getDatabaseManager().getTypeManager().typeToString(variableType)+m_buildJoinOrdering.indexOf(variable);
                result=false;
            }
            else {
                sqlName=literalInfo.getColumnName(variableIndex);
                result=true;
                if (!literal.isPositive() || variableType==TypeManager.TYPE_REGULAR_EXPRESSION || variableType==TypeManager.TYPE_LIKE)
                    m_flounderingVariables.add(variable);
            }
            m_sqlNamesByVariables.put(variable,sqlName);
            m_typesByVariables.put(variable,new Integer(variableType));
            return result;
        }
        else {
            if (m_flounderingVariables.contains(variable))
                throw new DatalogException("Variable '"+variable.getVariableName()+"' flounders and can't be bound.");
            if (getVariableType(variable)!=variableType)
                throw new DatalogException("Variable '"+variable+"' fails the type check.");
            return false;
        }
    }
    /**
     * For given variable returns the SQL name that matches to the entity URI with value
     * equal to the variable.
     *
     * @param variable                          the variable
     * @return                                  the SQL name that matches to the entity URI with value equal to the variable
     */
    protected String getSQLNameForVariable(Variable variable) {
        return (String)m_sqlNamesByVariables.get(variable);
    }
    /**
     * Returns the type of given variable.
     *
     * @param variable                          the variable
     * @return                                  the type of given variable
     */
    protected int getVariableType(Variable variable) {
        Integer result=(Integer)m_typesByVariables.get(variable);
        if (result==null)
            return -1;
        else
            return result.intValue();
    }
    /**
     * Returns the name of the table for bindings with given arity.
     *
     * @return                                  the name of the of the table for bindings
     */
    protected String getBindingsTableName() {
        return m_extensionalDatabase.getDatabaseManager().getTemporaryTableName(m_buildJoinTypes);
    }
    /**
     * Generates the query.
     *
     * @param buffer                            the buffer receiving the query
     */
    protected abstract void generateQuery(StringBuffer buffer);
    /**
     * Creates the query operator.
     *
     * @param bindingsOperator                  the operator for bindings
     * @return                                  the query operator
     * @throws DatalogException                 thrown if there is an error
     */
    public QueryOperator getQueryOperator(QueryOperator bindingsOperator) throws DatalogException {
        QueryOperator resultOperator;
        StringBuffer buffer=new StringBuffer();
        generateQuery(buffer);
        if (m_buildJoinIndices.length+m_buildNonJoinIndices.length==0)
            resultOperator=new NoBoundValuesExpressionOperator(m_extensionalDatabase,buffer.toString(),m_probeTypes);
        else
            resultOperator=new BoundValuesExpressionOperator(m_extensionalDatabase,bindingsOperator,buffer.toString(),m_probeTypes,m_buildJoinTypes,m_buildJoinIndices,m_buildNonJoinIndices,m_buildIndicesToCopy,m_probeJoinIndices,m_probeNonJoinIndices,m_probeIndicesToCopy);
        if (!m_regularExpressionVariables.isEmpty()) {
            String[] regularExpressions=new String[m_regularExpressions.size()];
            m_regularExpressions.toArray(regularExpressions);
            int[] regularExpressionFlags=new int[regularExpressions.length];
            for (int i=0;i<regularExpressionFlags.length;i++)
                regularExpressionFlags[i]=Pattern.CASE_INSENSITIVE;
            resultOperator=new RegularExpressionFilter(resultOperator,regularExpressions,regularExpressionFlags,m_regularExpressionsFilterOnIndices);
            resultOperator=new ShortenTuple(resultOperator,m_regularExpressionVariables.size());
        }
        return resultOperator;
    }

    /**
     * The information about literals.
     */
    public static class LiteralInfo {
        /** The information about the predicate in the DB. */
        protected JDBCExtensionalDatabase.PredicateDBInfo m_predicateDBInfo;
        /** The name of the alias for the table. */
        protected String m_aliasName;
        /** The join conditions. */
        protected List m_joinConditions;
        /** The constant conditions. */
        protected List m_constantConditions;

        public LiteralInfo(JDBCExtensionalDatabase.PredicateDBInfo predicateDBInfo,String aliasName) {
            m_predicateDBInfo=predicateDBInfo;
            m_aliasName=aliasName;
        }
        public String getAliasName() {
            return m_aliasName;
        }
        public String getTableName() {
            return m_predicateDBInfo.getTableName();
        }
        public int getColumnType(int columnIndex) {
            return m_predicateDBInfo.getColumnType(columnIndex);
        }
        public String getColumnName(int columnIndex) {
            return m_aliasName+"."+m_predicateDBInfo.getColumnName(columnIndex);
        }
        public boolean hasJoinConditions() {
            return m_joinConditions!=null;
        }
        public void addJoinCondition(int literalColumnIndex,String operator,String compareWith) {
            if (m_joinConditions==null)
                m_joinConditions=new LinkedList();
            m_joinConditions.add(new Object[] { new Integer(literalColumnIndex),operator,compareWith });
        }
        public Iterator getJoinConditions() {
            if (m_joinConditions==null)
                return Collections.EMPTY_LIST.iterator();
            else
                return m_joinConditions.iterator();
        }
        public boolean hasConstantConditions() {
            return m_constantConditions!=null;
        }
        public void addConstantCondition(int literalColumnIndex,String operator,Object constant) {
            if (m_constantConditions==null)
                m_constantConditions=new LinkedList();
            m_constantConditions.add(getColumnName(literalColumnIndex)+operator+constant);
        }
        public void addConstantCondition(String literalColumnName,String operator,Object constant) {
            if (m_constantConditions==null)
                m_constantConditions=new LinkedList();
            m_constantConditions.add(m_aliasName+"."+literalColumnName+operator+constant);
        }
        public void addConstantCondition(String condition) {
            if (m_constantConditions==null)
                m_constantConditions=new LinkedList();
            m_constantConditions.add(condition);
        }
        public Iterator getConstantConditions() {
            if (m_constantConditions==null)
                return Collections.EMPTY_LIST.iterator();
            else
                return m_constantConditions.iterator();
        }
    }


    /**
     * Query operator for removing regular expression columns from tuple ends.
     */
    protected static class ShortenTuple implements QueryOperator {
        /** The source operator. */
        protected QueryOperator m_sourceOperator;
        /** The current tuple. */
        protected Object[] m_currentTuple;
        /** The number of columns to remove from the end of each tuple. */
        protected int m_columnsToRemove;

        /**
         * Creates an instance of this class.
         *
         * @param sourceOperator                        the source operator
         * @param columnsToRemove                       the number of columns to remove from the end of each tuple
         */
        public ShortenTuple(QueryOperator sourceOperator,int columnsToRemove) {
            m_sourceOperator=sourceOperator;
            m_columnsToRemove=columnsToRemove;
        }
        /**
         * Starts the processing of the query. The cursor is positioned on the first tuple, or at end.
         *
         * @throws DatalogException                     thrown if there is an error in evaluation
         */
        public void start() throws DatalogException {
            m_sourceOperator.start();
            next();
        }
        /**
         * Stops the processing of the query.
         *
         * @throws DatalogException                     thrown if there is an error in evaluation
         */
        public void stop() throws DatalogException {
            try {
                m_sourceOperator.stop();
            }
            finally {
                m_currentTuple=null;
            }
        }
        /**
         * Returns the current tuple of the operator. If the tuple stream is at the end, <code>null</code> is returned.
         *
         * @return                                      the current tuple (or <code>null</code> if the tuple stream is at the end)
         */
        public Object[] tuple() {
            return m_currentTuple;
        }
        /**
         * Returns <code>true</code> if the current stream is after the last tuple.
         *
         * @return                                      <code>true</code> if the stream is after the last tuple
         */
        public boolean afterLast() {
            return m_currentTuple==null;
        }
        /**
         * Moves the cursor to the next position. If the tuple stream is at the end this method call has no effect.
         *
         * @throws DatalogException                     thrown if there is an error in evaluation
         */
        public void next() throws DatalogException {
            if (m_sourceOperator.afterLast())
                m_currentTuple=null;
            else {
                Object[] tuple=m_sourceOperator.tuple();
                m_currentTuple=new Object[tuple.length-m_columnsToRemove];
                System.arraycopy(tuple,0,m_currentTuple,0,m_currentTuple.length);
                m_sourceOperator.next();
            }
        }
    }
}
