package edu.unika.aifb.kaon.datalog.evaluation;

import edu.unika.aifb.kaon.datalog.*;
import edu.unika.aifb.kaon.datalog.program.*;

/**
 * The operator implementing the unification of the input stream with some unary function calls.
 */
public class UnificationOperator implements QueryOperator {
    /** The special constant specifying that unification can't be performed. */
    protected static final Object UNIFICATION_FAILED=new Object();

    /** The query operator producing the tuples. */
    protected QueryOperator m_sourceOperator;
    /** The pairs of indices to copy for each source. */
    protected int[][] m_indicesToCopy;
    /** Function calls which are unified. */
    protected UnaryFunctionCall[] m_unaryFunctionCalls;
    /** The indices of arguments for which the unification is performed. */
    protected int[] m_indicesOfArguments;
    /** The to <code>true</code> if the function calls should be instantiated. */
    protected boolean[] m_instantiateCalls;
    /** The indices where the resulting values should be placed. */
    protected int[] m_indicesOfResults;
    /** The arity of the resulting tuple. */
    protected int m_resultingTupleArity;
    /** The current tuple. */
    protected Object[] m_currentTuple;

    /**
     * Creates an instance of this class.
     *
     * @param sourceOperator                    the source operator
     * @param indicesToCopy                     the indices of the source stream to copy
     * @param unaryFunctionCalls                the function calls to be unified
     * @param indicesOfArguments                the indices of arguments to function calls
     * @param instantiateCalls                  determines whether instantiation or inverse instantiation is performed
     * @param indicesOfResults                  the indices where the unification result should be placed
     */
    public UnificationOperator(QueryOperator sourceOperator,int[][] indicesToCopy,UnaryFunctionCall[] unaryFunctionCalls,int[] indicesOfArguments,boolean[] instantiateCalls,int[] indicesOfResults) {
        m_sourceOperator=sourceOperator;
        m_indicesToCopy=indicesToCopy;
        m_unaryFunctionCalls=unaryFunctionCalls;
        m_indicesOfArguments=indicesOfArguments;
        m_instantiateCalls=instantiateCalls;
        m_indicesOfResults=indicesOfResults;
        m_resultingTupleArity=m_indicesToCopy.length;
        for (int i=0;i<m_indicesOfResults.length;i++)
            m_resultingTupleArity++;
    }
    /**
     * Starts the processing of the query. The cursor is positioned on the first tuple, or at end.
     *
     * @throws DatalogException                 thrown if there is an error in evaluation
     */
    public void start() throws DatalogException {
        m_sourceOperator.start();
        next();
    }
    /**
     * Stops the processing of the query.
     *
     * @throws DatalogException                 thrown if there is an error in evaluation
     */
    public void stop() throws DatalogException {
        m_sourceOperator.stop();
        m_currentTuple=null;
    }
    /**
     * Returns the current tuple of the operator. If the tuple stream is at the end, <code>null</code> is returned.
     *
     * @return                                  the current tuple (or <code>null</code> if the tuple stream is at the end)
     */
    public Object[] tuple() {
        return m_currentTuple;
    }
    /**
     * Returns <code>true</code> if the current stream is after the last tuple.
     *
     * @return                                  <code>true</code> if the stream is after the last tuple
     */
    public boolean afterLast() {
        return m_currentTuple==null;
    }
    /**
     * Moves the cursor to the next position. If the tuple stream is at the end this method call has no effect.
     *
     * @throws DatalogException                 thrown if there is an error in evaluation
     */
    public void next() throws DatalogException {
        nextTuple: while (true) {
            if (!m_sourceOperator.afterLast()) {
                Object[] tuple=m_sourceOperator.tuple();
                m_sourceOperator.next();
                m_currentTuple=new Object[m_resultingTupleArity];
                for (int i=m_unaryFunctionCalls.length-1;i>=0;--i) {
                    Object argument=tuple[m_indicesOfArguments[i]];
                    Object value;
                    if (m_instantiateCalls[i])
                        value=instantiateFunctionCall(m_unaryFunctionCalls[i],argument);
                    else
                        value=undoFunctionCall(m_unaryFunctionCalls[i],argument);
                    if (value==UNIFICATION_FAILED)
                        continue nextTuple;
                    if (m_indicesOfResults[i]!=-1)
                        m_currentTuple[m_indicesOfResults[i]]=value;
                }
                for (int i=m_indicesToCopy.length-1;i>=0;--i)
                    m_currentTuple[m_indicesToCopy[i][1]]=tuple[m_indicesToCopy[i][0]];
            }
            else
                m_currentTuple=null;
            break nextTuple;
        }
    }
    /**
     * Instantiates a function call for given value.
     *
     * @param unaryFunctionCall                 the function call to be instantiated
     * @param value                             the value for which the call is instantiated
     * @return                                  the instantiation of the function call
     */
    protected Object instantiateFunctionCall(UnaryFunctionCall unaryFunctionCall,Object value) {
        Term argument=unaryFunctionCall.getArgument();
        if (argument instanceof Variable)
            return new UnaryFunctionCallValue(unaryFunctionCall.getFunctionName(),value);
        else if (argument instanceof Constant)
            return new UnaryFunctionCallValue(unaryFunctionCall.getFunctionName(),((Constant)argument).getValue());
        else if (argument instanceof UnaryFunctionCall)
            return new UnaryFunctionCallValue(unaryFunctionCall.getFunctionName(),instantiateFunctionCall((UnaryFunctionCall)argument,value));
        else
            throw new IllegalStateException("Internal error - argument is of unexpected type.");
    }
    /**
     * Undoes a function call if possible.
     *
     * @param unaryFunctionCall                 the function call to be undone
     * @param value                             the value for which the call is undone
     * @return                                  the obtained value
     */
    protected Object undoFunctionCall(UnaryFunctionCall unaryFunctionCall,Object value) {
        Term term=unaryFunctionCall;
        while (term instanceof UnaryFunctionCall) {
            if (!(value instanceof UnaryFunctionCallValue))
                return UNIFICATION_FAILED;
            UnaryFunctionCallValue callValue=(UnaryFunctionCallValue)value;
            UnaryFunctionCall call=(UnaryFunctionCall)term;
            if (!call.getFunctionName().equals(callValue.getFunctionName()))
                return UNIFICATION_FAILED;
            value=callValue.getArgument();
            term=call.getArgument();
        }
        if (term instanceof Variable)
            return value;
        else if (term instanceof Constant)
            return ((Constant)term).getValue();
        else
            throw new IllegalStateException("Internal error - argument is of unexpected type.");
    }
}
