package edu.unika.aifb.kaon.datalog.evaluation;

import java.util.Iterator;
import java.util.NoSuchElementException;

/**
 * A hash set of tuples.
 */
public class TupleHashSet extends AbstractTupleCollection {
    /** The array of entries. */
    protected Entry[] m_entries;
    /** The load factor. */
    protected float m_loadFactor;
    /** The size of the map. */
    protected int m_size;
    /** The threshold for the resizing. */
    protected int m_threshold;

    /**
     * Creates an instance of this class.
     *
     * @param keyEqualityIndices        the indices determining the key equality
     */
    public TupleHashSet(int[] keyEqualityIndices) {
        super(keyEqualityIndices);
        m_entries=new Entry[16];
        m_loadFactor=0.75f;
        m_threshold=(int)(m_entries.length*m_loadFactor);
    }
    /**
     * Clears this set.
     */
    public void clear() {
        for (int i=m_entries.length-1;i>=0;--i)
            m_entries[i]=null;
        m_size=0;
    }
    /**
     * Adds a value into the hash-table.
     *
     * @param tuple                     the tuple
     * @return                          <code>true</code> if the set was changed
     */
    public boolean add(Object[] tuple) {
        int hashCode=tupleKeyHashCode(tuple,m_keyEqualityIndices);
        int index=getIndex(hashCode);
        for (Entry entry=m_entries[index];entry!=null;entry=entry.m_next)
            if (hashCode==entry.m_hashCode && tuplesKeyEqual(tuple,m_keyEqualityIndices,entry.m_tuple,m_keyEqualityIndices))
                return false;
        m_entries[index]=new Entry(tuple,hashCode,m_entries[index]);
        if (m_size++>=m_threshold)
            resize(2*m_entries.length);
        return true;
    }
    /**
     * Returns the existing tuple from the set that is equal to the passed tuple.
     *
     * @param tuple                     the tuple which should be returned from the set
     * @return                          existing tuple that is equal to the passed tuple or <code>null</code> if the passed tuple is not in the extension
     */
    public Object[] getExistingTuple(Object[] tuple) {
        return getExistingTuple(tuple,m_keyEqualityIndices);
    }
    /**
     * Returns the existing tuple from the set that is equal to the passed tuple.
     *
     * @param tuple                     the tuple which should be returned from the set
     * @param equalityIndices           the indices determining the equality
     * @return                          existing tuple that is equal to the passed tuple or <code>null</code> if the passed tuple is not in the extension
     */
    public Object[] getExistingTuple(Object[] tuple,int[] equalityIndices) {
        int hashCode=tupleKeyHashCode(tuple,equalityIndices);
        int index=getIndex(hashCode);
        for (Entry entry=m_entries[index];entry!=null;entry=entry.m_next)
            if (hashCode==entry.m_hashCode && tuplesKeyEqual(tuple,equalityIndices,entry.m_tuple,m_keyEqualityIndices))
                return entry.m_tuple;
        return null;
    }
    /**
     * Checks whether the element is already in the set.
     *
     * @param tuple                     the tuple
     * @return                          <code>true</code> if the set contains the tuple
     */
    public boolean contains(Object[] tuple) {
        return contains(tuple,m_keyEqualityIndices);
    }
    /**
     * Checks whether the element is already in the set.
     *
     * @param tuple                     the tuple
     * @param equalityIndices           the indices determining the equality
     * @return                          <code>true</code> if the set contains the tuple
     */
    public boolean contains(Object[] tuple,int[] equalityIndices) {
        return getExistingTuple(tuple,equalityIndices)!=null;
    }
    /**
     * Removes a value from the hash-table.
     *
     * @param tuple                     the tuple to be removed
     * @return                          the old tuple or <code>null</code> if there is no such thing
     */
    public Object[] remove(Object[] tuple) {
        return remove(tuple,m_keyEqualityIndices);
    }
    /**
     * Removes a value from the hash-table.
     *
     * @param tuple                     the tuple to be removed
     * @param equalityIndices           the indices determining the equality
     * @return                          the old tuple or <code>null</code> if there is no such thing
     */
    public Object[] remove(Object[] tuple,int[] equalityIndices) {
        int hashCode=tupleKeyHashCode(tuple,equalityIndices);
        int index=getIndex(hashCode);
        Entry previous=null;
        Entry entry=m_entries[index];
        while (entry!=null) {
            if (hashCode==entry.m_hashCode && tuplesKeyEqual(tuple,equalityIndices,entry.m_tuple,m_keyEqualityIndices)) {
                if (previous==null)
                    m_entries[index]=entry.m_next;
                else
                    previous.m_next=entry.m_next;
                return entry.m_tuple;
            }
            previous=entry;
            entry=entry.m_next;
        }
        return null;
    }
    /**
     * Returns the index for given hash-code.
     *
     * @param hashCode                  the hash-code
     * @return                          the index for given hash-code
     */
    protected int getIndex(int hashCode) {
        return (hashCode-(hashCode<<7)) & (m_entries.length-1);
    }
    /**
     * Resizes the contents of the hash-table.
     *
     * @param newCapacity               the new capacity of the table (must be a power of two)
     */
    public void resize(int newCapacity) {
        Entry[] oldEntries=m_entries;
        m_entries=new Entry[newCapacity];
        for (int i=0;i<oldEntries.length;i++) {
            Entry entry=oldEntries[i];
            while (entry!=null) {
                Entry next=entry.m_next;
                int index=getIndex(entry.m_hashCode);
                entry.m_next=m_entries[index];
                m_entries[index]=entry;
                entry=next;
            }
        }
        m_threshold=(int)(m_entries.length*m_loadFactor);
    }
    /**
     * Returns the iterator of the tuples in this set.
     *
     * @return                          the iterator of the tuples in the set
     */
    public Iterator iterator() {
        return new TupleIterator();
    }
    /**
     * Returns the number of tuples in the set.
     *
     * @return                          the number of tuples in the set
     */
    public int size() {
        return m_size;
    }

    /**
     * The entry in the map.
     */
    protected static class Entry {
        /** The tuple. */
        public Object[] m_tuple;
        /** The hash-code. */
        public int m_hashCode;
        /** The next entry. */
        public Entry m_next;

        public Entry(Object[] tuple,int hashCode,Entry next) {
            m_tuple=tuple;
            m_hashCode=hashCode;
            m_next=next;
        }
    }

    /**
     * The iterator of tuples.
     */
    protected class TupleIterator implements Iterator {
        protected int m_index;
        protected Entry m_currentEntry;

        public TupleIterator() {
            while (m_currentEntry==null && m_index<m_entries.length)
                m_currentEntry=m_entries[m_index++];
        }
        public boolean hasNext() {
            return m_currentEntry!=null;
        }
        public Object next() {
            if (m_currentEntry==null)
                throw new NoSuchElementException();
            Object tuple=m_currentEntry.m_tuple;
            m_currentEntry=m_currentEntry.m_next;
            while (m_currentEntry==null && m_index<m_entries.length)
                m_currentEntry=m_entries[m_index++];
            return tuple;
        }
        public void remove()  {
            throw new UnsupportedOperationException();
        }
    }
}
