package edu.unika.aifb.kaon.datalog.evaluation;

import java.util.Iterator;
import java.util.NoSuchElementException;

/**
 * A hash map using tuples as keys.
 */
public class TupleHashMap extends AbstractTupleCollection {
    /** The array of entries. */
    protected Entry[] m_entries;
    /** The load factor. */
    protected float m_loadFactor;
    /** The size of the map. */
    protected int m_size;
    /** The threshold for the resizing. */
    protected int m_threshold;

    /**
     * Creates an instance of this class.
     *
     * @param keyEqualityIndices        the indices determining the key equality
     */
    public TupleHashMap(int[] keyEqualityIndices) {
        super(keyEqualityIndices);
        m_entries=new Entry[16];
        m_loadFactor=0.75f;
        m_threshold=(int)(m_entries.length*m_loadFactor);
    }
    /**
     * Clears this set.
     */
    public void clear() {
        for (int i=m_entries.length-1;i>=0;--i)
            m_entries[i]=null;
        m_size=0;
    }
    /**
     * Puts a value into the hash-table.
     *
     * @param key                       the key tuple
     * @param value                     the value tuple
     * @return                          existing value
     */
    public Object put(Object[] key,Object value) {
        int hashCode=tupleKeyHashCode(key,m_keyEqualityIndices);
        int index=getIndex(hashCode);
        for (Entry entry=m_entries[index];entry!=null;entry=entry.m_next) {
            if (hashCode==entry.m_hashCode && tuplesKeyEqual(key,m_keyEqualityIndices,entry.m_key,m_keyEqualityIndices)) {
                Object oldValue=entry.m_value;
                entry.m_value=value;
                return oldValue;
            }
        }
        m_entries[index]=new Entry(key,value,hashCode,m_entries[index]);
        if (m_size++>=m_threshold)
            resize(2*m_entries.length);
        return null;
    }
    /**
     * Returns a value with given key from the map.
     *
     * @param key                       the key tuple
     * @return                          the value under given key (or <code>null</code> if the value hasn't been set for given key)
     */
    public Object get(Object[] key) {
        return get(key,m_keyEqualityIndices);
    }
    /**
     * Returns a value with given key from the map.
     *
     * @param key                       the key tuple
     * @param equalityIndices           the indices determining the equality
     * @return                          the value under given key (or <code>null</code> if the value hasn't been set for given key)
     */
    public Object get(Object[] key,int[] equalityIndices) {
        int hashCode=tupleKeyHashCode(key,equalityIndices);
        int index=getIndex(hashCode);
        for (Entry entry=m_entries[index];entry!=null;entry=entry.m_next)
            if (hashCode==entry.m_hashCode && tuplesKeyEqual(key,equalityIndices,entry.m_key,m_keyEqualityIndices))
                return entry.m_value;
        return null;
    }
    /**
     * Removes a value with given key from the map.
     *
     * @param key                       the key tuple
     * @return                          the value under given key (or <code>null</code> if the value hasn't been set for given key)
     */
    public Object remove(Object[] key) {
        return remove(key,m_keyEqualityIndices);
    }
    /**
     * Removes a value with given key from the map.
     *
     * @param key                       the key tuple
     * @param equalityIndices           the indices determining the equality
     * @return                          the value under given key (or <code>null</code> if the value hasn't been set for given key)
     */
    public Object remove(Object[] key,int[] equalityIndices) {
        int hashCode=tupleKeyHashCode(key,equalityIndices);
        int index=getIndex(hashCode);
        Entry previous=null;
        Entry entry=m_entries[index];
        while (entry!=null) {
            if (hashCode==entry.m_hashCode && tuplesKeyEqual(key,equalityIndices,entry.m_key,m_keyEqualityIndices)) {
                if (previous==null)
                    m_entries[index]=entry.m_next;
                else
                    previous.m_next=entry.m_next;
                return entry.m_value;
            }
            previous=entry;
            entry=entry.m_next;
        }
        return null;
    }
    /**
     * Returns the index for given hash-code.
     *
     * @param hashCode                  the hash-code
     * @return                          the index for given hash-code
     */
    protected int getIndex(int hashCode) {
        return (hashCode-(hashCode<<7)) & (m_entries.length-1);
    }
    /**
     * Resizes the contents of the hash-table.
     *
     * @param newCapacity               the new capacity of the table (must be a power of two)
     */
    public void resize(int newCapacity) {
        Entry[] oldEntries=m_entries;
        m_entries=new Entry[newCapacity];
        for (int i=0;i<oldEntries.length;i++) {
            Entry entry=oldEntries[i];
            while (entry!=null) {
                Entry next=entry.m_next;
                int index=getIndex(entry.m_hashCode);
                entry.m_next=m_entries[index];
                m_entries[index]=entry;
                entry=next;
            }
        }
        m_threshold=(int)(m_entries.length*m_loadFactor);
    }
    /**
     * Returns the iterator of the tuple keys in this set.
     *
     * @return                          the iterator of the tuple keys in the set
     */
    public Iterator keyIterator() {
        return new KeyTupleIterator();
    }
    /**
     * Returns the number of tuples in the map.
     *
     * @return                          the number of tuples in the map
     */
    public int size() {
        return m_size;
    }

    /**
     * The entry in the map.
     */
    protected static class Entry {
        /** The key tuple. */
        public Object[] m_key;
        /** The value in the entry. */
        public Object m_value;
        /** The hash-code. */
        public int m_hashCode;
        /** The next entry. */
        public Entry m_next;

        public Entry(Object[] key,Object value,int hashCode,Entry next) {
            m_key=key;
            m_value=value;
            m_hashCode=hashCode;
            m_next=next;
        }
    }

    /**
     * The iterator of keys.
     */
    protected class KeyTupleIterator implements Iterator {
        protected int m_index;
        protected Entry m_currentEntry;

        public KeyTupleIterator() {
            while (m_currentEntry==null)
                m_currentEntry=m_entries[m_index++];
        }
        public boolean hasNext() {
            return m_currentEntry!=null;
        }
        public Object next() {
            if (m_currentEntry==null)
                throw new NoSuchElementException();
            Object key=m_currentEntry.m_key;
            m_currentEntry=m_currentEntry.m_next;
            while (m_currentEntry==null && m_index<m_entries.length)
                m_currentEntry=m_entries[m_index++];
            return key;
        }
        public void remove()  {
            throw new UnsupportedOperationException();
        }
    }
}
