package edu.unika.aifb.kaon.datalog.evaluation;

import java.util.List;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Comparator;
import java.util.Collections;

import edu.unika.aifb.kaon.datalog.*;

/**
 * An operator that provides sorting of the tuple stream on given tuple indices.
 */
public class SortOperator implements QueryOperator,Comparator {
    /** The source operator. */
    protected QueryOperator m_sourceOperator;
    /** The indices of the tuple to apply the sort on.*/
    protected int[] m_sortOn;
    /** The comparators for the values. */
    protected Comparator[] m_comparators;
    /** The iterator of sorted tuples. */
    protected Iterator m_sortedTuples;
    /** The current tuple. */
    protected Object[] m_currentTuple;

    /**
     * Creates an instance of this class.
     *
     * @param sourceOperator                        the source operator
     * @param sortOn                                the indices on which the tuples are sorted
     * @param comparators                           the comparators for the tuple positions
     */
    public SortOperator(QueryOperator sourceOperator,int[] sortOn,Comparator[] comparators) {
        m_sourceOperator=sourceOperator;
        m_sortOn=sortOn;
        m_comparators=comparators;
    }
    /**
     * Starts the processing of the query. The cursor is positioned on the first tuple, or at end.
     *
     * @throws DatalogException                     thrown if there is an error in evaluation
     */
    public void start() throws DatalogException {
        List tuples=new ArrayList();
        m_sourceOperator.start();
        try {
            while (!m_sourceOperator.afterLast()) {
                Object[] tuple=m_sourceOperator.tuple();
                tuples.add(tuple);
                m_sourceOperator.next();
            }
        }
        finally {
            m_sourceOperator.stop();
        }
        Collections.sort(tuples,this);
        m_sortedTuples=tuples.iterator();
        next();
    }
    /**
     * Stops the processing of the query.
     */
    public void stop() {
        m_currentTuple=null;
        m_sortedTuples=null;
    }
    /**
     * Returns the current tuple of the operator. If the tuple stream is at the end, <code>null</code> is returned.
     *
     * @return                                      the current tuple (or <code>null</code> if the tuple stream is at the end)
     */
    public Object[] tuple() {
        return m_currentTuple;
    }
    /**
     * Returns <code>true</code> if the current stream is after the last tuple.
     *
     * @return                                      <code>true</code> if the stream is after the last tuple
     */
    public boolean afterLast() {
        return m_currentTuple==null;
    }
    /**
     * Moves the cursor to the next position. If the tuple stream is at the end this method call has no effect.
     *
     */
    public void next() {
        if (m_sortedTuples.hasNext())
            m_currentTuple=(Object[])m_sortedTuples.next();
        else
            m_currentTuple=null;
    }
    /**
     * This is an implementation side-effect, sonce this class implements <code>Comparator</code>.
     *
     * @param o1                                    the first object to compare
     * @param o2                                    the second object to compare
     * @return                                      the compare value for the objects
     */
    public int compare(Object o1,Object o2) {
        Object[] t1=(Object[])o1;
        Object[] t2=(Object[])o2;
        for (int i=0;i<m_sortOn.length;i++) {
            int index=m_sortOn[i];
            int result=m_comparators[i].compare(t1[index],t2[index]);
            if (result!=0)
                return result;
        }
        return 0;
    }

    public static class ComparableComparator implements Comparator {
        public static final Comparator INSTANCE=new ComparableComparator();

        public int compare(Object o1,Object o2) {
            try {
                if (!(o1 instanceof Comparable) || !(o2 instanceof Comparable))
                    return 0;
                Comparable c1=(Comparable)o1;
                Comparable c2=(Comparable)o2;
                if (c1==c2)
                    return 0;
                else if (c1==null)
                    return -1;
                else
                    return c1.compareTo(c2);
            }
            catch (ClassCastException error) {
                return 0;
            }
        }
    }
}
