package edu.unika.aifb.kaon.datalog.evaluation;

import edu.unika.aifb.kaon.datalog.*;
import edu.unika.aifb.kaon.datalog.program.*;

/**
 * This evaluator evaluates a rule and stores the extension of the rule into appropriate memory extension.
 */
public class RuleEvaluator implements Evaluator {
    /** The operator of the rule. */
    protected QueryOperator m_ruleOperator;
    /** The consumer for the tuple. */
    protected TupleConsumer m_tupleConsumer;
    protected Rule m_rule;

    /**
     * Creates an instance of this class.
     *
     * @param rule                                  the rule to be evaluated
     * @param extensionalManager                    the manager for database extensions
     * @throws DatalogException                     thrown if there is an error
     */
    public RuleEvaluator(Rule rule,ExtensionalManager extensionalManager) throws DatalogException {
        this(rule,extensionalManager,new RuleCompiler(extensionalManager,rule));
    }
    /**
     * Creates an instance of this class.
     *
     * @param rule                                  the rule to be evaluated
     * @param extensionalManager                    the manager for database extensions
     * @param ruleCompiler                          the compiler to use
     * @throws DatalogException                     thrown if there is an error
     */
    public RuleEvaluator(Rule rule,ExtensionalManager extensionalManager,RuleCompiler ruleCompiler) throws DatalogException {
        m_rule=rule;
        ruleCompiler.compile();
        m_ruleOperator=ruleCompiler.getQueryOperator();
        if (ruleCompiler.getDisjunctionInfosPassed() || rule.getHeadLength()!=1 || extensionalManager.getExtensionalDatabaseEx(rule.getHeadLiteral(0).getPredicate()).getContainsDisjunctionInfo(rule.getHeadLiteral(0).getPredicate()))
            m_tupleConsumer=new DisjunctiveConsumer(extensionalManager,ruleCompiler.getHeadPredicates(),ruleCompiler.getTupleRenames(),ruleCompiler.getDisjunctionInfoPositions());
        else
            m_tupleConsumer=new NonDisjunctiveConsumer(extensionalManager,ruleCompiler.getHeadPredicate(0),ruleCompiler.getTupleRename(0));
    }
    /**
     * Executes this evaluator.
     *
     * @throws DatalogException                     thrown if there is an error in evaluation
     */
    public void execute() throws DatalogException {
        m_ruleOperator.start();
        try {
            while (!m_ruleOperator.afterLast()) {
                m_tupleConsumer.consumeTuple(m_ruleOperator.tuple());
                m_ruleOperator.next();
            }
        }
        finally {
            m_ruleOperator.stop();
        }
    }

    /**
     * An object that can consume the tuple of the rule.
     */
    protected static interface TupleConsumer {
        void consumeTuple(Object[] tuple) throws DatalogException;
    }

    /**
     * Consumer for non-disjunctive rules.
     */
    protected static class NonDisjunctiveConsumer implements TupleConsumer {
        /** The predicate extension receiving the tuples. */
        protected MemoryPredicateExtension m_target;
        /** The renamer for the tuple. */
        protected TupleRename m_tupleRename;

        public NonDisjunctiveConsumer(ExtensionalManager targetManager,Predicate predicate,TupleRename tupleRename) throws DatalogException {
            ExtensionalDatabase database=targetManager.getExtensionalDatabase(predicate);
            if (!(database instanceof MemoryExtensionalDatabase))
                throw new DatalogException("The target database must be MemoryExtensionalDatabase.");
            m_target=((MemoryExtensionalDatabase)database).getMemoryPredicateExtension(predicate);
            m_tupleRename=tupleRename;
        }
        public void consumeTuple(Object[] tuple) {
            Object[] transformedTuple;
            if (m_target.getContainsDisjunctionInfo())
                transformedTuple=m_tupleRename.renameTupleAddDisjunctionInfo(tuple);
            else
                transformedTuple=m_tupleRename.renameTuple(tuple);
            m_target.addTuple(transformedTuple);
        }
    }

    /**
     * Consumer for the disjunctive rules.
     */
    protected static class DisjunctiveConsumer implements TupleConsumer {
        /** The extensional manager receiving the results. */
        protected ExtensionalManager m_targetManager;
        /** The predicates of the head. */
        protected Predicate[] m_predicates;
        /** Renamers for generating tuples. */
        protected TupleRename[] m_tupleRenames;
        /** The indices of the disjunctions. */
        protected int[] m_disjunctionIndices;
        /** The disjunction infos utility array. */
        protected DisjunctionRestInfo[] m_disjuntionRestInfos;
        /** The dummy disjunction info node. */
        protected DisjunctionRestInfo.DisjunctInfo m_dummyNode;

        public DisjunctiveConsumer(ExtensionalManager targetManager,Predicate[] predicates,TupleRename[] tupleRenames,int[] disjunctionIndices) {
            m_targetManager=targetManager;
            m_predicates=predicates;
            m_tupleRenames=tupleRenames;
            m_disjunctionIndices=disjunctionIndices;
            m_disjuntionRestInfos=new DisjunctionRestInfo[m_disjunctionIndices.length+1];
            m_dummyNode=new DisjunctionRestInfo.DisjunctInfo(null,null,null);
        }
        public void consumeTuple(Object[] tuple) throws DatalogException {
            m_dummyNode.m_next=null;
            nextTuple: for (int i=0;i<m_tupleRenames.length;i++) {
                Predicate newPredicate=m_predicates[i];
                Object[] newTuple=m_tupleRenames[i].renameTuple(tuple);
                DisjunctionRestInfo.DisjunctInfo last=m_dummyNode;
                DisjunctionRestInfo.DisjunctInfo current=m_dummyNode.m_next;
                while (current!=null) {
                    if (m_targetManager.getGroundLiteralOrdering().compareTo(current.m_predicate,current.m_tuple,newPredicate,newTuple)>0) {
                        last.m_next=new DisjunctionRestInfo.DisjunctInfo(newPredicate,newTuple,last.m_next);
                        continue nextTuple;
                    }
                    last=current;
                    current=current.m_next;
                }
                last.m_next=new DisjunctionRestInfo.DisjunctInfo(newPredicate,newTuple,null);
            }
            DisjunctionRestInfo info=new DisjunctionRestInfo(null,m_dummyNode.m_next);
            m_dummyNode.m_next=null;
            consumeTupleAt(tuple,0,info);
            for (int i=0;i<m_disjuntionRestInfos.length;i++)
                m_disjuntionRestInfos[i]=null;
        }
        protected void consumeTupleAt(Object[] tuple,int index,DisjunctionRestInfo info) throws DatalogException {
            m_disjuntionRestInfos[index]=info;
            while (m_disjuntionRestInfos[index]!=null) {
                if (index+1==m_disjuntionRestInfos.length)
                    m_targetManager.addDisjunctionByMerging(m_disjuntionRestInfos);
                else
                    consumeTupleAt(tuple,index+1,(DisjunctionRestInfo)tuple[m_disjunctionIndices[index]]);
                m_disjuntionRestInfos[index]=m_disjuntionRestInfos[index].m_next;
            }
        }
    }
}
