package edu.unika.aifb.kaon.datalog.evaluation;

import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

import edu.unika.aifb.kaon.datalog.*;

/**
 * An operator that filters the results of some operator against a regular expression.
 */
public class RegularExpressionFilter implements QueryOperator {
    /** The source operator. */
    protected QueryOperator m_sourceOperator;
    /** The current tuple. */
    protected Object[] m_currentTuple;
    /** The patterns. */
    protected Pattern[] m_patterns;
    /** The indices being filtered on. */
    protected int[] m_filterOn;

    /**
     * Creates an instance of this class.
     *
     * @param sourceOperator                        the source operator
     * @param regularExpressions                    the regular expressions performing the filtering
     * @param regularExpressionFlags                the flags for the regular expressions
     * @param filterOn                              the indices being sorted on
     * @throws DatalogException                     thrown if there is an error
     */
    public RegularExpressionFilter(QueryOperator sourceOperator,String[] regularExpressions,int[] regularExpressionFlags,int[] filterOn) throws DatalogException {
        m_sourceOperator=sourceOperator;
        m_filterOn=filterOn;
        m_patterns=new Pattern[regularExpressions.length];
        try {
            for (int i=0;i<regularExpressions.length;i++)
                m_patterns[i]=Pattern.compile(regularExpressions[i],regularExpressionFlags[i]);
        }
        catch (PatternSyntaxException error) {
            throw new DatalogException("Invalid pattern.",error);
        }
    }
    /**
     * Starts the processing of the query. The cursor is positioned on the first tuple, or at end.
     *
     * @throws DatalogException                     thrown if there is an error in evaluation
     */
    public void start() throws DatalogException {
        m_sourceOperator.start();
        next();
    }
    /**
     * Stops the processing of the query.
     *
     * @throws DatalogException                     thrown if there is an error in evaluation
     */
    public void stop() throws DatalogException {
        try {
            m_sourceOperator.stop();
        }
        finally {
            m_currentTuple=null;
        }
    }
    /**
     * Returns the current tuple of the operator. If the tuple stream is at the end, <code>null</code> is returned.
     *
     * @return                                      the current tuple (or <code>null</code> if the tuple stream is at the end)
     */
    public Object[] tuple() {
        return m_currentTuple;
    }
    /**
     * Returns <code>true</code> if the current stream is after the last tuple.
     *
     * @return                                      <code>true</code> if the stream is after the last tuple
     */
    public boolean afterLast() {
        return m_currentTuple==null;
    }
    /**
     * Moves the cursor to the next position. If the tuple stream is at the end this method call has no effect.
     *
     * @throws DatalogException                     thrown if there is an error in evaluation
     */
    public void next() throws DatalogException {
        while (!m_sourceOperator.afterLast()) {
            m_currentTuple=m_sourceOperator.tuple();
            boolean matches=true;
            for (int i=0;matches && i<m_patterns.length;i++) {
                String value=(String)m_currentTuple[m_filterOn[i]];
                if (value==null || !m_patterns[i].matcher(value).matches())
                    matches=false;
            }
            m_sourceOperator.next();
            if (matches)
                return;
        }
        m_currentTuple=null;
    }
}
