package edu.unika.aifb.kaon.datalog.evaluation;

import java.util.Set;
import java.util.HashSet;
import java.util.Iterator;

import edu.unika.aifb.kaon.datalog.*;
import edu.unika.aifb.kaon.datalog.program.*;

/**
 * This class evaluates a block of mutually recursive rules.
 */
public class RecursionBlockEvaluator implements Evaluator {
    /** The number of different head predicatas evaluated in this block. */
    protected int m_numberOfComputedPredicates;
    /** The predicate extensions for the rules. */
    protected MemoryPredicateExtension[] m_headExtensions;
    /** The predicate extensions for the deltas. */
    protected MemoryPredicateExtension[] m_deltaHeadExtensions;
    /** The predicate extensions for the new deltas. */
    protected MemoryPredicateExtension[] m_newDeltaHeadExtensions;
    /** The evaluators for the rules in this block. */
    protected Evaluator[][] m_ruleEvaluators;

    /**
     * Creates an instance of this class.
     *
     * @param rules                                 the set of rules to be evaluated in the block
     * @param precedenceGraph                       the precedence graph
     * @param extensionalManager                    the manager for database extensions
     * @throws DatalogException                     thrown if there is an error
     */
    public RecursionBlockEvaluator(Set rules,PrecedenceGraph precedenceGraph,ExtensionalManager extensionalManager) throws DatalogException {
        MemoryExtensionalDatabase deltaHeadDatabase=new MemoryExtensionalDatabase();
        MemoryExtensionalDatabase newDeltaHeadDatabase=new MemoryExtensionalDatabase();
        ExtensionalManager newDeltaHeadManager=new ExtensionalManager(extensionalManager.getGroundLiteralOrdering());
        newDeltaHeadManager.registerExtensionalDatabase("newDeltaHeadDatabase",newDeltaHeadDatabase);
        Set computedPredicates=new HashSet();
        Iterator iterator=rules.iterator();
        while (iterator.hasNext()) {
            Rule rule=(Rule)iterator.next();
            computedPredicates.addAll(precedenceGraph.getPredicatesGeneratedByRule(rule));
        }
        m_numberOfComputedPredicates=computedPredicates.size();
        m_headExtensions=new MemoryPredicateExtension[m_numberOfComputedPredicates];
        m_deltaHeadExtensions=new MemoryPredicateExtension[m_numberOfComputedPredicates];
        m_newDeltaHeadExtensions=new MemoryPredicateExtension[m_numberOfComputedPredicates];
        int index=0;
        iterator=computedPredicates.iterator();
        while (iterator.hasNext()) {
            Predicate headPredicate=(Predicate)iterator.next();
            m_headExtensions[index]=extensionalManager.getMemoryPredicateExtension(headPredicate);
            if (m_headExtensions[index]==null)
                throw new DatalogException("Cannot find memory predicate extension for predicate '"+headPredicate.getFullName()+"'.");
            boolean containsDisjunctionInfo=m_headExtensions[index].getContainsDisjunctionInfo();
            m_deltaHeadExtensions[index]=deltaHeadDatabase.createPredicateExtension(headPredicate);
            m_deltaHeadExtensions[index].setContainsDisjunctionInfo(containsDisjunctionInfo);
            m_deltaHeadExtensions[index].indexAll();
            m_newDeltaHeadExtensions[index]=newDeltaHeadDatabase.createPredicateExtension(headPredicate);
            m_newDeltaHeadExtensions[index].setContainsDisjunctionInfo(containsDisjunctionInfo);
            index++;
        }
        m_ruleEvaluators=new Evaluator[rules.size()][];
        iterator=rules.iterator();
        index=0;
        while (iterator.hasNext()) {
            Rule rule=(Rule)iterator.next();
            m_ruleEvaluators[index]=createEvaluatorsForRule(rule,extensionalManager,computedPredicates,deltaHeadDatabase,newDeltaHeadManager);
            index++;
        }
    }
    /**
     * For given rule generates the evaluators that evaluate the rule by freezing each of the
     * IDB predicates in the body, successively.
     *
     * @param rule                                  the rule being compiled
     * @param extensionalManager                    the manager for database extensions
     * @param computedPredicates                    the set of computed predicates
     * @param deltaHeadDatabase                     the memory extensional database that holds the extensions of the deltas
     * @param newDeltaHeadManager                   the manager for the extensional database that holds the new extensions of the deltas
     * @return                                      the evaluators for the rule
     * @throws DatalogException                     thrown if there is an error
     */
    protected Evaluator[] createEvaluatorsForRule(Rule rule,ExtensionalManager extensionalManager,Set computedPredicates,final MemoryExtensionalDatabase deltaHeadDatabase,ExtensionalManager newDeltaHeadManager) throws DatalogException {
        int numberOfIDBPredicates=0;
        for (int i=0;i<rule.getBodyLength();i++)
            if (computedPredicates.contains(rule.getBodyLiteral(i).getPredicate()))
                numberOfIDBPredicates++;
        Evaluator[] evaluators=new Evaluator[numberOfIDBPredicates];
        int index=0;
        for (int i=0;i<rule.getBodyLength();i++) {
            Predicate predicate=rule.getBodyLiteral(i).getPredicate();
            if (computedPredicates.contains(predicate)) {
                final int bodyIndex=i;
                RuleCompiler compiler=new RuleCompiler(extensionalManager,rule) {
                    protected ExtensionalDatabase getExtensionalDatabase(int literalIndex,Predicate predicate) throws DatalogException {
                        if (literalIndex==bodyIndex)
                            return deltaHeadDatabase;
                        else
                            return super.getExtensionalDatabase(literalIndex,predicate);
                    }
                };
                evaluators[index++]=new RuleEvaluator(rule,newDeltaHeadManager,compiler);
            }
        }
        return evaluators;
    }
    /**
     * Executes this evaluator.
     *
     * @throws DatalogException                     thrown if there is an error in evaluation
     */
    public void execute() throws DatalogException {
        for (int predicateIndex=0;predicateIndex<m_numberOfComputedPredicates;predicateIndex++)
            m_deltaHeadExtensions[predicateIndex].replaceWith(m_headExtensions[predicateIndex]);
        boolean hasChange=true;
        while (hasChange) {
            for (int predicateIndex=0;predicateIndex<m_numberOfComputedPredicates;predicateIndex++)
                m_newDeltaHeadExtensions[predicateIndex].clear();
            for (int ruleIndex=0;ruleIndex<m_ruleEvaluators.length;ruleIndex++) {
                Evaluator[] evaluators=m_ruleEvaluators[ruleIndex];
                for (int evaluatorIndex=0;evaluatorIndex<evaluators.length;evaluatorIndex++)
                    evaluators[evaluatorIndex].execute();
            }
            hasChange=false;
            for (int predicateIndex=0;predicateIndex<m_numberOfComputedPredicates;predicateIndex++) {
                MemoryPredicateExtension deltaRuleExtension=m_deltaHeadExtensions[predicateIndex];
                MemoryPredicateExtension newDeltaRuleExtension=m_newDeltaHeadExtensions[predicateIndex];
                if (mergeExtensions(m_headExtensions[predicateIndex],newDeltaRuleExtension,deltaRuleExtension))
                    hasChange=true;
            }
        }
        for (int predicateIndex=0;predicateIndex<m_numberOfComputedPredicates;predicateIndex++) {
            m_deltaHeadExtensions[predicateIndex].clear();
            m_newDeltaHeadExtensions[predicateIndex].clear();
        }
    }
    /**
     * Copies the extension of the new delta to the actual extension. Those tuples that don't exist
     * in the actual extension will be copied into the old delta extension.
     *
     * @param actualExtension                       the actual predicate extension
     * @param newDeltaExtension                     the extension of the new delta to be copied
     * @param deltaExtension                        the extension of the old delta receiving the tuples
     * @return                                      <code>true</code> if the actual extension changed
     */
    protected boolean mergeExtensions(MemoryPredicateExtension actualExtension,MemoryPredicateExtension newDeltaExtension,MemoryPredicateExtension deltaExtension) {
        deltaExtension.clear();
        boolean changed=false;
        Iterator tuples=newDeltaExtension.selectTuples(new Object[0],new int[0]);
        while (tuples.hasNext()) {
            Object[] tuple=(Object[])tuples.next();
            if (actualExtension.addTuple(tuple)) {
                changed=true;
                Object[] deltaTuple=(Object[])tuple.clone();
                if (actualExtension.getContainsDisjunctionInfo()) {
                    DisjunctionRestInfo copyFrom=(DisjunctionRestInfo)tuple[tuple.length-1];
                    DisjunctionRestInfo copyTo=null;
                    while (copyFrom!=null) {
                        DisjunctionRestInfo newInfo=new DisjunctionRestInfo(null,copyFrom);
                        if (copyTo==null)
                            deltaTuple[deltaTuple.length-1]=newInfo;
                        else
                            copyTo.m_next=newInfo;
                        copyTo=newInfo;
                        copyFrom=copyFrom.m_next;
                    }
                }
                deltaExtension.addTuple(deltaTuple);
            }
            else {
                if (actualExtension.getContainsDisjunctionInfo()) {
                    Object[] deltaTuple=null;
                    Object[] existingTuple=actualExtension.getExistingTuple(tuple);
                    DisjunctionRestInfo existingDisjunctionInfo=(DisjunctionRestInfo)existingTuple[existingTuple.length-1];
                    DisjunctionRestInfo existingDisjunctionInfoStart=existingDisjunctionInfo;
                    DisjunctionRestInfo sourceDisjunctionInfo=(DisjunctionRestInfo)tuple[tuple.length-1];
                    nextSourceInfo: while (sourceDisjunctionInfo!=null) {
                        DisjunctionRestInfo pointer=existingDisjunctionInfoStart;
                        while (pointer!=null) {
                            if (sourceDisjunctionInfo.isEqual(pointer)) {
                                sourceDisjunctionInfo=sourceDisjunctionInfo.m_next;
                                continue nextSourceInfo;
                            }
                            pointer=pointer.m_next;
                        }
                        existingDisjunctionInfo=new DisjunctionRestInfo(existingDisjunctionInfo,sourceDisjunctionInfo);
                        if (deltaTuple==null) {
                            deltaTuple=new Object[tuple.length];
                            for (int i=0;i<tuple.length-1;i++)
                                deltaTuple[i]=tuple[i];
                            deltaExtension.addTuple(deltaTuple);
                        }
                        deltaTuple[deltaTuple.length-1]=new DisjunctionRestInfo((DisjunctionRestInfo)deltaTuple[deltaTuple.length-1],sourceDisjunctionInfo);
                        sourceDisjunctionInfo=sourceDisjunctionInfo.m_next;
                        changed=true;
                    }
                    existingTuple[existingTuple.length-1]=existingDisjunctionInfo;
                }
            }
        }
        return changed;
    }
}
