package edu.unika.aifb.kaon.datalog.evaluation;

import java.util.Set;
import java.util.Map;
import java.util.List;
import java.util.ArrayList;

import edu.unika.aifb.kaon.datalog.*;
import edu.unika.aifb.kaon.datalog.program.*;

/**
 * Compiles a program. The compiler is intialized with a set of extensional databases. The rules of the program are split into strongly
 * connected components which are then evaluated as blocks.
 */
public class ProgramCompiler {
    /** The extensional manager. */
    protected ExtensionalManager m_extensionalManager;
    /** The program being compiled. */
    protected Program m_program;
    /** The map of sets of predicates that can occur in disjunctions with the key predicate as the active literal. */
    protected Map m_possibleDisjunctionRestsByPredicates;

    /**
     * Initializes this class.
     *
     * @param extensionalManager                        the extensional manager
     * @param program                                   the program being compiled
     * @param possibleDisjunctionRestsByPredicates      the map of sets of predicates that can occur in disjunctions with the key predicate as the active literal
     */
    public ProgramCompiler(ExtensionalManager extensionalManager,Program program,Map possibleDisjunctionRestsByPredicates) {
        m_extensionalManager=extensionalManager;
        m_program=program;
        m_possibleDisjunctionRestsByPredicates=possibleDisjunctionRestsByPredicates;
    }
    /**
     * Initializes this class.
     *
     * @param extensionalManager                        the extensional manager
     * @param program                                   the program being compiled
     * @param possibleDisjunctionRestsByPredicates      the map of sets of predicates that can occur in disjunctions with the key predicate as the active literal
     * @param queryPredicate                            the query predicate that is placed last in the literal ordering
     */
    public ProgramCompiler(ExtensionalManager extensionalManager,Program program,Map possibleDisjunctionRestsByPredicates,Predicate queryPredicate) {
        this(new ExtensionalManager(extensionalManager.getExtensionalDatabases(),extensionalManager.getGroundLiteralOrdering().getOrderingForQueryPredicate(queryPredicate)),program,possibleDisjunctionRestsByPredicates);
    }
    /**
     * Obtains the evaluator that evaluates the program.
     *
     * @return                                          the evaluator that evaluates the program
     * @throws DatalogException                         thrown if there is an error
     */
    public Evaluator getEvaluator() throws DatalogException {
        PrecedenceGraph precedenceGraph=new PrecedenceGraph(m_program,m_possibleDisjunctionRestsByPredicates,m_extensionalManager.getGroundLiteralOrdering());
        int numberOfMutuallyRecursiveRulesSets=precedenceGraph.getNumberOfSetsOfMutuallyRecursiveRulesSets();
        if (numberOfMutuallyRecursiveRulesSets==0)
            return new SequenceEvaluator(new Evaluator[0]);
        else {
            List evaluators=new ArrayList();
            for (int i=0;i<numberOfMutuallyRecursiveRulesSets;i++) {
                Set mutuallyRecursiveRulesSet=precedenceGraph.getMutuallyRecursiveRulesSet(i);
                Evaluator evaluator;
                if (mutuallyRecursiveRulesSet.size()==1 && !precedenceGraph.getMutuallyRecursiveRulesSetHasCycle(i)) {
                    Rule rule=(Rule)mutuallyRecursiveRulesSet.iterator().next();
                    evaluator=new RuleEvaluator(rule,m_extensionalManager);
                }
                else
                    evaluator=new RecursionBlockEvaluator(mutuallyRecursiveRulesSet,precedenceGraph,m_extensionalManager);
                evaluators.add(evaluator);
            }
            if (evaluators.size()==1)
                return (Evaluator)evaluators.get(0);
            else {
                Evaluator[] evaluatorsArray=new Evaluator[evaluators.size()];
                evaluators.toArray(evaluatorsArray);
                return new SequenceEvaluator(evaluatorsArray);
            }
        }
    }
}
