package edu.unika.aifb.kaon.datalog.evaluation;

import java.util.Iterator;

import edu.unika.aifb.kaon.datalog.*;

/**
 * Implements a nested loop hash-join where each inner loop is obtained by looking up elements in
 * predicate extension.
 */
public class PredicateExtensionNestedLoopJoin extends AbstractNestedLoopJoin {
    /** The predicate extension. */
    protected PredicateExtension m_predicateExtension;
    /** The array of selection values. */
    protected Object[] m_selectionValues;
    /** The indices of the selection positions. */
    protected int[] m_selectionPositions;

    /**
     * Creates an instance of this class.
     *
     * @param buildOperator                     the operator generating the build tuple stream
     * @param predicateExtension                the predicate extension
     * @param buildJoinIndices                  the indices from the build stream that are part of the join
     * @param buildNonJoinIndices               the indices from the build stream that are not part of the join, but that with the join indices determine the uniquess of the tuple
     * @param probeJoinIndices                  the indices from the probe stream that are part of the join
     * @param probeNonJoinIndices               the indices from the probe stream that are not part of the join, but that with the join indices determine the uniquess of the tuple
     * @param buildIndicesToCopy                the indices to copy from the build stream to the result
     * @param probeIndicesToCopy                the indices to copy from the probe stream to the result
     * @param residualConditions                the residual join conditions
     * @param additionalBoundValues             the array specifying additional bound values
     * @param additionalBoundPositions          the array specifying additional bound positions
     */
    public PredicateExtensionNestedLoopJoin(QueryOperator buildOperator,PredicateExtension predicateExtension,int[] buildJoinIndices,int[] buildNonJoinIndices,int[] probeJoinIndices,int[] probeNonJoinIndices,int[][] buildIndicesToCopy,int[][] probeIndicesToCopy,JoinTupleFilter residualConditions,Object[] additionalBoundValues,int[] additionalBoundPositions) {
        super(buildOperator,buildJoinIndices,buildNonJoinIndices,probeJoinIndices,probeNonJoinIndices,buildIndicesToCopy,probeIndicesToCopy,residualConditions);
        m_predicateExtension=predicateExtension;
        m_selectionValues=new Object[m_probeJoinIndices.length+additionalBoundPositions.length];
        m_selectionPositions=new int[m_probeJoinIndices.length+additionalBoundPositions.length];
        for (int i=0;i<m_probeJoinIndices.length;i++)
            m_selectionPositions[i]=m_probeJoinIndices[i];
        for (int i=0;i<additionalBoundValues.length;i++) {
            m_selectionValues[i+m_probeJoinIndices.length]=additionalBoundValues[i];
            m_selectionPositions[i+m_probeJoinIndices.length]=additionalBoundPositions[i];
        }
    }
    /**
     * Returns an nested loop iterator for the given outer tuple. This method should always return
     * a valid iterator (if the nested loop shouldn't be executed, an empty iterator should be returned).
     *
     * @param buildTuple                        the build tuple for which the loop is started
     * @return                                  the nested loop iterator for given build tuple
     * @throws DatalogException                 thrown if there is an error in evaluation
     */
    protected Iterator getNestedLoopIterator(Object[] buildTuple) throws DatalogException {
        for (int i=m_buildJoinIndices.length-1;i>=0;--i)
            m_selectionValues[i]=buildTuple[m_buildJoinIndices[i]];
        return m_predicateExtension.selectTuples(m_selectionValues,m_selectionPositions);
    }
}
