package edu.unika.aifb.kaon.datalog.evaluation;

import java.util.Iterator;

import edu.unika.aifb.kaon.datalog.*;

/**
 * Looks up specified value in the sequential predicate.
 */
public class PredicateExtensionLookup implements QueryOperator {
    /** The predicate extension being scanned. */
    protected PredicateExtension m_predicateExtension;
    /** The pairs of indices to copy for each tuple. */
    protected int[][] m_indicesToCopy;
    /** The iterator of tuples. */
    protected Iterator m_iterator;
    /** The current tuple. */
    protected Object[] m_currentTuple;
    /** The filter for the tuples. */
    protected TupleFilter m_tupleFilter;
    /** The values to look up. */
    protected Object[] m_values;
    /** The indices where the lookup is performed. */
    protected int[] m_positions;

    /**
     * Creates an instance of this class.
     *
     * @param predicateExtension                the extension of the predicate
     * @param tupleFilter                       the filter to apply to each tuple
     * @param indicesToCopy                     the indices to copy
     * @param values                            the values to look up
     * @param positions                         the positions to look up
     */
    public PredicateExtensionLookup(PredicateExtension predicateExtension,TupleFilter tupleFilter,int[][] indicesToCopy,Object[] values,int[] positions) {
        m_predicateExtension=predicateExtension;
        m_tupleFilter=tupleFilter;
        m_indicesToCopy=indicesToCopy;
        m_values=values;
        m_positions=positions;
    }
    /**
     * Starts the processing of the query. The cursor is positioned on the first tuple, or at end.
     *
     * @throws DatalogException                 thrown if there is an error in evaluation
     */
    public void start() throws DatalogException {
        m_iterator=m_predicateExtension.selectTuples(m_values,m_positions);
        next();
    }
    /**
     * Stops the processing of the query.
     */
    public void stop() {
        m_iterator=null;
        m_currentTuple=null;
    }
    /**
     * Returns the current tuple of the operator. If the tuple stream is at the end, <code>null</code> is returned.
     *
     * @return                                  the current tuple (or <code>null</code> if the tuple stream is at the end)
     */
    public Object[] tuple() {
        return m_currentTuple;
    }
    /**
     * Returns <code>true</code> if the current stream is after the last tuple.
     *
     * @return                                  <code>true</code> if the stream is after the last tuple
     */
    public boolean afterLast() {
        return m_currentTuple==null;
    }
    /**
     * Moves the cursor to the next position. If the tuple stream is at the end this method call has no effect.
     */
    public void next() {
        while (m_iterator.hasNext()) {
            Object[] tuple=(Object[])m_iterator.next();
            if (m_tupleFilter.evaluate(tuple)) {
                m_currentTuple=new Object[m_indicesToCopy.length];
                for (int i=m_indicesToCopy.length-1;i>=0;--i)
                    m_currentTuple[m_indicesToCopy[i][1]]=tuple[m_indicesToCopy[i][0]];
                return;
            }
        }
        m_currentTuple=null;
    }
}
