package edu.unika.aifb.kaon.datalog.evaluation;

import edu.unika.aifb.kaon.datalog.*;
import edu.unika.aifb.kaon.datalog.program.*;

/**
 * Implements the extensional database that works by using predicate extensions.
 */
public abstract class PredicateExtensionExtensionalDatabase implements ExtensionalDatabase {
    /** The constant specifying an empty variable array. */
    protected static final Variable[] NO_VARIABLES=new Variable[0];

    /**
     * Creates an instance of this class.
     */
    public PredicateExtensionExtensionalDatabase() {
    }
    /**
     * Returns <code>true</code> if supplied predicate can be evaluated in this database.
     *
     * @param predicate                     the predicate
     * @return                              <code>true</code> if supplied predicate can be evaluated in this database
     */
    public boolean canEvaluatePredicate(Predicate predicate) {
        return getPredicateExtension(predicate)!=null;
    }
    /**
     * Returns <code>true</code> if supplied predicate in this database contains disjunction information.
     *
     * @param predicate                         the predicate
     * @return                                  <code>true</code> if supplied predicate in this database contains disjunction information
     * @throws DatalogException                 thrown if there is an error (e.g. if the predicate can't be evaluated by this database)
     */
    public boolean getContainsDisjunctionInfo(Predicate predicate) throws DatalogException {
        PredicateExtension predicateExtension=getPredicateExtension(predicate);
        if (predicateExtension==null)
            throw new DatalogException("Supplied predicate can't be evaluated by this database.");
        return predicateExtension.getContainsDisjunctionInfo();
    }
    /**
     * Creates an operator that evaluates specified literals using the values from the operator passed in.
     *
     * @param literals                          the literals to be evaluated
     * @param bindingsOperator                  the operator producing the bindings
     * @param boundVariables                    the variables of the bindings
     * @param returnedVariables                 the variables that the operator should return
     * @return                                  the accessor for given literals
     * @throws DatalogException                 thrown if there is an error
     */
    public QueryOperator createQueryOperator(Literal[] literals,QueryOperator bindingsOperator,Variable[] boundVariables,Variable[] returnedVariables) throws DatalogException {
        ExpressionCompiler compiler=new ExpressionCompiler(literals,boundVariables,returnedVariables);
        QueryOperator currentOperator=bindingsOperator;
        while (!compiler.expressionCompiled()) {
            compiler.compileNextLiteral();
            PredicateExtension predicateExtension=getPredicateExtension(compiler.getCurrentPredicate());
            if (predicateExtension==null)
                throw new DatalogException("Database doens't contain the extension of the predicate '"+compiler.getCurrentPredicate().getFullName()+"'.");
            Object[] boundConstantValues=compiler.getBoundConstantValues();
            int[] boundConstantPositions=compiler.getBoundConstantPositions();
            if (compiler.hasPassedValues()) {
                int[] buildJoinIndices=compiler.getPassedValuesJoinIndices();
                int[] buildNonJoinIndices=compiler.getPassedValuesNonJoinIndices();
                int[] probeJoinIndices=compiler.getCurrentLiteralJoinIndices();
                int[] probeNonJoinIndices=compiler.getCurrentLiteralNonJoinIndices();
                int[][] buildIndicesToCopy=compiler.getPassedValuesIndicesToCopy();
                int[][] probeIndicesToCopy=compiler.getCurrentLiteralIndicesToCopy();
                if (compiler.getCurrentLiteral().isPositive())
                    currentOperator=new PredicateExtensionNestedLoopJoin(currentOperator,predicateExtension,buildJoinIndices,buildNonJoinIndices,probeJoinIndices,probeNonJoinIndices,buildIndicesToCopy,probeIndicesToCopy,compiler.getJoinTupleFilter(),boundConstantValues,boundConstantPositions);
                else
                    currentOperator=new NegatedPredicateExtensionNestedLoopJoin(currentOperator,predicateExtension,buildJoinIndices,buildNonJoinIndices,probeJoinIndices,probeNonJoinIndices,buildIndicesToCopy,probeIndicesToCopy,compiler.getJoinTupleFilter(),boundConstantValues,boundConstantPositions);
            }
            else {
                int[][] indicesToCopy=compiler.getCurrentLiteralIndicesToCopy();
                currentOperator=new PredicateExtensionLookup(predicateExtension,compiler.getTupleFilter(),indicesToCopy,boundConstantValues,boundConstantPositions);
            }
        }
        return currentOperator;
    }
    /**
     * Creates an operator that evaluatesspecified literals with no variables passed in.
     *
     * @param literals                          the literals to be evaluated
     * @param returnedVariables                 the variables that the operator should return
     * @return                                  the query operator evaluating specified literals
     * @throws DatalogException                 thrown if there is an error
     */
    public QueryOperator createQueryOperator(Literal[] literals,Variable[] returnedVariables) throws DatalogException {
        return createQueryOperator(literals,null,NO_VARIABLES,returnedVariables);
    }
    /**
     * Returns the extension for given predicate.
     *
     * @param predicate                     the predicate
     * @return                              the extension for the given predicate
     */
    public abstract PredicateExtension getPredicateExtension(Predicate predicate);
}
