package edu.unika.aifb.kaon.datalog.evaluation;

import java.util.Iterator;
import java.util.NoSuchElementException;

/**
 * An index for objects.
 */
public class ObjectIndex {
    /** The array of entries. */
    protected Entry[] m_entries;
    /** The load factor. */
    protected float m_loadFactor;
    /** The size of the map. */
    protected int m_size;
    /** The threshold for the resizing. */
    protected int m_threshold;

    /**
     * Creates an instance of this class.
     */
    public ObjectIndex() {
        m_entries=new Entry[16];
        m_loadFactor=0.75f;
        m_threshold=(int)(m_entries.length*m_loadFactor);
    }
    /**
     * Clears the index.
     */
    public void clear() {
        for (int i=m_entries.length-1;i>=0;--i)
            m_entries[i]=null;
        m_size=0;
    }
    /**
     * Looks up the iterator by given value.
     *
     * @param key                       the key by which the lookup is performed
     * @return                          the iterator of values for given key
     */
    public Iterator lookup(Object key) {
        int hashCode=key==null ? 0 : key.hashCode();
        int index=getIndex(hashCode);
        for (Entry entry=m_entries[index];entry!=null;entry=entry.m_next)
            if (hashCode==entry.m_hashCode && (key==entry.m_key || key!=null && key.equals(entry.m_key))) {
                Object value=entry.m_value;
                if (value instanceof ListNode)
                    return new ListIterator((ListNode)value);
                else
                    return new SingletonIterator(value);
            }
        return EmptyIterator.INSTANCE;
    }
    /**
     * Adds an element into the index.
     *
     * @param key                       the key of the element
     * @param value                     the new value of the element
     */
    public void addElement(Object key,Object value) {
        int hashCode=key==null ? 0 : key.hashCode();
        int index=getIndex(hashCode);
        for (Entry entry=m_entries[index];entry!=null;entry=entry.m_next) {
            if (hashCode==entry.m_hashCode && (key==entry.m_key || key!=null && key.equals(entry.m_key))) {
                Object existingValue=entry.m_value;
                if (!(existingValue instanceof ListNode))
                    existingValue=new ListNode(existingValue,null);
                entry.m_value=new ListNode(value,(ListNode)existingValue);
                return;
            }
        }
        m_entries[index]=new Entry(key,value,hashCode,m_entries[index]);
        if (m_size++>=m_threshold)
            resize(2*m_entries.length);
    }
    /**
     * Removes an element from the index.
     *
     * @param key                       the key of the element
     * @param value                     the value of the element
     * @return                          <code>true<code> if something was changed
     */
    public boolean removeElement(Object key,Object value) {
        int hashCode=key==null ? 0 : key.hashCode();
        int index=getIndex(hashCode);
        Entry previous=null;
        Entry entry=m_entries[index];
        while (entry!=null) {
            if (hashCode==entry.m_hashCode && (key==entry.m_key || key!=null && key.equals(entry.m_key))) {
                Object existingValue=entry.m_value;
                if (existingValue instanceof ListNode) {
                    ListNode node=(ListNode)existingValue;
                    ListNode previousNode=null;
                    while (node!=null) {
                        if (node.m_value.equals(value)) {
                            if (previousNode==null)
                                entry.m_value=node.m_next;
                            else
                                previousNode.m_next=node.m_next;
                            ListNode firstNode=(ListNode)entry.m_value;
                            if (firstNode.m_next==null)
                                entry.m_value=firstNode.m_value;
                            return true;
                        }
                        previousNode=node;
                        node=node.m_next;
                    }
                }
                else if (existingValue.equals(value)) {
                    if (previous==null)
                        m_entries[index]=entry.m_next;
                    else
                        previous.m_next=entry.m_next;
                    return true;
                }
                return false;
            }
            previous=entry;
            entry=entry.m_next;
        }
        return false;
    }
    /**
     * Returns the index for given hash-code.
     *
     * @param hashCode                  the hash-code
     * @return                          the index for given hash-code
     */
    protected int getIndex(int hashCode) {
        return (hashCode-(hashCode<<7)) & (m_entries.length-1);
    }
    /**
     * Resizes the contents of the hash-table.
     *
     * @param newCapacity               the new capacity of the table (must be a power of two)
     */
    public void resize(int newCapacity) {
        Entry[] oldEntries=m_entries;
        m_entries=new Entry[newCapacity];
        for (int i=0;i<oldEntries.length;i++) {
            Entry entry=oldEntries[i];
            while (entry!=null) {
                Entry next=entry.m_next;
                int index=getIndex(entry.m_hashCode);
                entry.m_next=m_entries[index];
                m_entries[index]=entry;
                entry=next;
            }
        }
        m_threshold=(int)(m_entries.length*m_loadFactor);
    }

    /**
     * The entry in the map.
     */
    protected static class Entry {
        /** The key object. */
        public Object m_key;
        /** The value in the entry. */
        public Object m_value;
        /** The hash-code. */
        public int m_hashCode;
        /** The next entry. */
        public Entry m_next;

        public Entry(Object key,Object value,int hashCode,Entry next) {
            m_key=key;
            m_value=value;
            m_hashCode=hashCode;
            m_next=next;
        }
    }

    /**
     * The node for the list of equal tuples.
     */
    public static class ListNode {
        /** The value in the node. */
        public Object m_value;
        /** The next node. */
        public ListNode m_next;

        public ListNode(Object value,ListNode next) {
            m_value=value;
            m_next=next;
        }
    }

    /**
     * The list iterator
     */
    protected static class ListIterator implements Iterator {
        protected ListNode m_current;

        public ListIterator(ListNode startNode) {
            m_current=startNode;
        }
        public boolean hasNext() {
            return m_current!=null;
        }
        public Object next() {
            if (m_current==null)
                throw new NoSuchElementException();
            Object result=m_current.m_value;
            m_current=m_current.m_next;
            return result;
        }
        public void remove() {
            throw new UnsupportedOperationException();
        }
    }

    /**
     * A singleton iterator.
     */
    protected static class SingletonIterator implements Iterator {
        protected Object m_object;

        public SingletonIterator(Object object) {
            m_object=object;
        }
        public boolean hasNext() {
            return m_object!=null;
        }
        public Object next() {
            if (m_object==null)
                throw new NoSuchElementException();
            Object result=m_object;
            m_object=null;
            return result;
        }
        public void remove() {
            throw new UnsupportedOperationException();
        }
    }

    /**
     * An empty iterator.
     */
    protected static class EmptyIterator implements Iterator {
        public static final Iterator INSTANCE=new EmptyIterator();

        public boolean hasNext() {
            return false;
        }
        public Object next() {
            throw new NoSuchElementException();
        }
        public void remove() {
            throw new UnsupportedOperationException();
        }
    }
}
