package edu.unika.aifb.kaon.datalog.evaluation;

import java.util.Iterator;
import java.util.Collections;

import edu.unika.aifb.kaon.datalog.*;

/**
 * Implements a nested loop join where the build stream is the negated predicate. This operator works
 * only if all variables in the lookup are bound by the generating tuple stream. In another words,
 * this operator cannot bind any new values.
 */
public class NegatedPredicateExtensionNestedLoopJoin extends PredicateExtensionNestedLoopJoin {
    /**
     * Creates an instance of this class.
     *
     * @param buildOperator                     the operator generating the build tuple stream
     * @param predicateExtension                the predicate extension
     * @param buildJoinIndices                  the indices from the build stream that are part of the join
     * @param buildNonJoinIndices               the indices from the build stream that are not part of the join, but that with the join indices determine the uniquess of the tuple
     * @param probeJoinIndices                  the indices from the probe stream that are part of the join
     * @param probeNonJoinIndices               the indices from the probe stream that are not part of the join, but that with the join indices determine the uniquess of the tuple
     * @param buildIndicesToCopy                the indices to copy from the build stream to the result
     * @param probeIndicesToCopy                the indices to copy from the probe stream to the result
     * @param residualConditions                the residual join conditions
     * @param additionalBoundValues             the array specifying additional bound values
     * @param additionalBoundPositions          the array specifying additional bound positions
     */
    public NegatedPredicateExtensionNestedLoopJoin(QueryOperator buildOperator,PredicateExtension predicateExtension,int[] buildJoinIndices,int[] buildNonJoinIndices,int[] probeJoinIndices,int[] probeNonJoinIndices,int[][] buildIndicesToCopy,int[][] probeIndicesToCopy,JoinTupleFilter residualConditions,Object[] additionalBoundValues,int[] additionalBoundPositions) {
        super(buildOperator,predicateExtension,buildJoinIndices,buildNonJoinIndices,probeJoinIndices,probeNonJoinIndices,buildIndicesToCopy,probeIndicesToCopy,residualConditions,additionalBoundValues,additionalBoundPositions);
    }
    /**
     * Returns an nested loop iterator for the given outer tuple. This method should always return
     * a valid iterator (if the nested loop shouldn't be executed, an empty iterator should be returned).
     *
     * @param buildTuple                        the build tuple for which the loop is started
     * @return                                  the nested loop iterator for given build tuple
     * @throws DatalogException                 thrown if there is an error in evaluation
     */
    protected Iterator getNestedLoopIterator(Object[] buildTuple) throws DatalogException {
        for (int i=m_buildJoinIndices.length-1;i>=0;--i)
            m_selectionValues[i]=buildTuple[m_buildJoinIndices[i]];
        Iterator lookup=m_predicateExtension.selectTuples(m_selectionValues,m_selectionPositions);
        if (lookup.hasNext())
            return Collections.EMPTY_SET.iterator();
        else {
            Object[] probeTuple=new Object[m_probeJoinIndices.length+m_probeNonJoinIndices.length];
            for (int i=m_probeJoinIndices.length-1;i>=0;--i)
                probeTuple[m_probeJoinIndices[i]]=buildTuple[m_buildJoinIndices[i]];
            for (int i=m_probeNonJoinIndices.length-1;i>=0;--i)
                probeTuple[m_probeNonJoinIndices[i]]=new Object();
            return Collections.singleton(probeTuple).iterator();
        }
    }
}
