package edu.unika.aifb.kaon.datalog.evaluation;

import java.util.Iterator;
import java.util.NoSuchElementException;

import edu.unika.aifb.kaon.datalog.program.*;

/**
 * Represents an extension of the predicate. If the extension contains disjunctions, then
 * each tuple contains another column with the information about the disjunction.
 */
public class MemoryPredicateExtension implements PredicateExtension {
    /** The predicate of this extension. */
    protected Predicate m_predicate;
    /** The set of tuples in the predicate. */
    protected TupleHashSet m_tuples;
    /** The indices. */
    protected ObjectIndex[] m_componentIndices;
    /** Set to <code>true</code> if this extension contains disjunction information. */
    protected boolean m_containsDisjunctionInfo;

    /**
     * Creates an instance of this class.
     *
     * @param predicate                             the predicate of the extension
     */
    public MemoryPredicateExtension(Predicate predicate) {
        m_predicate=predicate;
        m_componentIndices=new ObjectIndex[m_predicate.getArity()];
        int[] allIndices=new int[m_componentIndices.length];
        for (int i=0;i<allIndices.length;i++)
            allIndices[i]=i;
        m_tuples=new TupleHashSet(allIndices);
    }
    /**
     * Returns the predicate of this extension.
     *
     * @return                                      the predicate of this extension
     */
    public Predicate getPredicate() {
        return m_predicate;
    }
    /**
     * Returns whether this predicate extension contains disjunctions.
     *
     * @return                                      <code>true</code> if this predicate extension contains disjunctions
     */
    public boolean getContainsDisjunctionInfo() {
        return m_containsDisjunctionInfo;
    }
    /**
     * Determines whether this predicate extension contains disjunctions.
     *
     * @param containsDisjunctionInfo               <code>true</code> if this predicate extension contains disjunction info
     */
    public void setContainsDisjunctionInfo(boolean containsDisjunctionInfo) {
        m_containsDisjunctionInfo=containsDisjunctionInfo;
    }
    /**
     * Creates an index on given position.
     *
     * @param position                              the position on which the index is created
     */
    public void createIndex(int position) {
        if (m_componentIndices[position]==null) {
            ObjectIndex index=new ObjectIndex();
            m_componentIndices[position]=index;
            Iterator iterator=m_tuples.iterator();
            while (iterator.hasNext()) {
                Object[] tuple=(Object[])iterator.next();
                index.addElement(tuple[position],tuple);
            }
        }
    }
    /**
     * Index the extension on all positions.
     */
    public void indexAll() {
        for (int i=m_componentIndices.length-1;i>=0;--i)
            createIndex(i);
    }
    /**
     * Adds the tuple to the extension.
     *
     * @param tuple                                 the tuple
     * @return                                      <code>true</code> if the tuple was added
     */
    public boolean addTuple(Object[] tuple) {
        boolean result=m_tuples.add(tuple);
        if (result) {
            for (int i=m_componentIndices.length-1;i>=0;--i)
                if (m_componentIndices[i]!=null)
                    m_componentIndices[i].addElement(tuple[i],tuple);
        }
        return result;
    }
    /**
     * Removes the tuple from the extension.
     *
     * @param tuple                                 the tuple
     * @return                                      existing tuple if there was one
     */
    public Object[] removeTuple(Object[] tuple) {
        Object[] result=m_tuples.remove(tuple);
        if (result!=null) {
            for (int i=m_componentIndices.length-1;i>=0;--i)
                if (m_componentIndices[i]!=null)
                    m_componentIndices[i].removeElement(tuple[i],result);
        }
        return result;
    }
    /**
     * Empties this predicate extension.
     */
    public void clear() {
        m_tuples.clear();
        for (int i=m_componentIndices.length-1;i>=0;--i)
            if (m_componentIndices[i]!=null)
                m_componentIndices[i].clear();
    }
    /**
     * Returns <code>true</code> if this extension is empty.
     *
     * @return                                      <code>true</code> if this extension is empty
     */
    public boolean isEmpty() {
        return m_tuples.size()==0;
    }
    /**
     * Returns the number of tuples in the extension.
     *
     * @return                                      the number of tuples in the extension
     */
    public int size() {
        return m_tuples.size();
    }
    /**
     * Selects the tuples with specified value at given position.
     *
     * @param values                                the values
     * @param positions                             the positions of the value
     * @return                                      the iterator of tuples matching the condition
     */
    public Iterator selectTuples(Object[] values,int[] positions) {
        if (positions.length==0)
            return m_tuples.iterator();
        else {
            for (int i=0;i<positions.length;i++) {
                ObjectIndex objectIndex=m_componentIndices[positions[i]];
                if (objectIndex!=null) {
                    Iterator iterator=objectIndex.lookup(values[i]);
                    if (!iterator.hasNext())
                        return iterator;
                    else
                        return new TupleFilterIterator(iterator,positions,values);
                }
            }
            return new TupleFilterIterator(m_tuples.iterator(),positions,values);
        }
    }
    /**
     * Returns the existing tuple that is equal to the passed tuple.
     *
     * @param tuple                                 the tuple for which the existing tuple should be fetched
     * @return                                      the existing tuple or <code>null</code> if such tuple doesn't exist
     */
    public Object[] getExistingTuple(Object[] tuple) {
        return m_tuples.getExistingTuple(tuple);
    }
    /**
     * Clears the extension of this predicate and appends the extension of the other predicate.
     *
     * @param addFrom                               the other memory extension from which the contents is added
     */
    public void replaceWith(MemoryPredicateExtension addFrom) {
        clear();
        Iterator tuples=addFrom.m_tuples.iterator();
        while (tuples.hasNext()) {
            Object[] tuple=(Object[])tuples.next();
            addTuple(tuple);
        }
    }

    /**
     * The iterator that filters the values.
     */
    protected class TupleFilterIterator implements Iterator {
        /** The original iterator. */
        protected Iterator m_iterator;
        /** The positions which must be equal to specified values. */
        protected int[] m_positions;
        /** The values which must be equal to specified positions. */
        protected Object[] m_values;
        /** The current tuple. */
        protected Object[] m_currentTuple;

        public TupleFilterIterator(Iterator iterator,int[] positions,Object[] values) {
            m_iterator=iterator;
            m_positions=positions;
            m_values=values;
            advance();
        }
        public boolean hasNext() {
            return m_currentTuple!=null;
        }
        public Object next() {
            if (m_currentTuple==null)
                throw new NoSuchElementException();
            Object[] result=m_currentTuple;
            advance();
            return result;
        }
        public void remove() {
            throw new UnsupportedOperationException();
        }
        protected void advance() {
            nextTuple: while (m_iterator.hasNext()) {
                m_currentTuple=(Object[])m_iterator.next();
                for (int i=m_positions.length-1;i>=0;i--) {
                    Object value1=m_currentTuple[m_positions[i]];
                    Object value2=m_values[i];
                    if (value1!=value2 && (value1==null || !value1.equals(value2)))
                        continue nextTuple;
                }
                return;
            }
            m_currentTuple=null;
        }
    }
}
