package edu.unika.aifb.kaon.datalog.evaluation;

import java.util.Map;
import java.util.HashMap;
import java.util.Set;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Arrays;
import java.util.List;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.EventListener;
import javax.swing.event.EventListenerList;

import edu.unika.aifb.kaon.datalog.*;
import edu.unika.aifb.kaon.datalog.program.*;

/**
 * A manager of the extensional databases.
 */
public class ExtensionalManager {
    /** The map of all extensional databases indexed by some key. */
    protected Map m_extensionalDatabases;
    /** The ordering of the ground literals to use in the disjunctive case. */
    protected GroundLiteralOrdering m_ordering;
    /** The comparator for literal values. */
    protected Comparator m_literalValueComparator;
    /** The list of registered event listeners. */
    protected EventListenerList m_listenerList;

    /**
     * Creates an instance of this class.
     */
    public ExtensionalManager() {
        this(new HashMap(),new LexicographicalGroundLiteralOrdering(null));
    }
    /**
     * Creates an instance of this class.
     *
     * @param groundLiteralOrdering             the ordering for the ground literals to use in the disjunctive case
     */
    public ExtensionalManager(GroundLiteralOrdering groundLiteralOrdering) {
        this(new HashMap(),groundLiteralOrdering);
    }
    /**
     * Creates an instance of this class.
     *
     * @param extensionalDatabases              the map of extensional databases
     * @param groundLiteralOrdering             the ordering for the ground literals to use in the disjunctive case
     */
    public ExtensionalManager(Map extensionalDatabases,GroundLiteralOrdering groundLiteralOrdering) {
        m_extensionalDatabases=extensionalDatabases;
        m_ordering=groundLiteralOrdering;
        m_literalValueComparator=new LiteralValueComparator();
        m_listenerList=new EventListenerList();
    }
    /**
     * Returns the ordering used by this manager.
     *
     * @return                                  the ordering used by this manager
     */
    public GroundLiteralOrdering getGroundLiteralOrdering() {
        return m_ordering;
    }
    /**
     * Adds an extension change listener to this manager.
     *
     * @param listener                      the listener to add
     */
    public void addExensionChangeListener(ExensionChangeListener listener) {
        m_listenerList.add(ExensionChangeListener.class,listener);
    }
    /**
     * Removes an extension change listener from this manager.
     *
     * @param listener                      the listener to remove
     */
    public void removeExensionChangeListener(ExensionChangeListener listener) {
        m_listenerList.remove(ExensionChangeListener.class,listener);
    }
    /**
     * Fires a notification about changes.
     *
     * @param changeInfos                   the collection of <code>ChangeInfo</code> objects
     */
    protected void fireExtensionsWillChange(Collection changeInfos) {
        Object[] listeners=m_listenerList.getListenerList();
        for (int i=listeners.length-2;i>=0;i-=2)
            if (listeners[i]==ExensionChangeListener.class)
                ((ExensionChangeListener)listeners[i+1]).extensionsWillChange(this,changeInfos);
    }
    /**
     * Fires a notification about changes.
     *
     * @param changeInfos                   the collection of <code>ChangeInfo</code> objects
     */
    protected void fireExtensionsChanged(Collection changeInfos) {
        Object[] listeners=m_listenerList.getListenerList();
        for (int i=listeners.length-2;i>=0;i-=2)
            if (listeners[i]==ExensionChangeListener.class)
                ((ExensionChangeListener)listeners[i+1]).extensionsChanged(this,changeInfos);
    }
    /**
     * Returns an extensional database with given name.
     *
     * @param databaseName                  the name of the extensional database
     * @return                              the extensional database with given name (or <code>null</code> if such database doesn't exist)
     */
    public ExtensionalDatabase getExtensionalDatabase(String databaseName) {
        return (ExtensionalDatabase)m_extensionalDatabases.get(databaseName);
    }
    /**
     * Registers an extensional database with given name. If there is already a database with this name, it is replacerd with a new database.
     *
     * @param databaseName                  the name of the extensional database
     * @param extensionalDatabase           the extensional database with given name (or <code>null</code> if such database should be removed)
     */
    public void registerExtensionalDatabase(String databaseName,ExtensionalDatabase extensionalDatabase) {
        if (extensionalDatabase==null)
            m_extensionalDatabases.remove(databaseName);
        else
            m_extensionalDatabases.put(databaseName,extensionalDatabase);
    }
    /**
     * Returns the map of extensional databases
     *
     * @return                                  the map of extensional databases
     */
    public Map getExtensionalDatabases() {
        return m_extensionalDatabases;
    }
    /**
     * Returns the extensional database that can evaluate given predicate.
     *
     * @param predicate                         the predicate to be evaluated
     * @return                                  the database that can evaluate given predicate (<code>null</code> if such a database doesn't exist)
     * @throws DatalogException                 thrown if there is an error
     */
    public ExtensionalDatabase getExtensionalDatabase(Predicate predicate) throws DatalogException {
        Iterator iterator=m_extensionalDatabases.values().iterator();
        while (iterator.hasNext()) {
            ExtensionalDatabase extensionalDatabase=(ExtensionalDatabase)iterator.next();
            if (extensionalDatabase.canEvaluatePredicate(predicate))
                return extensionalDatabase;
        }
        return null;
    }
    /**
     * Returns the extensional database that can evaluate given predicate and throws an exception if the database can't be found.
     *
     * @param predicate                         the predicate to be evaluated
     * @return                                  the database that can evaluate given predicate (this is never <code>null</code>)
     * @throws DatalogException                 thrown if the database can't be found
     */
    public ExtensionalDatabase getExtensionalDatabaseEx(Predicate predicate) throws DatalogException {
        ExtensionalDatabase extensionalDatabase=getExtensionalDatabase(predicate);
        if (extensionalDatabase!=null)
            return extensionalDatabase;
        else
            throw new DatalogException("Can't locate the extensional database that can evaluate predicate '"+predicate.toString()+"'.");
    }
    /**
     * Returns the memory extension for the given predicate.
     *
     * @param predicate                         the predicate to be evaluated
     * @return                                  the database that can evaluate given predicate (<code>null</code> if such a database doesn't exist)
     * @throws DatalogException                 thrown if there is an error
     */
    public MemoryPredicateExtension getMemoryPredicateExtension(Predicate predicate) throws DatalogException {
        ExtensionalDatabase extensionalDatabase=getExtensionalDatabase(predicate);
        if (extensionalDatabase instanceof MemoryExtensionalDatabase)
            return ((MemoryExtensionalDatabase)extensionalDatabase).getMemoryPredicateExtension(predicate);
        else
            return null;
    }
    /**
     * For given program, this method creates predicates missing in the available databases. Predicates are created in a database
     * with given key.
     *
     * @param program                           the program for which predicates are created
     * @param missingPredicatesDatabaseName     the name of the extensional database receiving missing predicates
     * @throws DatalogException                 thrown if there is an error
     */
    public void createMissingPredicates(Program program,String missingPredicatesDatabaseName) throws DatalogException {
        ExtensionalDatabase missingPredicatesDatabase=getExtensionalDatabase(missingPredicatesDatabaseName);
        if (missingPredicatesDatabase==null)
            throw new DatalogException("Can't locate database '"+missingPredicatesDatabaseName+"'.");
        if (!(missingPredicatesDatabase instanceof MemoryExtensionalDatabase))
            throw new DatalogException("Database '"+missingPredicatesDatabaseName+"' is not a memory database.");
        MemoryExtensionalDatabase missingPredicatesMemoryDatabase=(MemoryExtensionalDatabase)missingPredicatesDatabase;
        Map nonDisjunctivePredicates=new HashMap();
        for (int i=0;i<program.getNumberOfRules();i++) {
            Rule rule=program.getRule(i);
            boolean isHorn=rule.isHorn();
            for (int j=0;j<rule.getHeadLength();j++) {
                Predicate predicate=rule.getHeadLiteral(j).getPredicate();
                ExtensionalDatabase extensionalDatabase=getExtensionalDatabase(predicate);
                if (extensionalDatabase!=null && !(extensionalDatabase instanceof MemoryExtensionalDatabase))
                    throw new DatalogException("IDB predicate '"+predicate.getFullName()+"' exists, but not in a MemoryExtensionalDatabase.");
                if (extensionalDatabase==null) {
                    MemoryPredicateExtension predicateExtension=missingPredicatesMemoryDatabase.createPredicateExtension(predicate);
                    nonDisjunctivePredicates.put(predicate,new HashSet());
                    predicateExtension.indexAll();
                }
                if (!isHorn && missingPredicatesMemoryDatabase.canEvaluatePredicate(predicate)) {
                    MemoryPredicateExtension predicateExtension=missingPredicatesMemoryDatabase.getMemoryPredicateExtension(predicate);
                    nonDisjunctivePredicates.remove(predicate);
                    predicateExtension.setContainsDisjunctionInfo(true);
                }
            }
            for (int j=0;j<rule.getBodyLength();j++) {
                Predicate predicate=rule.getBodyLiteral(j).getPredicate();
                if (getExtensionalDatabase(predicate)==null) {
                    MemoryPredicateExtension predicateExtension=missingPredicatesMemoryDatabase.createPredicateExtension(predicate);
                    nonDisjunctivePredicates.put(predicate,new HashSet());
                    predicateExtension.indexAll();
                }
            }
        }
        // build the dependencies among predicates
        for (int i=0;i<program.getNumberOfRules();i++) {
            Rule rule=program.getRule(i);
            for (int j=0;j<rule.getHeadLength();j++) {
                Predicate predicate=rule.getHeadLiteral(j).getPredicate();
                Set set=(Set)nonDisjunctivePredicates.get(predicate);
                if (set!=null) {
                    for (int k=0;k<rule.getBodyLength();k++)
                        set.add(rule.getBodyLiteral(k).getPredicate());
                }
            }
        }
        // now update all non-disjunctive predicates
        boolean change=true;
        while (change) {
            change=false;
            Iterator iterator=nonDisjunctivePredicates.entrySet().iterator();
            while (iterator.hasNext()) {
                Map.Entry entry=(Map.Entry)iterator.next();
                Iterator dependencies=((Set)entry.getValue()).iterator();
                while (dependencies.hasNext()) {
                    Predicate dependsOn=(Predicate)dependencies.next();
                    if (getExtensionalDatabase(dependsOn).getContainsDisjunctionInfo(dependsOn)) {
                        missingPredicatesMemoryDatabase.getMemoryPredicateExtension((Predicate)entry.getKey()).setContainsDisjunctionInfo(true);
                        iterator.remove();
                        change=true;
                        break;
                    }
                }
            }
        }
    }
    /**
     * Utility method that adds a unit disjunction to the database.
     *
     * @param predicate                         the predicate
     * @param tuple                             the tuple
     * @return                                  <code>true</code> if the set of databases changed
     * @throws DatalogException                 thrown if the disjunction can't be added
     */
    public boolean addLiteralValue(Predicate predicate,Object[] tuple) throws DatalogException {
        return applyChanges(Collections.singleton(getChangeInfo(predicate,tuple,true)));
    }
    /**
     * Utility method that removes a unit disjunction to the database.
     *
     * @param predicate                         the predicate
     * @param tuple                             the tuple
     * @return                                  <code>true</code> if the set of databases changed
     * @throws DatalogException                 thrown if the disjunction can't be removed
     */
    public boolean removeLiteralValue(Predicate predicate,Object[] tuple) throws DatalogException {
        return applyChanges(Collections.singleton(getChangeInfo(predicate,tuple,false)));
    }
    /**
     * Utility method that adds a disjunction to the databases.
     *
     * @param disjuncts                         the disjuncts
     * @return                                  <code>true</code> if the set of databases changed
     * @throws DatalogException                 thrown if the disjunction can't be added
     */
    public boolean addDisjunction(LiteralValue[] disjuncts) throws DatalogException {
        return applyChanges(Collections.singleton(getChangeInfo(disjuncts,true)));
    }
    /**
     * Utility method that removes a disjunction from the databases.
     *
     * @param disjuncts                         the disjuncts
     * @return                                  <code>true</code> if the set of databases changed
     * @throws DatalogException                 thrown if the disjunction can't be removed
     */
    public boolean removeDisjunction(LiteralValue[] disjuncts) throws DatalogException {
        return applyChanges(Collections.singleton(getChangeInfo(disjuncts,false)));
    }
    /**
     * Processes a batch update of the databases.
     *
     * @param objectsToAdd                      the collection of objects to add, where elements may be unit disjunctions (<code>LiteralValue</code>) or disjunctions (<code>LiteralValue[]</code>)
     * @param objectsToRemove                   the collection of objects to remove, where elements may be unit disjunctions (<code>LiteralValue</code>) or disjunctions (<code>LiteralValue[]</code>)
     * @return                                  <code>true</code> if at least one database extension was changed
     * @throws DatalogException                 thrown if this change is invalid
     */
    public boolean bulkChange(Collection objectsToAdd,Collection objectsToRemove) throws DatalogException {
        List changeInfos=new ArrayList(objectsToAdd.size()+objectsToRemove.size());
        Iterator iterator=objectsToAdd.iterator();
        while (iterator.hasNext()) {
            Object change=iterator.next();
            if (change instanceof LiteralValue[])
                changeInfos.add(getChangeInfo((LiteralValue[])change,true));
            else {
                LiteralValue literalValue=(LiteralValue)change;
                changeInfos.add(getChangeInfo(literalValue.getPredicate(),literalValue.getTuple(),true));
            }
        }
        iterator=objectsToRemove.iterator();
        while (iterator.hasNext()) {
            Object change=iterator.next();
            if (change instanceof LiteralValue[])
                changeInfos.add(getChangeInfo((LiteralValue[])change,false));
            else {
                LiteralValue literalValue=(LiteralValue)change;
                changeInfos.add(getChangeInfo(literalValue.getPredicate(),literalValue.getTuple(),false));
            }
        }
        return applyChanges(changeInfos);
    }
    /**
     * Applies the collection of changes to the databases.
     *
     * @param changeInfos                       the collection of changes being applied
     * @return                                  <code>true</code> if at least one extension changed
     * @throws DatalogException                 thrown if changes can't be applied
     */
    protected boolean applyChanges(Collection changeInfos) throws DatalogException {
        boolean result=false;
        fireExtensionsWillChange(changeInfos);
        Iterator iterator=changeInfos.iterator();
        while (iterator.hasNext()) {
            ChangeInfo changeInfo=(ChangeInfo)iterator.next();
            if (changeInfo.m_isAddition) {
                if (addDisjunctionInternal(changeInfo.m_minimalPredicateExtension,changeInfo.m_minimalTuple,changeInfo.m_disjunctionRestInfo))
                    result=true;
            }
            else {
                if (removeDisjunctionInternal(changeInfo.m_minimalPredicateExtension,changeInfo.m_minimalTuple,changeInfo.m_disjunctionRestInfo))
                    result=true;
            }
        }
        fireExtensionsChanged(changeInfos);
        return result;
    }
    /**
     * Returns a <code>ChangeInfo</code> object for given disjunction.
     *
     * @param disjuncts                         the disjunction being changed
     * @param isAddition                        <code>true</code> if this change is addition
     * @return                                  the <code>ChangeInfo</code> object
     * @throws DatalogException                 thrown if disjunction is invalid
     */
    protected ChangeInfo getChangeInfo(LiteralValue[] disjuncts,boolean isAddition) throws DatalogException {
        if (disjuncts.length==0)
            throw new DatalogException("Empty disjunction can't be a part of the database.");
        if (disjuncts.length>1) {
            disjuncts=(LiteralValue[])disjuncts.clone();
            Arrays.sort(disjuncts,m_literalValueComparator);
        }
        Predicate minimalPredicate=disjuncts[0].m_predicate;
        Object[] minimalTuple=disjuncts[0].m_tuple;
        ExtensionalDatabase extensionalDatabase=getExtensionalDatabaseEx(minimalPredicate);
        if (!(extensionalDatabase instanceof MemoryExtensionalDatabase))
            throw new DatalogException("Can't locate the database to change for predicate '"+minimalPredicate.getFullName()+"'.");
        MemoryExtensionalDatabase memoryExtensionalDatabase=(MemoryExtensionalDatabase)extensionalDatabase;
        MemoryPredicateExtension predicateExtension=memoryExtensionalDatabase.getMemoryPredicateExtension(minimalPredicate);
        DisjunctionRestInfo.DisjunctInfo headDisjunct=null;
        for (int disjunctIndex=disjuncts.length-1;disjunctIndex>=1;disjunctIndex--) {
            LiteralValue literalValue=disjuncts[disjunctIndex];
            if (headDisjunct==null || !headDisjunct.isEqual(literalValue.m_predicate,literalValue.m_tuple))
                headDisjunct=new DisjunctionRestInfo.DisjunctInfo(literalValue.m_predicate,literalValue.m_tuple,headDisjunct);
        }
        return new ChangeInfo(predicateExtension,minimalTuple,headDisjunct==null ? null : new DisjunctionRestInfo(null,headDisjunct),isAddition);
    }
    /**
     * Returns a <code>ChangeInfo</code> object for given unit disjunction.
     *
     * @param predicate                         the predicate
     * @param tuple                             the tuple
     * @param isAddition                        <code>true</code> if this change is addition
     * @return                                  the <code>ChangeInfo</code> object
     * @throws DatalogException                 thrown if change is invalid
     */
    protected ChangeInfo getChangeInfo(Predicate predicate,Object[] tuple,boolean isAddition) throws DatalogException {
        ExtensionalDatabase extensionalDatabase=getExtensionalDatabaseEx(predicate);
        if (!(extensionalDatabase instanceof MemoryExtensionalDatabase))
            throw new DatalogException("Can't locate the database to change for predicate '"+predicate.getFullName()+"'.");
        MemoryExtensionalDatabase memoryExtensionalDatabase=(MemoryExtensionalDatabase)extensionalDatabase;
        MemoryPredicateExtension predicateExtension=memoryExtensionalDatabase.getMemoryPredicateExtension(predicate);
        return new ChangeInfo(predicateExtension,tuple,null,isAddition);
    }
    /**
     * Adds a disjunction to the databases by merging specified disjunction infos.
     *
     * @param disjunctionRestInfos              the array with the information about the rest of the disjunctions
     * @return                                  <code>true</code> if the set of databases changed
     * @throws DatalogException                 thrown if the disjunction can't be added
     */
    protected boolean addDisjunctionByMerging(DisjunctionRestInfo[] disjunctionRestInfos) throws DatalogException {
        // first add a dummy minimal disjunct - this prevents us from having to deal with updating the head
        DisjunctionRestInfo.DisjunctInfo mergedDisjuncts=new DisjunctionRestInfo.DisjunctInfo(null,null,null);
        for (int i=0;i<disjunctionRestInfos.length;i++) {
            DisjunctionRestInfo.DisjunctInfo pointer1=mergedDisjuncts.m_next;
            DisjunctionRestInfo.DisjunctInfo lastDisjunct=mergedDisjuncts;
            DisjunctionRestInfo.DisjunctInfo pointer2=disjunctionRestInfos[i].m_headDisjunct;
            while (pointer1!=null && pointer2!=null) {
                int comparison=m_ordering.compareTo(pointer1.m_predicate,pointer1.m_tuple,pointer2.m_predicate,pointer2.m_tuple);
                if (comparison<0) {
                    lastDisjunct=pointer1;
                    pointer1=pointer1.m_next;
                }
                else if (comparison==0)
                    pointer2=pointer2.m_next;
                else {
                    lastDisjunct.m_next=new DisjunctionRestInfo.DisjunctInfo(pointer2.m_predicate,pointer2.m_tuple,lastDisjunct.m_next);
                    lastDisjunct=lastDisjunct.m_next;
                    pointer2=pointer2.m_next;
                }
            }
            while (pointer2!=null) {
                lastDisjunct.m_next=new DisjunctionRestInfo.DisjunctInfo(pointer2.m_predicate,pointer2.m_tuple,null);
                lastDisjunct=lastDisjunct.m_next;
                pointer2=pointer2.m_next;
            }
        }
        // now throw out the dummy disjunct
        mergedDisjuncts=mergedDisjuncts.m_next;
        // mergedDisjuncts now contains the list of merged disjuncts
        if (mergedDisjuncts==null)
            throw new DatalogException("Can't add empty disjunction to the database.");
        DisjunctionRestInfo disjunctionRestInfo=new DisjunctionRestInfo(null,mergedDisjuncts.m_next);
        ExtensionalDatabase addToDatabase=getExtensionalDatabaseEx(mergedDisjuncts.m_predicate);
        if (!(addToDatabase instanceof MemoryExtensionalDatabase))
            throw new DatalogException("Can't locate the database to add the predicate '"+mergedDisjuncts.m_predicate.getFullName()+"'.");
        MemoryExtensionalDatabase addToMemoryDatabase=(MemoryExtensionalDatabase)addToDatabase;
        MemoryPredicateExtension minimalPredicateExtension=addToMemoryDatabase.getMemoryPredicateExtension(mergedDisjuncts.m_predicate);
        return addDisjunctionInternal(minimalPredicateExtension,mergedDisjuncts.m_tuple,disjunctionRestInfo);
    }
    /**
     * This method must be exposed publicly, but is intended for internal purposes. Don't call it!
     *
     * @param minimalPredicate                  the minial predicate
     * @param minimalTuple                      the minimal tuple
     * @param disjunctionRestInfo               the information about the rest of the disjunction
     * @return                                  <code>true</code> if the set of databases changed
     * @throws DatalogException                 thrown if the disjunction can't be added
     */
    public boolean addDisjunctionInternal(Predicate minimalPredicate,Object[] minimalTuple,DisjunctionRestInfo disjunctionRestInfo) throws DatalogException {
        ExtensionalDatabase addToDatabase=getExtensionalDatabaseEx(minimalPredicate);
        if (!(addToDatabase instanceof MemoryExtensionalDatabase))
            throw new DatalogException("Can't locate the database to add the predicate '"+minimalPredicate.getFullName()+"'.");
        MemoryExtensionalDatabase addToMemoryDatabase=(MemoryExtensionalDatabase)addToDatabase;
        MemoryPredicateExtension minimalPredicateExtension=addToMemoryDatabase.getMemoryPredicateExtension(minimalPredicate);
        return addDisjunctionInternal(minimalPredicateExtension,minimalTuple,disjunctionRestInfo);
    }
    /**
     * Adds a disjunction to the databases. The disjunction must be correctly ordered.
     *
     * @param minimalPredicateExtension         the extension for the minial predicate
     * @param minimalTuple                      the minimal tuple
     * @param disjunctionRestInfo               the information about the rest of the disjunction
     * @return                                  <code>true</code> if the set of databases changed
     * @throws DatalogException                 thrown if the disjunction can't be added
     */
    protected boolean addDisjunctionInternal(MemoryPredicateExtension minimalPredicateExtension,Object[] minimalTuple,DisjunctionRestInfo disjunctionRestInfo) throws DatalogException {
        if (minimalPredicateExtension.getContainsDisjunctionInfo()) {
            Object[] databaseTuple=minimalPredicateExtension.getExistingTuple(minimalTuple);
            if (databaseTuple==null) {
                databaseTuple=new Object[minimalTuple.length+1];
                for (int i=0;i<minimalTuple.length;i++)
                    databaseTuple[i]=minimalTuple[i];
                minimalPredicateExtension.addTuple(databaseTuple);
            }
            if (disjunctionRestInfo==null)
                disjunctionRestInfo=new DisjunctionRestInfo(null);
            DisjunctionRestInfo existingDisjunction=(DisjunctionRestInfo)databaseTuple[databaseTuple.length-1];
            while (existingDisjunction!=null) {
                if (disjunctionRestInfo.isEqual(existingDisjunction))
                    return false;
                existingDisjunction=existingDisjunction.m_next;
            }
            disjunctionRestInfo.m_next=(DisjunctionRestInfo)databaseTuple[databaseTuple.length-1];
            databaseTuple[databaseTuple.length-1]=disjunctionRestInfo;
            return true;
        }
        else {
            if (disjunctionRestInfo!=null && (disjunctionRestInfo.m_next!=null || disjunctionRestInfo.m_headDisjunct!=null))
                throw new DatalogException("Can't store a disjunction into a non-disjunction extension for predicate '"+minimalPredicateExtension.getPredicate().getFullName()+"'");
            return minimalPredicateExtension.addTuple(minimalTuple);
        }
    }
    /**
     * Removes a disjunction from the databases. The disjunction must be correctly ordered.
     *
     * @param minimalPredicateExtension         the extension for the minial predicate
     * @param minimalTuple                      the minimal tuple
     * @param disjunctionRestInfo               the information about the rest of the disjunction
     * @return                                  <code>true</code> if the set of databases changed
     * @throws DatalogException                 thrown if the disjunction can't be removed
     */
    protected boolean removeDisjunctionInternal(MemoryPredicateExtension minimalPredicateExtension,Object[] minimalTuple,DisjunctionRestInfo disjunctionRestInfo) throws DatalogException {
        if (minimalPredicateExtension.getContainsDisjunctionInfo()) {
            Object[] databaseTuple=minimalPredicateExtension.getExistingTuple(minimalTuple);
            if (databaseTuple!=null) {
                if (disjunctionRestInfo==null)
                    disjunctionRestInfo=new DisjunctionRestInfo(null);
                DisjunctionRestInfo lastDisjunction=null;
                DisjunctionRestInfo disjunction=(DisjunctionRestInfo)databaseTuple[databaseTuple.length-1];
                while (disjunction!=null) {
                    if (disjunctionRestInfo.isEqual(disjunction)) {
                        if (lastDisjunction==null)
                            databaseTuple[databaseTuple.length-1]=disjunction.m_next;
                        else
                            lastDisjunction.m_next=disjunction.m_next;
                        if (databaseTuple[databaseTuple.length-1]==null)
                            minimalPredicateExtension.removeTuple(databaseTuple);
                        return true;
                    }
                    lastDisjunction=disjunction;
                    disjunction=disjunction.m_next;
                }
            }
            return false;
        }
        else {
            if (disjunctionRestInfo!=null && (disjunctionRestInfo.m_next!=null || disjunctionRestInfo.m_headDisjunct!=null))
                throw new DatalogException("Can't remove a disjunction from a non-disjunction extension for predicate '"+minimalPredicateExtension.getPredicate().getFullName()+"'");
            return minimalPredicateExtension.removeTuple(minimalTuple)!=null;
        }
    }

    /**
     * The comparator for literal values.
     */
    protected class LiteralValueComparator implements Comparator {
        public int compare(Object o1,Object o2) {
            LiteralValue lv1=(LiteralValue)o1;
            LiteralValue lv2=(LiteralValue)o2;
            return m_ordering.compareTo(lv1.m_predicate,lv1.m_tuple,lv2.m_predicate,lv2.m_tuple);
        }
    }

    /**
     * The listener interface receiving information about changes in the database extensions.
     */
    public static interface ExensionChangeListener extends EventListener {
        /**
         * Receives a notification that some extensions in the databases managed by this manager will change.
         *
         * @param extensionalManager            the manager that is being changed
         * @param changeInfos                   the collection if information about changes
         */
        void extensionsWillChange(ExtensionalManager extensionalManager,Collection changeInfos);
        /**
         * Receives a notification that some extensions in the databases managed by this manager have changes.
         *
         * @param extensionalManager            the manager that is being changed
         * @param changeInfos                   the collection if information about changes
         */
        void extensionsChanged(ExtensionalManager extensionalManager,Collection changeInfos);
    }

    /**
     * Contains information about addition or removal of a disjunction in the database.
     */
    public static class ChangeInfo {
        /** The predicate extension being changed. */
        public MemoryPredicateExtension m_minimalPredicateExtension;
        /** The tuple in the predicate extension. */
        public Object[] m_minimalTuple;
        /** The rest of the disjunction. */
        public DisjunctionRestInfo m_disjunctionRestInfo;
        /** Set to <code>true</code> if the change is addition. */
        public boolean m_isAddition;

        /**
         * Creates an instance of this class.
         *
         * @param minimalPredicateExtension     the predicate extension
         * @param minimalTuple                  the tuple
         * @param disjunctionRestInfo           the rest of the disjunction (can be <code>null</code>)
         * @param isAddition                    set to <code>true</code> if tuple is being added
         */
        public ChangeInfo(MemoryPredicateExtension minimalPredicateExtension,Object[] minimalTuple,DisjunctionRestInfo disjunctionRestInfo,boolean isAddition) {
            m_minimalPredicateExtension=minimalPredicateExtension;
            m_minimalTuple=minimalTuple;
            m_disjunctionRestInfo=disjunctionRestInfo;
            m_isAddition=isAddition;
        }
    }
}
