package edu.unika.aifb.kaon.datalog.evaluation;

import edu.unika.aifb.kaon.datalog.program.*;

/**
 * Represents the information about the rest of a ground disjunction.
 */
public class DisjunctionRestInfo {
    /** The pointer to the next disjunction rest. */
    public DisjunctionRestInfo m_next;
    /** The pointer to the first disjunct. */
    public DisjunctInfo m_headDisjunct;
    /** Hash used for optimized checking whether two disjunctions are equal. */
    public int m_hash;

    /**
     * Creates an instance of this class.
     *
     * @param next                                  the pointer to the next disjunction rest
     */
    public DisjunctionRestInfo(DisjunctionRestInfo next) {
        m_next=next;
    }
    /**
     * Creates an instance of this class.
     *
     * @param next                                  the pointer to the next disjunction rest
     * @param disjuncts                             the list of disjuncts
     */
    public DisjunctionRestInfo(DisjunctionRestInfo next,DisjunctInfo disjuncts) {
        m_next=next;
        m_headDisjunct=disjuncts;
        while (disjuncts!=null) {
            m_hash=m_hash*7+disjuncts.hashCode();
            disjuncts=disjuncts.m_next;
        }
    }
    /**
     * Creates an instance of this class.
     *
     * @param next                                  the pointer to the next disjunction rest
     * @param copyFrom                              the element to copy
     */
    public DisjunctionRestInfo(DisjunctionRestInfo next,DisjunctionRestInfo copyFrom) {
        m_next=next;
        m_headDisjunct=copyFrom.m_headDisjunct;
        m_hash=copyFrom.m_hash;
    }
    /**
     * Checks whether this object is equal to the other object. We deliberately don't override equals() because this equality doesn't
     * mean that the m_next member is equal - it means that the disjunction is equal.
     *
     * @param that                                  the other disjunction
     * @return                                      <code>true</code> if disjunction rests are equal
     */
    public boolean isEqual(DisjunctionRestInfo that) {
        if (this==that)
            return true;
        if (m_hash!=that.m_hash)
            return false;
        DisjunctInfo pointer1=m_headDisjunct;
        DisjunctInfo pointer2=that.m_headDisjunct;
        while (pointer1!=null && pointer2!=null) {
            if (!pointer1.equals(pointer2))
                return false;
            pointer1=pointer1.m_next;
            pointer2=pointer2.m_next;
        }
        return pointer1==pointer2;  // this is actually a comparison whether both of them are null, since we know that the previous loop will terminate only if one of them is null
    }

    /**
     * Information about a single disjunct.
     */
    public static class DisjunctInfo extends LiteralValue {
        /** The link to the next disjunct. */
        public DisjunctInfo m_next;

        public DisjunctInfo(Predicate predicate,Object[] tuple,DisjunctInfo next) {
            super(predicate,tuple);
            m_next=next;
        }
        public boolean isEqual(Predicate predicate,Object[] tuple) {
            if (!m_predicate.equals(predicate))
                return false;
            for (int i=0;i<m_tuple.length;i++) {
                Object object1=m_tuple[i];
                Object object2=tuple[i];
                if (object1!=object2 && (object1==null || !object1.equals(object2)))
                    return false;
            }
            return true;
        }
    }
}
