package edu.unika.aifb.kaon.datalog.evaluation;

import java.util.Iterator;
import java.util.Collections;

import edu.unika.aifb.kaon.datalog.*;

/**
 * Implements a nested loop join between the build and probe tuple stream. The algorithm is based
 * on the hash-join - the hashing is used to eliminate duplicates from the build stream.
 */
public abstract class AbstractNestedLoopJoin extends AbstractHashJoin {
    /** The iterator over the build keys. */
    protected Iterator m_buildKeys;
    /** The iterator over the current nested loop. */
    protected Iterator m_nestedLoopIterator;
    /** The current probe tuple. */
    protected Object[] m_probeCurrentTuple;

    /**
     * Creates an instance of this class.
     *
     * @param buildOperator                     the operator generating the build tuple stream
     * @param buildJoinIndices                  the indices from the build stream that are part of the join
     * @param buildNonJoinIndices               the indices from the build stream that are not part of the join, but that with the join indices determine the uniquess of the tuple
     * @param probeJoinIndices                  the indices from the probe stream that are part of the join
     * @param probeNonJoinIndices               the indices from the probe stream that are not part of the join, but that with the join indices determine the uniquess of the tuple
     * @param buildIndicesToCopy                the indices to copy from the build stream to the result
     * @param probeIndicesToCopy                the indices to copy from the probe stream to the result
     * @param residualConditions                the residual join conditions
     */
    public AbstractNestedLoopJoin(QueryOperator buildOperator,int[] buildJoinIndices,int[] buildNonJoinIndices,int[] probeJoinIndices,int[] probeNonJoinIndices,int[][] buildIndicesToCopy,int[][] probeIndicesToCopy,JoinTupleFilter residualConditions) {
        super(buildOperator,buildJoinIndices,buildNonJoinIndices,probeJoinIndices,probeNonJoinIndices,buildIndicesToCopy,probeIndicesToCopy,residualConditions);
    }
    /**
     * Stops the processing of the query.
     *
     * @throws DatalogException                 thrown if there is an error in evaluation
     */
    public void stop() throws DatalogException {
        m_buildKeys=null;
        m_nestedLoopIterator=null;
        m_probeCurrentTuple=null;
        super.stop();
    }
    /**
     * Called after the build is processed. Probe is generated here.
     *
     * @throws DatalogException                 thrown if there is an error in evaluation
     */
    protected void buildProcessed() throws DatalogException {
        m_buildKeys=m_joinMap.keyIterator();
        m_nestedLoopIterator=Collections.EMPTY_SET.iterator();
        probeNext();
    }
    /**
     * Returns the current tuple of the probe stream. If the stream is at the end, <code>null</code> is returned.
     *
     * @return                                  the current tuple of the probe stream (or <code>null</code> if the stream is at the end)
     */
    protected Object[] probeTuple() {
        return m_probeCurrentTuple;
    }
    /**
     * Moves the probe stream to the next element. If the probe stream is at the end, this method has no effects.
     *
     * @throws DatalogException                 thrown if there is an error in evaluation
     */
    protected void probeNext() throws DatalogException {
        while (true) {
            if (!m_nestedLoopIterator.hasNext()) {
                if (!m_buildKeys.hasNext()) {
                    m_probeCurrentTuple=null;
                    return;
                }
                Object[] buildTuple=(Object[])m_buildKeys.next();
                m_nestedLoopIterator=getNestedLoopIterator(buildTuple);
            }
            if (m_nestedLoopIterator.hasNext()) {
                m_probeCurrentTuple=(Object[])m_nestedLoopIterator.next();
                return;
            }
        }
    }
    /**
     * Returns <code>true</code> if the probe stream has completely been used up.
     *
     * @return                                  <code>true</code> if the probe stream has been used up
     */
    protected boolean probeAfterLast() {
        return m_probeCurrentTuple==null;
    }
    /**
     * Returns an nested loop iterator for the given outer tuple. This method should always return
     * a valid iterator (if the nested loop shouldn't be executed, an empty iterator should be returned).
     *
     * @param buildTuple                        the build tuple for which the loop is started
     * @return                                  the nested loop iterator for given build tuple
     * @throws DatalogException                 thrown if there is an error in evaluation
     */
    protected abstract Iterator getNestedLoopIterator(Object[] buildTuple) throws DatalogException;
}
