package edu.unika.aifb.kaon.datalog.evaluation;

import edu.unika.aifb.kaon.datalog.*;

/**
 * An abstract class for all operators implementing a join. The join is performed between the build
 * and a probe stream.
 */
public abstract class AbstractJoin implements QueryOperator {
    /** The current tuple. */
    protected Object[] m_currentTuple;
    /** The join indices from the build stream. */
    protected int[] m_buildJoinIndices;
    /** The other indices from build stream. */
    protected int[] m_buildNonJoinIndices;
    /** The join indices from the probe stream. */
    protected int[] m_probeJoinIndices;
    /** The other indices from the probe stream. */
    protected int[] m_probeNonJoinIndices;
    /** The pairs of indices from the build stream to be copied to the result. */
    protected int[][] m_buildIndicesToCopy;
    /** The pairs of indices from the probe stream to be copied to the result. */
    protected int[][] m_probeIndicesToCopy;
    /** The size of the resulting tuple. */
    protected int m_resultTupleSize;
    /** The join filter for the residual conditions. */
    protected JoinTupleFilter m_residualConditions;

    /**
     * Creates an instance of this class.
     *
     * @param buildJoinIndices                  the indices from the build stream that are part of the join
     * @param buildNonJoinIndices               the indices from the build stream that are not part of the join, but that with the join indices determine the uniquess of the tuple
     * @param probeJoinIndices                  the indices from the probe stream that are part of the join
     * @param probeNonJoinIndices               the indices from the probe stream that are not part of the join, but that with the join indices determine the uniquess of the tuple
     * @param buildIndicesToCopy                the indices to copy from the build stream to the result
     * @param probeIndicesToCopy                the indices to copy from the probe stream to the result
     * @param residualConditions                the residual join conditions
     */
    public AbstractJoin(int[] buildJoinIndices,int[] buildNonJoinIndices,int[] probeJoinIndices,int[] probeNonJoinIndices,int[][] buildIndicesToCopy,int[][] probeIndicesToCopy,JoinTupleFilter residualConditions) {
        m_buildJoinIndices=buildJoinIndices;
        m_buildNonJoinIndices=buildNonJoinIndices;
        m_probeJoinIndices=probeJoinIndices;
        m_probeNonJoinIndices=probeNonJoinIndices;
        m_buildIndicesToCopy=buildIndicesToCopy;
        m_probeIndicesToCopy=probeIndicesToCopy;
        m_residualConditions=residualConditions;
        m_resultTupleSize=m_buildIndicesToCopy.length+m_probeIndicesToCopy.length;
    }
    /**
     * Stops the processing of the query.
     *
     * @throws DatalogException                 thrown if there is an error in evaluation
     */
    public void stop() throws DatalogException {
        m_currentTuple=null;
    }
    /**
     * Returns the current tuple of the operator. If the tuple stream is at the end, <code>null</code> is returned.
     *
     * @return                                  the current tuple (or <code>null</code> if the tuple stream is at the end)
     */
    public Object[] tuple() {
        return m_currentTuple;
    }
    /**
     * Returns <code>true</code> if the current stream is after the last tuple.
     *
     * @return                                  <code>true</code> if the stream is after the last tuple
     */
    public boolean afterLast() {
        return m_currentTuple==null;
    }
    /**
     * Builds a resulting tuple from given build and probe tuples.
     *
     * @param buildTuple                        the build tuple
     * @param probeTuple                        the probe tuple
     * @return                                  the result tuple
     */
    protected Object[] resultTuple(Object[] buildTuple,Object[] probeTuple) {
        Object[] result=new Object[m_resultTupleSize];
        for (int i=m_buildIndicesToCopy.length-1;i>=0;--i)
            result[m_buildIndicesToCopy[i][1]]=buildTuple[m_buildIndicesToCopy[i][0]];
        for (int i=m_probeIndicesToCopy.length-1;i>=0;--i)
            result[m_probeIndicesToCopy[i][1]]=probeTuple[m_probeIndicesToCopy[i][0]];
        return result;
    }
}
