package edu.unika.aifb.kaon.datalog.evaluation;

import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.Collections;

import edu.unika.aifb.kaon.datalog.*;

/**
 * An abstract class for all operators implementing a hash-join. The operator must be able to receive
 * duplicate tuples from the build operator, but duplicates are not allowed in the probe stream.
 * The hash-join is performed by first reading the entire input stream. Each tuple from the input
 * stream is indexed into the hash-map of tuples through the key values. The value in the hash-table is
 * the set of tuples (the set is used to eliminate duplicates). Then the probe stream is scanned and
 * the join is performed.
 */
public abstract class AbstractHashJoin extends AbstractJoin {
    /** The build operator. */
    protected QueryOperator m_buildOperator;
    /** The map of tuples built during a join. */
    protected TupleHashMap m_joinMap;
    /** The current probe tuple. */
    protected Object[] m_currentProbeTuple;
    /** The tuples from the build operator that were matched with the probe stream. */
    protected Iterator m_matchedBuildTuples;

    /**
     * Creates an instance of this class.
     *
     * @param buildOperator                     the operator generating the build tuple stream
     * @param buildJoinIndices                  the indices from the build stream that are part of the join
     * @param buildNonJoinIndices               the indices from the build stream that are not part of the join, but that with the join indices determine the uniquess of the tuple
     * @param probeJoinIndices                  the indices from the probe stream that are part of the join
     * @param probeNonJoinIndices               the indices from the probe stream that are not part of the join, but that with the join indices determine the uniquess of the tuple
     * @param buildIndicesToCopy                the indices to copy from the build stream to the result
     * @param probeIndicesToCopy                the indices to copy from the probe stream to the result
     * @param residualConditions                the residual join conditions
     */
    public AbstractHashJoin(QueryOperator buildOperator,int[] buildJoinIndices,int[] buildNonJoinIndices,int[] probeJoinIndices,int[] probeNonJoinIndices,int[][] buildIndicesToCopy,int[][] probeIndicesToCopy,JoinTupleFilter residualConditions) {
        super(buildJoinIndices,buildNonJoinIndices,probeJoinIndices,probeNonJoinIndices,buildIndicesToCopy,probeIndicesToCopy,residualConditions);
        m_buildOperator=buildOperator;
    }
    /**
     * Starts the processing of the query. The cursor is positioned on the first tuple, or at end.
     *
     * @throws DatalogException                 thrown if there is an error in evaluation
     */
    public void start() throws DatalogException {
        m_joinMap=new TupleHashMap(m_buildJoinIndices);
        m_buildOperator.start();
        while (!m_buildOperator.afterLast()) {
            Object[] tuple=m_buildOperator.tuple();
            Object existingObject=m_joinMap.get(tuple);
            if (existingObject==null)
                m_joinMap.put(tuple,tuple);
            else if (existingObject instanceof TupleHashSet)
                ((TupleHashSet)existingObject).add(tuple);
            else {
                TupleHashSet set=new TupleHashSet(m_buildNonJoinIndices);
                set.add(tuple);
                set.add((Object[])existingObject);
                m_joinMap.put(tuple,set);
            }
            m_buildOperator.next();
        }
        if (m_joinMap.size()!=0) {
            buildProcessed();
            m_matchedBuildTuples=Collections.EMPTY_SET.iterator();
            next();
        }
    }
    /**
     * Stops the processing of the query.
     *
     * @throws DatalogException                 thrown if there is an error in evaluation
     */
    public void stop() throws DatalogException {
        super.stop();
        m_joinMap=null;
        m_matchedBuildTuples=null;
        m_currentProbeTuple=null;
        m_buildOperator.stop();
    }
    /**
     * Moves the cursor to the next position. If the tuple stream is at the end this method call has no effect.
     *
     * @throws DatalogException                 thrown if there is an error in evaluation
     */
    public void next() throws DatalogException {
        while (m_matchedBuildTuples!=null) {
            if (!m_matchedBuildTuples.hasNext())
                advanceProbe();
            if (m_matchedBuildTuples!=null) {
                Object[] buildTuple=(Object[])m_matchedBuildTuples.next();
                if (m_residualConditions.shouldJoin(buildTuple,m_currentProbeTuple)) {
                    m_currentTuple=resultTuple(buildTuple,m_currentProbeTuple);
                    return;
                }
            }
        }
        m_currentTuple=null;
        m_matchedBuildTuples=null;
    }
    /**
     * Advances the probe and sets the next iterator of build tuples that match the probe tuple.
     *
     * @throws DatalogException                 thrown if there is an error in evaluation
     */
    protected void advanceProbe() throws DatalogException {
        while (!probeAfterLast()) {
            m_currentProbeTuple=probeTuple();
            probeNext();
            Object existingObject=m_joinMap.get(m_currentProbeTuple,m_probeJoinIndices);
            if (existingObject instanceof TupleHashSet) {
                m_matchedBuildTuples=((TupleHashSet)existingObject).iterator();
                return;
            }
            else if (existingObject!=null) {
                m_matchedBuildTuples=new SingletonIterator(existingObject);
                return;
            }
        }
        m_matchedBuildTuples=null;
        m_currentProbeTuple=null;
    }
    /**
     * Called after the build stream is processed. Subclasses can do some processing here.
     *
     * @throws DatalogException                 thrown if there is an error in evaluation
     */
    protected void buildProcessed() throws DatalogException {
    }
    /**
     * Returns the current tuple of the probe stream. If the stream is at the end, <code>null</code> is returned.
     *
     * @return                                  the current tuple of the probe stream (or <code>null</code> if the stream is at the end)
     * @throws DatalogException                 thrown if there is an error in evaluation
     */
    protected abstract Object[] probeTuple() throws DatalogException;
    /**
     * Moves the probe stream to the next element. If the probe stream is at the end, this method has no effects.
     *
     * @throws DatalogException                 thrown if there is an error in evaluation
     */
    protected abstract void probeNext() throws DatalogException;
    /**
     * Returns <code>true</code> if the probe stream has completely been used up.
     *
     * @return                                  <code>true</code> if the probe stream has been used up
     * @throws DatalogException                 thrown if there is an error in evaluation
     */
    protected abstract boolean probeAfterLast() throws DatalogException;

    /**
     * A singleton iterator.
     */
    protected static class SingletonIterator implements Iterator {
        protected Object m_object;

        public SingletonIterator(Object object) {
            m_object=object;
        }
        public boolean hasNext() {
            return m_object!=null;
        }
        public Object next() {
            if (m_object==null)
                throw new NoSuchElementException();
            Object result=m_object;
            m_object=null;
            return result;
        }
        public void remove() {
            throw new UnsupportedOperationException();
        }
    }
}
