package edu.unika.aifb.kaon.apiproxy.source;

import java.util.Map;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;

/**
 * Represents a source for an OI-model. There may be different sources, each providing the
 * local version of the API with KAON data from particular source. OI-model source must be thread-safe.
 */
public interface OIModelSource {
    /**
     * Returns the logical URI of this source.
     *
     * @return                              the logical URI of this source
     * @throws KAONException                thrown in case of an error
     */
    String getLogicalURI() throws KAONException;
    /**
     * Returns the physical URI of this source.
     *
     * @return                              the physical URI of this source
     * @throws KAONException                thrown in case of an error
     */
    String getPhysicalURI() throws KAONException;
    /**
     * Called to load information about specified entities.
     *
     * @param entityURIs                    an array of entity URIs to load
     * @param loadFlag                      the flag specifying what to load
     * @return                              information about loaded entities
     * @throws KAONException                thrown if there is a problem with fetching information
     */
    EntityInfo[] loadEntities(String[] entityURIs,int loadFlag) throws KAONException;
    /**
     * Checks whether given instance is connected to specified value through a specified property.
     *
     * @param instanceURI                   the URI of the instance from which the check is made
     * @param propertyURI                   the URI of the property
     * @param targetInstanceURI             the URI of the target instance
     * @param targetValue                   the target value
     * @return                              <code>true</code> if this instance points to given value through given property (by including all inferred facts)
     * @throws KAONException                thrown if the check can't be performed
     */
    boolean pointsToValue(String instanceURI,String propertyURI,String targetInstanceURI,Object targetValue) throws KAONException;
    /**
     * Returns the information about all property instances connected to given instance, possibly filtered through given property.
     *
     * @param instanceURI                   the URI of the instance for which property instances are loaded
     * @param propertyURI                   the URI of the property for which property instances are loaded (may be <code>null</code>)
     * @return                              equal to the structure of <code>InstanceInfo.m_propertyValuesFrom</code>
     * @throws KAONException                thrown if the property instances can't be determined
     */
    Object[][] getAllFromPropertyValues(String instanceURI,String propertyURI) throws KAONException;
    /**
     * Returns the information about all property instances pointing from given instance, possibly filtered through given property.
     *
     * @param instanceURI                   the URI of the instance from which property instances are loaded
     * @param propertyURI                   the URI of the property for which property instances are loaded (may be <code>null</code>)
     * @return                              equal to the structure of <code>InstanceInfo.m_propertyValuesTo</code>
     * @throws KAONException                thrown if the property instances can't be determined
     */
    Object[][] getAllToPropertyValues(String instanceURI,String propertyURI) throws KAONException;
    /**
     * Returns the list of IDs of all concepts.
     *
     * @return                              the list of all IDs of concepts
     * @throws KAONException                thrown if there is a problem with fetching information
     */
    EntityID[] getConcepts() throws KAONException;
    /**
     * Returns the list of IDs of all properties.
     *
     * @return                              the list of all IDs of properties
     * @throws KAONException                thrown if there is a problem with fetching information
     */
    EntityID[] getProperties() throws KAONException;
    /**
     * Returns the list of IDs of all instances.
     *
     * @return                              the list of all IDs of instances
     * @throws KAONException                thrown if there is a problem with fetching information
     */
    EntityID[] getInstances() throws KAONException;
    /**
     * Executes given query and returns the results.
     *
     * @param queryString                   the query being executed
     * @return                              the results of the query
     * @throws KAONException                thrown if there is an error executing the query
     */
    Object[] executeQuery(String queryString) throws KAONException;
    /**
     * Deletes the model.
     *
     * @throws KAONException                thrown if there is a problem with fetching or updating information
     */
    void deleteOIModel() throws KAONException;
    /**
     * Returns an OI-model for given model ID.
     *
     * @param modelID                       the ID of the OI-model
     * @return                              the OI-model with given ID
     * @throws KAONException                thrown if the ID cannot be translated into an OI-model
     */
    OIModel getOIModel(int modelID) throws KAONException;
    /**
     * Sets the attribute of the OI-model.
     *
     * @param key                           the key
     * @param value                         the value (or <code>null</code> if the attribute should be deleted)
     * @throws KAONException                thrown if the attribute can't be set
     */
    void setAttribute(String key,String value) throws KAONException;
    /**
     * Returns the map of attributes of this OI-model.
     *
     * @return                              the map of attributes of the OI-model
     * @throws KAONException                thrown if the attribute can't be read
     */
    Map getAttributes() throws KAONException;
}
