package edu.unika.aifb.kaon.apiproxy;

import java.util.Set;
import java.util.HashSet;
import java.util.Map;
import java.util.HashMap;
import java.util.Iterator;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;

import edu.unika.aifb.kaon.apiproxy.source.*;

/**
 * Implementation of the Property interface that relies on the RDF API as the model.
 */
public class PropertyProxy extends AbstractEntityProxy implements Property {
    /** Inverse property. */
    transient protected Property m_inverseProperty;
    /** The model in which the inverse property was declared. */
    transient protected OIModel m_invesePropertyOIModel;
    /** <code>true</code> if this property is symmetric. */
    transient protected boolean m_symmetric;
    /** <code>true</code> if this property is transitive. */
    transient protected boolean m_transitive;
    /** <code>true</code> if this property is attribute. */
    transient protected boolean m_isAttribute;
    /** Set of domain concepts and cardinalities. */
    transient protected Map m_domainConceptsAndCardinalitites;
    /** Set of range concepts. */
    transient protected Map m_rangeConcepts;
    /** Set of subproperties. */
    transient protected Set m_subProperties;
    /** Set of superproperties. */
    transient protected Map m_superProperties;
    /** Set of property instances. */
    transient protected Set m_propertyInstances;
    /** A strong reference to the spanning instance. */
    transient protected Instance m_spanningInstance;

    /**
     * Creates an instance of this class and attaches it to the OI-model.
     *
     * @param oimodel                   OI-model
     * @param uri                       URI of the concept
     */
    public PropertyProxy(OIModelProxy oimodel,String uri) {
        super(oimodel,uri);
    }
    /**
     * This constructor initializes only the URI and the version, and should be used exclusively for
     * creating dummy objects for passing them from server.
     *
     * @param uri                       URI of the concept
     * @param version                   the version of the object
     */
    public PropertyProxy(String uri,int version) {
        super(uri,version);
    }
    /**
     * Unloads this object.
     */
    public void unload() {
        synchronized (getLock()) {
            super.unload();
            m_inverseProperty=null;
            m_invesePropertyOIModel=null;
            m_symmetric=false;
            m_transitive=false;
            m_isAttribute=false;
            m_domainConceptsAndCardinalitites=null;
            m_rangeConcepts=null;
            m_subProperties=null;
            m_superProperties=null;
            m_propertyInstances=null;
            m_spanningInstance=null;
        }
    }
    /**
     * Returns the bit mask specifying which parts of the object have been loaded.
     *
     * @return                              the bit mask specifying which parts of the object have been loaded
     */
    public int getLoadedState() {
        synchronized (getLock()) {
            int loadedState=0;
            if (hasLoadedBasics())
                loadedState|=OIModel.LOAD_PROPERTY_BASICS;
            if (m_domainConceptsAndCardinalitites!=null)
                loadedState|=OIModel.LOAD_PROPERTY_DOMAINS;
            if (m_rangeConcepts!=null)
                loadedState|=OIModel.LOAD_PROPERTY_RANGES;
            if (m_subProperties!=null)
                loadedState|=OIModel.LOAD_SUB_PROPERTIES;
            if (m_superProperties!=null)
                loadedState|=OIModel.LOAD_SUPER_PROPERTIES;
            if (m_propertyInstances!=null)
                loadedState|=OIModel.LOAD_PROPERTY_INSTANCES;
            return loadedState;
        }
    }
    /**
     * Returns the inverse property of this property. If this property doesn't have an inverse property, then
     * <code>null</code> is returned.
     *
     * @return                          inverse property of this property (or <code>null</code> if there is no inverse property)
     */
    public Property getInverseProperty() throws KAONException {
        synchronized (getLock()) {
            makeSureBasicsLoaded();
            return m_inverseProperty;
        }
    }
    /**
     * Returns the OI-model in which the inverse property of this property has been declared.
     *
     * @return                          the OI-model in which the inverse property of this property has been declared
     */
    public OIModel getInversePropertyOIModel() throws KAONException {
        synchronized (getLock()) {
            makeSureBasicsLoaded();
            return m_invesePropertyOIModel;
        }
    }
    /**
     * Returns <code>true</code> if the inverse property of this property has been specified within the OI-model as returned
     * by <code>getOIModel()</code>.
     *
     * @return                          <code>true</code> if the inverse property of this property has been specified whithin the OI-model as returned by <code>getOIModel()</code>
     */
    public boolean isInversePropertyDeclaredLocally() throws KAONException {
        synchronized (getLock()) {
            return getInversePropertyOIModel()==m_oimodel;
        }
    }
    /**
     * Returns <code>true</code> if this property is symmetric.
     *
     * @return                          <code>true</code> if this property is symmetric
     */
    public boolean isSymmetric() throws KAONException {
        synchronized (getLock()) {
            makeSureBasicsLoaded();
            return m_symmetric;
        }
    }
    /**
     * Returns <code>true</code> if this property is transitive.
     *
     * @return                          <code>true</code> if this property is transitive
     */
    public boolean isTransitive() throws KAONException {
        synchronized (getLock()) {
            makeSureBasicsLoaded();
            return m_transitive;
        }
    }
    /**
     * Return all concepts that consitute the domain of this property.
     *
     * @return                          array of concepts that consitute the domain of this property (may be empty)
     */
    public Set getDomainConcepts() throws KAONException {
        synchronized (getLock()) {
            if (m_domainConceptsAndCardinalitites==null)
                loadThisObject(OIModel.LOAD_PROPERTY_DOMAINS);
            return m_domainConceptsAndCardinalitites.keySet();
        }
    }
    /**
     * Return all concepts that consitute the domain of this property.
     *
     * @return                          set of concepts that consitute the domain of this property (may be empty)
     */
    public Set getAllDomainConcepts() throws KAONException {
        synchronized (getLock()) {
            // if there are hierarcy cycles, this method will loop forever
            Set result=new HashSet(getDomainConcepts());
            Iterator superProperties=getSuperProperties().iterator();
            while (superProperties.hasNext()) {
                Property property=(Property)superProperties.next();
                result.addAll(property.getAllDomainConcepts());
            }
            return result;
        }
    }
    /**
     * Checks whether supplied concept is in the domain of this property.
     *
     * @param concept                   concept that is checked
     * @return                          <code>true</code> if supplied concept is in the domain
     */
    public boolean isDomainConcept(Concept concept) throws KAONException {
        synchronized (getLock()) {
            if (m_domainConceptsAndCardinalitites==null)
                loadThisObject(OIModel.LOAD_PROPERTY_DOMAINS);
            return m_domainConceptsAndCardinalitites.keySet().contains(concept);
        }
    }
    /**
     * Returns the OI-model in which the supplied concept was added as the domain to this property.
     *
     * @param concept                   concept that is checked
     * @return                          the OI-model in which supplied concept was added as the domain to this property
     */
    public OIModel getDomainConceptOIModel(Concept concept) throws KAONException {
        synchronized (getLock()) {
            if (m_domainConceptsAndCardinalitites==null)
                loadThisObject(OIModel.LOAD_PROPERTY_DOMAINS);
            Cardinality cardinality=(Cardinality)m_domainConceptsAndCardinalitites.get(concept);
            return cardinality.m_sourceOIModel;
        }
    }
    /**
     * Returns <code>true</code> if supplied concept was added as the domain to this property in
     * the model which returned by <code>getOIModel()</code>.
     *
     * @param concept                   the concept that is cheched
     * @return                          <code>true</code> if supplied concept was added to the domain concepts wihtin the <code>getOIModel()</code> model
     */
    public boolean isDomainConceptDeclaredLocally(Concept concept) throws KAONException {
        synchronized (getLock()) {
            return getDomainConceptOIModel(concept)==m_oimodel;
        }
    }
    /**
     * Returns the minimum cardinality for given domain concept.
     *
     * @param concept                   concept for which cardinality is requested
     * @return                          minimum cardinality of this property for given concept
     */
    public int getMinimumCardinality(Concept concept) throws KAONException {
        synchronized (getLock()) {
            if (m_domainConceptsAndCardinalitites==null)
                loadThisObject(OIModel.LOAD_PROPERTY_DOMAINS);
            Cardinality cardinality=(Cardinality)m_domainConceptsAndCardinalitites.get(concept);
            if (cardinality==null)
                return 0;
            else
                return cardinality.m_minimum;
        }
    }
    /**
     * Returns the maximum cardinality for given domain concept. In case of unlimited cardinality, Integer.MAX_VALUE is returned.
     *
     * @param concept                   concept for which cardinality is requested
     * @return                          maximum cardinality of this property for given concept
     */
    public int getMaximumCardinality(Concept concept) throws KAONException {
        synchronized (getLock()) {
            if (m_domainConceptsAndCardinalitites==null)
                loadThisObject(OIModel.LOAD_PROPERTY_DOMAINS);
            Cardinality cardinality=(Cardinality)m_domainConceptsAndCardinalitites.get(concept);
            if (cardinality==null)
                return Integer.MAX_VALUE;
            else
                return cardinality.m_maximum;
        }
    }
    /**
     * Return all concepts that consitute the range of this property.
     *
     * @return                          array of concepts that consitute the range of this property (may be empty)
     */
    public Set getRangeConcepts() throws KAONException {
        synchronized (getLock()) {
            if (m_rangeConcepts==null)
                loadThisObject(OIModel.LOAD_PROPERTY_RANGES);
            return m_rangeConcepts.keySet();
        }
    }
    /**
     * Returns the set of all range concepts of this property. If a property is an attribute, empty set is returned.
     *
     * @return                          set of range concepts of this property, or an empty set if property is an attribute
     */
    public Set getAllRangeConcepts() throws KAONException {
        synchronized (getLock()) {
            // if there are hierarcy cycles, this method will loop forever
            Set result=new HashSet(getRangeConcepts());
            Iterator superProperties=getSuperProperties().iterator();
            while (superProperties.hasNext()) {
                Property property=(Property)superProperties.next();
                result.addAll(property.getAllRangeConcepts());
            }
            return result;
        }
    }
    /**
     * Checks whether supplied concept is in the range of this property.
     *
     * @param concept                   concept that is checked
     * @return                          <code>true</code> if supplied concept is in the range
     */
    public boolean isRangeConcept(Concept concept) throws KAONException {
        synchronized (getLock()) {
            return getRangeConcepts().contains(concept);
        }
    }
    /**
     * Returns the OI-model in which the supplied concept was added as the range to this property.
     *
     * @param concept                   concept that is checked
     * @return                          the OI-model in which supplied concept was added as the range to this property
     */
    public OIModel getRangeConceptOIModel(Concept concept) throws KAONException {
        synchronized (getLock()) {
            if (m_rangeConcepts==null)
                loadThisObject(OIModel.LOAD_PROPERTY_RANGES);
            return (OIModel)m_rangeConcepts.get(concept);
        }
    }
    /**
     * Returns <code>true</code> if supplied concept was added as the range to this property in
     * the model which returned by <code>getOIModel()</code>.
     *
     * @param concept                   the concept that is cheched
     * @return                          <code>true</code> if supplied concept was added to the range concepts wihtin the <code>getOIModel()</code> model
     */
    public boolean isRangeConceptDeclaredLocally(Concept concept) throws KAONException {
        synchronized (getLock()) {
            return getRangeConceptOIModel(concept)==m_oimodel;
        }
    }
    /**
     * Returns whether this property is an attribute.
     *
     * @return                          <code>true</code> if property is a literal
     */
    public boolean isAttribute() throws KAONException {
        synchronized (getLock()) {
            makeSureBasicsLoaded();
            return m_isAttribute;
        }
    }
    /**
     * Returns the set of direct subproperties of this property.
     *
     * @return                          set of subproperties (may be empty)
     */
    public Set getSubProperties() throws KAONException {
        synchronized (getLock()) {
            if (m_subProperties==null)
                loadThisObject(OIModel.LOAD_SUB_PROPERTIES);
            return m_subProperties;
        }
    }
    /**
     * Returns the set of all subproperties of this property.
     *
     * @return                          set of all subproperties (may be empty)
     */
    public Set getAllSubProperties() throws KAONException {
        synchronized (getLock()) {
            // if there are hierarcy cycles, this method will loop forever
            Set subProperties=getSubProperties();
            Set result=new HashSet(subProperties);
            Iterator iterator=subProperties.iterator();
            while (iterator.hasNext()) {
                Property property=(Property)iterator.next();
                result.addAll(property.getAllSubProperties());
            }
            return result;
        }
    }
    /**
     * Tests if this property is a direct subproperty of given property. If properties are the same, this method returns <code>false</code>.
     *
     * @param potentialSuperProperty    property being tested
     * @return                          <code>true</code> if this property is a direct subproperty
     */
    public boolean isDirectSubPropertyOf(Property potentialSuperProperty) throws KAONException {
        synchronized (getLock()) {
            return getSuperProperties().contains(potentialSuperProperty);
        }
    }
    /**
     * Returns the OI-model containing the superproperty - subproperty association.
     *
     * @param superProperty             the superproperty of this property
     * @return                          the OI-model in which the super-subproperty association is declared
     */
    public OIModel getSuperSubPropertyOIModel(Property superProperty) throws KAONException {
        synchronized (getLock()) {
            if (m_superProperties==null)
                loadThisObject(OIModel.LOAD_SUPER_PROPERTIES);
            return (OIModel)m_superProperties.get(superProperty);
        }
    }
    /**
     * Returns <code>true</code> if this property has been declared a subproperty of given property in this OI-model.
     *
     * @param superProperty             the subproperty of this property
     * @return                          <code>true</code> if supplied property has been declared a subproperty of this property in this OI-model
     */
    public boolean isSuperSubPropertyDeclaredLocally(Property superProperty) throws KAONException {
        synchronized (getLock()) {
            return getSuperSubPropertyOIModel(superProperty)==m_oimodel;
        }
    }
    /**
     * Tests if this property is a superproperty of given property. If properties are the same, this method returns <code>false</code>.
     *
     * @param potentialSuperProperty    property being tested
     * @return                          <code>true</code> if given property is a subproperty
     */
    public boolean isSubPropertyOf(Property potentialSuperProperty) throws KAONException {
        synchronized (getLock()) {
            return getAllSuperProperties().contains(potentialSuperProperty);
        }
    }
    /**
     * Returns the set of direct superproperties of this property.
     *
     * @return                          set of superproperties (may be empty)
     */
    public Set getSuperProperties() throws KAONException {
        synchronized (getLock()) {
            if (m_superProperties==null)
                loadThisObject(OIModel.LOAD_SUPER_PROPERTIES);
            return m_superProperties.keySet();
        }
    }
    /**
     * Returns the set of all superproperties of this property.
     *
     * @return                          set of all superproperties (may be empty)
     */
    public Set getAllSuperProperties() throws KAONException {
        synchronized (getLock()) {
            // if there are hierarcy cycles, this method will loop forever
            Set superProperties=getSuperProperties();
            Set result=new HashSet(superProperties);
            Iterator iterator=superProperties.iterator();
            while (iterator.hasNext()) {
                Property property=(Property)iterator.next();
                result.addAll(property.getAllSuperProperties());
            }
            return result;
        }
    }
    /**
     * Returns the set of instances of this property.
     *
     * @return                          set of <code>PropertyInstance</code> objects of this property
     */
    public Set getPropertyInstances() throws KAONException {
        synchronized (getLock()) {
            if (m_propertyInstances==null)
                loadThisObject(OIModel.LOAD_PROPERTY_INSTANCES);
            return m_propertyInstances;
        }
    }
    /**
     * Returns the instance associated with this object.
     *
     * @return                              instance associated with this object
     */
    public Instance getSpanningInstance() {
        synchronized (getLock()) {
            if (m_spanningInstance==null)
                m_spanningInstance=m_oimodel.getInstance(m_uri);
            return m_spanningInstance;
        }
    }
    /**
     * Makes sure that the basics of this object are loaded.
     */
    protected void makeSureBasicsLoaded() throws KAONException {
        if (!hasLoadedBasics())
            loadThisObject(OIModel.LOAD_PROPERTY_BASICS);
    }
    /**
     * Accepts an entity visitor.
     *
     * @param visitor                   visitor to apply
     */
    public void accept(EntityVisitor visitor) throws KAONException {
        visitor.visit(this);
    }
    /**
     * Sets the flag that determines whether this entity is in the model.
     *
     * @param isInOIModel               flag determining whether this entity is in the model
     * @param sourceOIModel             the source OI-model
     */
    void setIsInOIModel(boolean isInOIModel,OIModel sourceOIModel) throws KAONException {
        super.setIsInOIModel(isInOIModel,sourceOIModel);
        if (!isInOIModel) {
            m_inverseProperty=null;
            m_symmetric=false;
            m_transitive=false;
            m_isAttribute=false;
            if (m_domainConceptsAndCardinalitites!=null)
                m_domainConceptsAndCardinalitites.clear();
            else
                m_domainConceptsAndCardinalitites=new HashMap();
            if (m_rangeConcepts!=null)
                m_rangeConcepts.clear();
            else
                m_rangeConcepts=new HashMap();
            if (m_subProperties!=null)
                m_subProperties.clear();
            else
                m_subProperties=new HashSet();
            if (m_superProperties!=null)
                m_superProperties.clear();
            else
                m_superProperties=new HashMap();
            if (m_propertyInstances!=null)
                m_propertyInstances.clear();
            else
                m_propertyInstances=new HashSet();
        }
    }
    /**
     * Loads the basics of this object.
     *
     * @param propertyInfo              the information about object
     */
    void loadBasics(PropertyInfo propertyInfo) throws KAONException {
        if (!hasLoadedBasics()) {
            super.loadBasics(propertyInfo);
            if (propertyInfo.m_inversePropertyID!=null) {
                m_inverseProperty=m_oimodel.getProperty(propertyInfo.m_inversePropertyID);
                m_invesePropertyOIModel=m_oimodel.getOIModel(propertyInfo.m_inversePropertyID.m_sourceOIModelID);
            }
            m_symmetric=propertyInfo.m_symmetric;
            m_transitive=propertyInfo.m_transitive;
            m_isAttribute=propertyInfo.m_isAttribute;
        }
    }
    /**
     * Loads the domain map of the supplied info object.
     *
     * @param domainInfos               array of domain objects
     */
    Map loadDomainConceptsAndCardinalities(DomainInfo[] domainInfos) throws KAONException {
        Map map=new HashMap();
        for (int i=0;i<domainInfos.length;i++) {
            DomainInfo domainInfo=domainInfos[i];
            Concept concept=m_oimodel.getConcept(domainInfo.m_conceptID);
            OIModel sourceOIModel=m_oimodel.getOIModel(domainInfo.m_conceptID.m_sourceOIModelID);
            Cardinality cardinality=new Cardinality(domainInfo.m_minimumCardinality,domainInfo.m_maximumCardinality,sourceOIModel);
            map.put(concept,cardinality);
        }
        return map;
    }
    /**
     * Loads this object.
     *
     * @param propertyInfo              the information about a property
     */
    void loadObject(PropertyInfo propertyInfo) throws KAONException {
        checkVersions(propertyInfo.m_id);
        if (propertyInfo.m_modelID!=-1)
            loadBasics(propertyInfo);
        if (propertyInfo.m_domainInfos!=null && m_domainConceptsAndCardinalitites==null)
            m_domainConceptsAndCardinalitites=loadDomainConceptsAndCardinalities(propertyInfo.m_domainInfos);
        if (propertyInfo.m_rangeConceptIDs!=null && m_rangeConcepts==null)
            m_rangeConcepts=loadConcepts(propertyInfo.m_rangeConceptIDs);
        if (propertyInfo.m_superPropertyIDs!=null && m_superProperties==null)
            m_superProperties=loadProperties(propertyInfo.m_superPropertyIDs);
        if (propertyInfo.m_subPropertyIDs!=null && m_subProperties==null)
            m_subProperties=loadProperties(propertyInfo.m_subPropertyIDs);
        if (propertyInfo.m_propertyInstances!=null && m_propertyInstances==null)
            m_propertyInstances=loadPropertyInstances(propertyInfo.m_propertyInstances);
    }
    /**
     * Loads the set of property instances specified by the pairs of IDs.
     *
     * @param propertyInstances         the array of ID pairs
     */
    Set loadPropertyInstances(Object[][] propertyInstances) throws KAONException {
        Set set=new HashSet();
        for (int i=0;i<propertyInstances.length;i++) {
            RelationEntityID sourceInstanceID=(RelationEntityID)propertyInstances[i][0];
            Instance sourceInstance=m_oimodel.getInstance(sourceInstanceID);
            Object value=propertyInstances[i][1];
            Object targetValue;
            if (value instanceof EntityID)
                targetValue=m_oimodel.getInstance((EntityID)value);
            else
                targetValue=value;
            PropertyInstanceProxy propertyInstance=(PropertyInstanceProxy)m_oimodel.getPropertyInstance(this,sourceInstance,targetValue);
            propertyInstance.setSourceOIModel(m_oimodel.getOIModel(sourceInstanceID.m_sourceOIModelID));
            set.add(propertyInstance);
        }
        return set;
    }
    /**
     * Sets the inverse property of this property.
     *
     * @param inverseProperty           inverse property of this property
     * @param inversePropertyOIModel    the OI-model where properties are make inverse
     */
    void setInverseProperty(Property inverseProperty,OIModel inversePropertyOIModel) {
        m_inverseProperty=inverseProperty;
        m_invesePropertyOIModel=inversePropertyOIModel;
    }
    /**
     * Sets the symmetric flag of this property.
     *
     * @param symmetric                 symmetric flag of this property
     */
    void setSymmetric(boolean symmetric) throws KAONException {
        makeSureBasicsLoaded();
        m_symmetric=symmetric;
    }
    /**
     * Sets the transitive flag of this property.
     *
     * @param transitive                transitive flag of this property
     */
    void setTransitive(boolean transitive) throws KAONException {
        makeSureBasicsLoaded();
        m_transitive=transitive;
    }
    /**
     * Sets the attribute flag of this property.
     *
     * @param isAttribute               attribute flag of this property
     */
    void setIsAttribute(boolean isAttribute) {
        m_isAttribute=isAttribute;
    }
    /**
     * Adds a domain concept to the property.
     *
     * @param concept                   concept to add to the domain
     * @param sourceOIModel             the OI-model where this domain concept has been declared
     */
    void addDomainConcept(Concept concept,OIModel sourceOIModel) {
        if (m_domainConceptsAndCardinalitites!=null)
            m_domainConceptsAndCardinalitites.put(concept,new Cardinality(0,Integer.MAX_VALUE,sourceOIModel));
    }
    /**
     * Removes a domain concept from the property.
     *
     * @param concept                   concept to remove from the domain
     */
    void removeDomainConcept(Concept concept) {
        if (m_domainConceptsAndCardinalitites!=null)
            m_domainConceptsAndCardinalitites.remove(concept);
    }
    /**
     * Sets the minimum cardinality for the property.
     *
     * @param concept                   concept for which the cardinality is set
     * @param value                     cardinality
     */
    void setMinimumCardinality(Concept concept,int value) throws KAONException {
        if (m_domainConceptsAndCardinalitites==null)
            loadThisObject(OIModel.LOAD_PROPERTY_DOMAINS);
        Cardinality cardinality=(Cardinality)m_domainConceptsAndCardinalitites.get(concept);
        if (cardinality==null)
            throw new KAONException("Concept '"+concept.getURI()+"' is not in the domain of property '"+getURI());
        cardinality.m_minimum=value;
    }
    /**
     * Sets the maximum cardinality for the property.
     *
     * @param concept                   concept for which the cardinality is set
     * @param value                     cardinality
     */
    void setMaximumCardinality(Concept concept,int value) throws KAONException {
        if (m_domainConceptsAndCardinalitites==null)
            loadThisObject(OIModel.LOAD_PROPERTY_DOMAINS);
        Cardinality cardinality=(Cardinality)m_domainConceptsAndCardinalitites.get(concept);
        if (cardinality==null)
            throw new KAONException("Concept '"+concept.getURI()+"' is not in the domain of property '"+getURI());
        cardinality.m_maximum=value;
    }
    /**
     * Adds a range concept to the property.
     *
     * @param concept                   concept to add to the range
     * @param sourceOIModel             the OI-model where this range has been declared
     */
    void addRangeConcept(Concept concept,OIModel sourceOIModel) {
        if (m_rangeConcepts!=null)
            m_rangeConcepts.put(concept,sourceOIModel);
    }
    /**
     * Removes a range concept from the property.
     *
     * @param concept                   concept to remove from the range
     */
    void removeRangeConcept(Concept concept) {
        if (m_rangeConcepts!=null)
            m_rangeConcepts.remove(concept);
    }
    /**
     * Adds the superproperty to this property.
     *
     * @param superProperty             the superproperty to add
     * @param sourceOIModel             the OI-model where this relationship has been declared
     */
    void addSuperProperty(Property superProperty,OIModel sourceOIModel) {
        if (m_superProperties!=null)
            m_superProperties.put(superProperty,sourceOIModel);
    }
    /**
     * Removes the superproperty from this property.
     *
     * @param superProperty             the superproperty to remove
     */
    void removeSuperProperty(Property superProperty) {
        if (m_superProperties!=null)
            m_superProperties.remove(superProperty);
    }
    /**
     * Adds the subproperty to this property.
     *
     * @param subProperty               the subproperty to add
     */
    void addSubProperty(Property subProperty) {
        if (m_subProperties!=null)
            m_subProperties.add(subProperty);
    }
    /**
     * Removes the subproperty from this property.
     *
     * @param subProperty               the sunproperty to remove
     */
    void removeSubProperty(Property subProperty) {
        if (m_subProperties!=null)
            m_subProperties.remove(subProperty);
    }
    /**
     * Adds the property instance to this property.
     *
     * @param propertyInstance          the property instance to add
     */
    void addPropertyInstance(PropertyInstance propertyInstance) {
        if (m_propertyInstances!=null)
            m_propertyInstances.add(propertyInstance);
    }
    /**
     * Removes the property instance from this property.
     *
     * @param propertyInstance          the property instance to remove
     */
    void removePropertyInstance(PropertyInstance propertyInstance) {
        if (m_propertyInstances!=null)
            m_propertyInstances.remove(propertyInstance);
    }

    /**
     * Structure tracking cardinalities.
     */
    protected static class Cardinality {
        public int m_minimum;
        public int m_maximum;
        public OIModel m_sourceOIModel;

        public Cardinality(int minimum,int maximum,OIModel sourceOIModel) {
            m_minimum=minimum;
            m_maximum=maximum;
            m_sourceOIModel=sourceOIModel;
        }
    }
}
