package edu.unika.aifb.kaon.apiproxy;

import java.util.Set;
import java.util.HashSet;
import java.util.List;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.Map;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Collection;
import java.net.URLEncoder;
import java.io.Serializable;
import java.io.UnsupportedEncodingException;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.change.*;
import edu.unika.aifb.kaon.api.vocabulary.*;
import edu.unika.aifb.kaon.api.util.*;

import edu.unika.aifb.kaon.apiproxy.source.*;

/**
 * Implementation an OI-model based on the W3C RDF API model implementation.
 */
public class OIModelProxy implements OIModel,Serializable {
    /** The servial version. */
    public static final long serialVersionUID=-2086896920898952620L;
    /** The load flag for loading the lexicon. */
    protected static final int LOAD_LEXICON_FLAG=LOAD_CONCEPT_BASICS | LOAD_PROPERTY_BASICS | LOAD_INSTANCE_BASICS | LOAD_INSTANCE_PARENT_CONCEPTS | LOAD_INSTANCE_FROM_PROPERTY_VALUES | LOAD_INSTANCE_TO_PROPERTY_VALUES;

    /** The logical URI of the model. */
    protected String m_logicalURI;
    /** The physical URI of the model. */
    protected String m_physicalURI;
    /** KAON connection that created this OI-model. */
    transient protected KAONConnectionProxy m_kaonConnection;
    /** Set of included OI-models. */
    transient protected Set m_includedOIModels;
    /** The set of all OI-models that include this model. */
    transient protected Set m_allIncludedByOIModels;
    /** Event manager for this OI-model. */
    transient protected OIModelEventManager m_eventManager;
    /** The source for this OI-model. */
    transient protected OIModelSource m_oimodelSource;
    /** The visitor for events of included OI-models. */
    transient protected IncludedOIModelEventCopier m_includedOIModelEventCopier;
    /** Map of concept objects. */
    transient protected Map m_concepts;
    /** Map of property objects. */
    transient protected Map m_properties;
    /** Map of lexical entries. */
    transient protected Map m_lexicalEntries;
    /** Map of property instance objects. */
    transient protected Map m_propertyInstances;
   /** List of strong references to preloaded objects. */
    transient protected Set m_preloadedObjectStrongReferences;
    /** Set to <code>true</code> if preloading has been invoked on this model already. */
    transient protected boolean m_preloadingInvoked;
    /** The bookmark while model is being updated. */
    transient protected Object m_bookmark;
    /** The event processor. */
    transient protected EventProcessor m_eventProcessor;
    /** The set of object currently being loaded in this OI-model. */
    transient protected Set m_objectsBeingLoaded;

    /**
     * Creates an instance of this class and attaches it to an RDF model.
     *
     * @param kaonConnection            the connection that created this model
     * @param oimodelSource             the source for this OI-model
     * @param includedOIModels          array of included OI-models
     */
    public OIModelProxy(KAONConnectionProxy kaonConnection,OIModelSource oimodelSource,OIModel[] includedOIModels) throws KAONException {
        m_kaonConnection=kaonConnection;
        m_oimodelSource=oimodelSource;
        m_logicalURI=m_oimodelSource.getLogicalURI();
        m_physicalURI=m_oimodelSource.getPhysicalURI();
        if (m_logicalURI==null)
            throw new KAONException("OI-model doesn't have a logical URI.");
        m_eventManager=new OIModelEventManager(this);
        m_includedOIModelEventCopier=new IncludedOIModelEventCopier();
        m_concepts=new WeakValueHashMap();
        m_properties=new WeakValueHashMap();
        m_lexicalEntries=new WeakValueHashMap();
        m_propertyInstances=new WeakValueHashMap();
        m_includedOIModels=new HashSet();
        m_allIncludedByOIModels=new HashSet();
        for (int i=0;i<includedOIModels.length;i++)
            addIncludedOIModel(includedOIModels[i]);
        if (!KAONConnection.ROOT_OIMODEL_URI.equals(getLogicalURI()) && getIncludedOIModel(KAONConnection.ROOT_OIMODEL_URI)==null)
            throw new KAONException("Each OI-model must include the '"+KAONConnection.ROOT_OIMODEL_URI+"' OI-model.");
        m_preloadedObjectStrongReferences=new HashSet();
        m_eventProcessor=new EventProcessor(this);
        m_objectsBeingLoaded=new HashSet();
    }
    /**
     * Loads some well-known objects that are typically always needed in an OI-model.
     */
    protected void preloadObjects() throws KAONException {
        synchronized (getLock()) {
            m_preloadedObjectStrongReferences.clear();
            List list=list=new LinkedList();
            list.add(getConcept(KAONVocabularyAdaptor.INSTANCE.getRoot()));
            loadObjectsInternal(list,LOAD_CONCEPT_BASICS | LOAD_SUPER_CONCEPTS | LOAD_PROPERTIES_FROM | LOAD_PROPERTIES_TO | LOAD_PROPERTY_BASICS | LOAD_INSTANCE_BASICS,m_preloadedObjectStrongReferences);
            if (getIncludedOIModel(KAONConnection.LEXICAL_OIMODEL_URI)!=null || KAONConnection.LEXICAL_OIMODEL_URI.equals(getLogicalURI())) {
                list.clear();
                list.add(getConcept(KAONVocabularyAdaptor.INSTANCE.getLanguage()));
                list.add(getConcept(KAONVocabularyAdaptor.INSTANCE.getLexicalEntry()));
                list.add(getConcept(KAONVocabularyAdaptor.INSTANCE.getKAONLabel()));
                list.add(getConcept(KAONVocabularyAdaptor.INSTANCE.getStem()));
                list.add(getConcept(KAONVocabularyAdaptor.INSTANCE.getDocumentation()));
                list.add(getConcept(KAONVocabularyAdaptor.INSTANCE.getSynonym()));
                list.add(getProperty(KAONVocabularyAdaptor.INSTANCE.getInLanguage()));
                list.add(getProperty(KAONVocabularyAdaptor.INSTANCE.getValue()));
                list.add(getProperty(KAONVocabularyAdaptor.INSTANCE.getReferences()));
                loadObjectsInternal(list,LOAD_CONCEPT_BASICS | LOAD_SUPER_CONCEPTS | LOAD_SUB_CONCEPTS | LOAD_PROPERTIES_FROM | LOAD_PROPERTIES_TO | LOAD_PROPERTY_BASICS | LOAD_PROPERTY_DOMAINS | LOAD_PROPERTY_RANGES | LOAD_INSTANCE_BASICS,m_preloadedObjectStrongReferences);
            }
        }
        m_preloadingInvoked=true;
    }
    /**
     * Returns the capabilities of the model.
     *
     * @return                          the bit-mask defining the model's capaibilities
     */
    public int getCapabilities() {
        return CAPABILITY_ALWAYS_SAVED | CAPABILITY_SUPPORTS_NOTIFICATIONS | CAPABILITY_SUPPORTS_OPTIMIZED_LOADING;
    }
    /**
     * Makes given OI-model included into this OI-model.
     *
     * @param oimodel                   the OI-model
     */
    public void addIncludedOIModel(OIModel oimodel) throws KAONException {
        synchronized (getLock()) {
            if (!(oimodel instanceof OIModelProxy))
                throw new KAONException("Suppled model is not a proxy model.");
            if (!m_includedOIModels.contains(oimodel)) {
                m_includedOIModels.add(oimodel);
                List queue=new LinkedList();
                queue.add(oimodel);
                while (!queue.isEmpty()) {
                    OIModelProxy someOIModel=(OIModelProxy)queue.remove(0);
                    someOIModel.m_allIncludedByOIModels.add(this);
                    someOIModel.m_allIncludedByOIModels.addAll(m_allIncludedByOIModels);
                    queue.addAll(someOIModel.m_includedOIModels);
                }
            }
        }
    }
    /**
     * Makes given OI-model not included into this OI-model.
     *
     * @param oimodel                   the OI-model
     */
    public void removeIncludedOIModel(OIModel oimodel) throws KAONException {
        synchronized (getLock()) {
            if (!(oimodel instanceof OIModelProxy))
                throw new KAONException("Suppled model is not a proxy model.");
            if (m_includedOIModels.contains(oimodel)) {
                OIModelProxy includedOIModel=(OIModelProxy)oimodel;
                m_includedOIModels.remove(includedOIModel);
                // first compute the set of models that will still be the parents of the included model
                Set remainingModels=new HashSet();
                Iterator iterator=m_kaonConnection.getOpenOIModels().iterator();
                while (iterator.hasNext()) {
                    OIModelProxy someOIModel=(OIModelProxy)iterator.next();
                    if (someOIModel.m_includedOIModels.contains(includedOIModel)) {
                        remainingModels.add(someOIModel);
                        remainingModels.addAll(someOIModel.m_allIncludedByOIModels);
                    }
                }
                // now compute the models that are to be removed (this is the difference between includedOIModel.m_allIncludedByOIModels and models)
                Set modelsToRemove=new HashSet(includedOIModel.m_allIncludedByOIModels);
                modelsToRemove.removeAll(remainingModels);
                // now subtract these models from all submodels
                List queue=new LinkedList();
                queue.add(includedOIModel);
                while (!queue.isEmpty()) {
                    OIModelProxy someOIModel=(OIModelProxy)queue.remove(0);
                    someOIModel.m_allIncludedByOIModels.removeAll(modelsToRemove);
                    queue.addAll(someOIModel.m_includedOIModels);
                }
            }
        }
    }
    /**
     * Returns the KAON connection of this OI-model.
     *
     * @return                          KAON connection of this OI-model
     */
    public KAONConnection getKAONConnection() {
        return m_kaonConnection;
    }
    /**
     * Returns the logical URI of this OI-model.
     *
     * @return                          logical URI of this OI-model
     */
    public String getLogicalURI() {
        return m_logicalURI;
    }
    /**
     * Returns the physical URI of this OI-model.
     *
     * @return                          physical URI of this OI-model
     */
    public String getPhysicalURI() {
        return m_physicalURI;
    }
    /**
     * Returns the set of included OI-models.
     *
     * @return                          set of included OI-models
     */
    public Set getIncludedOIModels() {
        return m_includedOIModels;
    }
    /**
     * Returns the set of models that include this model.
     *
     * @return                          set of OI-models that include this model
     */
    public Set getAllIncludedByOIModels() {
        return m_allIncludedByOIModels;
    }
    /**
     * Returns an included OI-model with given URI.
     *
     * @param logicalURI                logical URI of the requested OI-model
     * @return                          OI-model with given logical URI or <code>null</code> if OI-model with this logical URI wasn't included
     */
    public OIModel getIncludedOIModel(String logicalURI) throws KAONException {
        synchronized (getLock()) {
            if (logicalURI.equals(getLogicalURI()))
                return this;
            Iterator iterator=m_includedOIModels.iterator();
            while (iterator.hasNext()) {
                OIModel oimodel=(OIModel)iterator.next();
                OIModel includedModel=oimodel.getIncludedOIModel(logicalURI);
                if (includedModel!=null)
                    return includedModel;
            }
            return null;
        }
    }
    /**
     * Adds a listener to this OI-model.
     *
     * @param listener                  listener to be added
     */
    public void addOIModelListener(OIModelListener listener) {
        synchronized (getLock()) {
            m_eventManager.addOIModelListener(listener);
        }
    }
    /**
     * Removes a listener from this OI-model.
     *
     * @param listener                  listener to be removed
     */
    public void removeOIModelListener(OIModelListener listener) {
        synchronized (getLock()) {
            m_eventManager.removeOIModelListener(listener);
        }
    }
    /**
     * Suspends entity pool events until {@link #resumeEvents} is called.
     */
    public void suspendEvents() {
        synchronized (getLock()) {
            m_eventManager.suspendEvents();
        }
    }
    /**
     * Resumes entity pool events.
     */
    public void resumeEvents() {
        synchronized (getLock()) {
            m_eventManager.resumeEvents();
        }
    }
    /**
     * Processes changes in the change list.
     *
     * @param changeList                list of changes to the model
     */
    public void applyChanges(List changeList) throws KAONException {
        synchronized (getLock()) {
            Iterator iterator=changeList.iterator();
            while (iterator.hasNext()) {
                ChangeEvent changeEvent=(ChangeEvent)iterator.next();
                changeEvent.setOIModel(this);
            }
            m_kaonConnection.applyChanges(changeList);
        }
    }
    /**
     * Loads objects in the collection.
     *
     * @param objects                   objects to be loaded
     * @param loadFlag                  the flag specifying what to load
     */
    public void loadObjects(Collection objects,int loadFlag) throws KAONException {
        synchronized (getLock()) {
            if (objects.isEmpty())
                return;
            if (!m_preloadingInvoked)
                preloadObjects();
            // This is important! It may happen that a label is loaded before the object to which the label belongs. If there
            // are many objects, then the label may be garbage-collected before the object is loaded. Hence, we make sure that
            // all objects stay in the memory until loading is complete by placing them in a list. After loading is done,
            // the objects parameter collection will hold the strong references to loaded objects, and these will in turn hold
            // strong references to labels.
            Set loadedObjects=new HashSet();
            loadObjectsInternal(objects,(loadFlag & LOAD_LEXICON)!=0 ? (loadFlag | LOAD_INSTANCE_TO_PROPERTY_VALUES) & ~LOAD_LEXICON : loadFlag,loadedObjects);
            if ((loadFlag & LOAD_LEXICON)!=0) {
                Set lexicalEntriesToLoad=new HashSet();
                Iterator iterator=objects.iterator();
                while (iterator.hasNext()) {
                    Entity entity=(Entity)iterator.next();
                    lexicalEntriesToLoad.addAll(entity.getLexicalEntries());
                }
                loadObjectsInternal(lexicalEntriesToLoad,LOAD_LEXICON_FLAG,new HashSet());
            }
        }
    }
    /**
     * Loads objects in the collection.
     *
     * @param objects                   objects to be loaded
     * @param loadedObjects             the set receiving loaded objects
     * @param loadFlag                  the flag specifying what to load
     */
    protected void loadObjectsInternal(Collection objects,int loadFlag,Set loadedObjects) throws KAONException {
        if (!objects.isEmpty()) {
            while (someObjectsBeingLoaded(objects))
                try {
                    getLock().wait();
                }
                catch (InterruptedException e) {
                    throw new KAONException("Load interrupted.",e);
                }
            int conceptLoadFlag=loadFlag & LOAD_CONCEPT_ALL;
            int propertyLoadFlag=loadFlag & LOAD_PROPERTY_ALL;
            int instanceLoadFlag=loadFlag & LOAD_INSTANCE_ALL;
            Set objectsToLoad=new HashSet();
            Iterator iterator=objects.iterator();
            while (iterator.hasNext()) {
                Entity entity=(Entity)iterator.next();
                if (entity instanceof Concept) {
                    if ((entity.getLoadedState() & conceptLoadFlag)!=conceptLoadFlag)
                        objectsToLoad.add(entity);
                }
                else if (entity instanceof Property) {
                    if ((entity.getLoadedState() & propertyLoadFlag)!=propertyLoadFlag)
                        objectsToLoad.add(entity);
                }
                else if (entity instanceof Instance) {
                    if ((entity.getLoadedState() & instanceLoadFlag)!=instanceLoadFlag)
                        objectsToLoad.add(entity);
                }
            }
            if (!objectsToLoad.isEmpty()) {
                m_objectsBeingLoaded.addAll(objectsToLoad);
                try {
                    EntityInfo[] entityInfos=m_kaonConnection.loadEntities(m_oimodelSource,getEntityURIs(objectsToLoad),loadFlag);
                    processEntityInfos(entityInfos,objectsToLoad,loadedObjects);
                }
                finally {
                    m_objectsBeingLoaded.removeAll(objectsToLoad);
                    getLock().notifyAll();
                }
            }
        }
    }
    /**
     * Returns <code>true</code> if some objects are being loaded.
     *
     * @param objects                   the collection of objects
     * @return                          <code>true</code> if at least one of the supplied objects is being loaded
     */
    protected boolean someObjectsBeingLoaded(Collection objects) {
        Iterator iterator=objects.iterator();
        while (iterator.hasNext())
            if (m_objectsBeingLoaded.contains(iterator.next()))
                return true;
        return false;
    }
    /**
     * Returns the array of entity URIs.
     *
     * @param objects                   the collection of entities
     * @return                          the array of entity URIs
     */
    protected String[] getEntityURIs(Collection objects) throws KAONException {
        Set entityURIs=new HashSet();
        Iterator iterator=objects.iterator();
        while (iterator.hasNext()) {
            Entity entity=(Entity)iterator.next();
            entityURIs.add(entity.getURI());
        }
        String[] entityURIsArray=new String[entityURIs.size()];
        entityURIs.toArray(entityURIsArray);
        return entityURIsArray;
    }
    /**
     * Processes loaded entity infos.
     *
     * @param entityInfos               the loaded entity infos
     * @param objects                   the objects to load
     * @param loadedObjects             the set receiving loaded objects
     */
    protected void processEntityInfos(EntityInfo[] entityInfos,Collection objects,Set loadedObjects) throws KAONException {
        for (int i=0;i<entityInfos.length;i++) {
            EntityInfo entityInfo=entityInfos[i];
            if (entityInfo instanceof ConceptInfo) {
                ConceptProxy concept=getConcept(entityInfo.m_id);
                loadedObjects.add(concept);
                concept.loadObject((ConceptInfo)entityInfo);
                // add a strong reference to the spanning instance if it has been loaded
                if (m_lexicalEntries.containsKey(concept.getURI()))
                    concept.getSpanningInstance().getSpanningConcept();
            }
            else if (entityInfo instanceof PropertyInfo) {
                PropertyProxy property=getProperty(entityInfo.m_id);
                loadedObjects.add(property);
                property.loadObject((PropertyInfo)entityInfo);
                // add a strong reference to the spanning instance if it has been loaded
                if (m_lexicalEntries.containsKey(property.getURI()))
                    property.getSpanningInstance().getSpanningProperty();
            }
            else if (entityInfo instanceof InstanceInfo) {
                InstanceProxy instance=getInstance(entityInfo.m_id);
                loadedObjects.add(instance);
                instance.loadObject((InstanceInfo)entityInfo);
                // add a strong reference to the spanning concept if it has been loaded
                if (m_concepts.containsKey(instance.getURI()))
                    instance.getSpanningConcept().getSpanningInstance();
                // add a strong reference to the spanning property if it has been loaded
                if (m_properties.containsKey(instance.getURI()))
                    instance.getSpanningProperty().getSpanningInstance();
            }
        }
        // objects that had to be loaded, but for which no data arrived either didn't exist ever, or they were deleted
        Iterator iterator=objects.iterator();
        while (iterator.hasNext()) {
            AbstractEntityProxy entity=(AbstractEntityProxy)iterator.next();
            if (!loadedObjects.contains(entity))
                entity.loadRequestedButNoDataReceived();
        }
    }
    /**
     * Executes a query and returns the set of instances or pairs matching given query.
     *
     * @param queryString               the query to be executed
     * @return                          the collection of result objects
     */
    public Collection executeQuery(String queryString) throws KAONException {
        Object[] result=m_oimodelSource.executeQuery(queryString);
        synchronized (getLock()) {
            List list=new ArrayList(result.length);
            for (int i=0;i<result.length;i++) {
                Object object=result[i];
                if (object instanceof EntityID)
                    list.add(getInstance((EntityID)object));
                else if (object instanceof Object[]) {
                    Object[] tuple=(Object[])object;
                    if (tuple[0] instanceof EntityID)
                        tuple[0]=getInstance((EntityID)tuple[0]);
                    if (tuple[1] instanceof EntityID)
                        tuple[1]=getInstance((EntityID)tuple[1]);
                    list.add(tuple);
                }
                else
                    list.add(object);
            }
            return list;
        }
    }
    /**
     * Returns the underlying OI-model source.
     *
     * @return                          underlying OI-model source
     */
    public OIModelSource getOIModelSource() {
        return m_oimodelSource;
    }
    /**
     * Returns a concept with given URI.
     *
     * @param uri                       URI of the requested concept
     * @return                          concept with given URI
     */
    public Concept getConcept(String uri) {
        synchronized (getLock()) {
            Concept concept=(Concept)m_concepts.get(uri);
            if (concept==null) {
                concept=new ConceptProxy(this,uri);
                m_concepts.put(uri,concept);
            }
            return concept;
        }
    }
    /**
     * Returns a concept with given EntityID.
     *
     * @param entityID                  the ID of the entity
     * @return                          concept with given ID
     */
    public ConceptProxy getConcept(EntityID entityID) throws KAONException {
        ConceptProxy concept=(ConceptProxy)getConcept(entityID.m_uri);
        concept.checkVersions(entityID);
        return concept;
    }
    /**
     * Returns the root concepts of the ontology. Root concepts are concepts that are not subclasses of any other concept.
     *
     * @return                          Set of root concepts (may be empty)
     */
    public Concept getRootConcept() {
        return getConcept(KAONVocabularyAdaptor.INSTANCE.getRoot());
    }
    /**
     * Returns all concepts of this ontology.
     *
     * @return                          array of all concepts in this ontology
     */
    public Set getConcepts() throws KAONException {
        synchronized (getLock()) {
            Set set=new HashSet();
            EntityID[] entityIDs=m_oimodelSource.getConcepts();
            for (int i=0;i<entityIDs.length;i++) {
                Concept concept=getConcept(entityIDs[i]);
                set.add(concept);
            }
            return set;
        }
    }
    /**
     * Returns a property with given URI.
     *
     * @param uri                       URI of the property
     * @return                          property with given URI
     */
    public Property getProperty(String uri) {
        synchronized (getLock()) {
            Property property=(Property)m_properties.get(uri);
            if (property==null) {
                property=new PropertyProxy(this,uri);
                m_properties.put(uri,property);
            }
            return property;
        }
    }
    /**
     * Returns a property with given EntityID.
     *
     * @param entityID                  the ID of the entity
     * @return                          property with given ID
     */
    public PropertyProxy getProperty(EntityID entityID) throws KAONException {
        PropertyProxy property=(PropertyProxy)getProperty(entityID.m_uri);
        property.checkVersions(entityID);
        return property;
    }
    /**
     * Returns all properties of this ontology.
     *
     * @return                          set of all relations of this ontology (may be empty)
     */
    public Set getProperties() throws KAONException {
        synchronized (getLock()) {
            Set set=new HashSet();
            EntityID[] entityIDs=m_oimodelSource.getProperties();
            for (int i=0;i<entityIDs.length;i++) {
                Property property=getProperty(entityIDs[i]);
                set.add(property);
            }
            return set;
        }
    }
    /**
     * Returns an instance with given URI.
     *
     * @param uri                       URI of the requested instance
     * @return                          instance with given URI
     */
    public Instance getInstance(String uri) {
        return getLexicalEntry(uri);
    }
    /**
     * Returns an instance with given EntityID.
     *
     * @param entityID                  the ID of the entity
     * @return                          instance with given ID
     */
    public InstanceProxy getInstance(EntityID entityID) throws KAONException {
        return (InstanceProxy)getLexicalEntry(entityID);
    }
    /**
     * Returns all instances of this ontology.
     *
     * @return                          array of all instances in this ontology
     */
    public Set getInstances() throws KAONException {
        synchronized (getLock()) {
            Set set=new HashSet();
            EntityID[] entityIDs=m_oimodelSource.getInstances();
            for (int i=0;i<entityIDs.length;i++) {
                Instance instance=getInstance(entityIDs[i]);
                set.add(instance);
            }
            return set;
        }
    }
    /**
     * Returns a lexical entry with given URI.
     *
     * @param uri                       URI of the lexical entry
     * @return                          lexical entry with given URI
     */
    public LexicalEntry getLexicalEntry(String uri) {
        synchronized (getLock()) {
            LexicalEntry lexicalEntry=(LexicalEntry)m_lexicalEntries.get(uri);
            if (lexicalEntry==null) {
                lexicalEntry=new LexicalEntryProxy(this,uri);
                m_lexicalEntries.put(uri,lexicalEntry);
            }
            return lexicalEntry;
        }
    }
    /**
     * Returns a lexical entry with given EntityID.
     *
     * @param entityID                  the ID of the entity
     * @return                          lexical entry with given ID
     */
    public LexicalEntryProxy getLexicalEntry(EntityID entityID) throws KAONException {
        LexicalEntryProxy lexicalEntry=(LexicalEntryProxy)getLexicalEntry(entityID.m_uri);
        lexicalEntry.checkVersions(entityID);
        return lexicalEntry;
    }
    /**
     * Factory method for property instance objects.
     *
     * @param property                  property
     * @param sourceInstance            source instance
     * @param targetValue               target value (<code>Instance</code> or a <code>String</code>)
     */
    public PropertyInstance getPropertyInstance(Property property,Instance sourceInstance,Object targetValue) throws KAONException {
        synchronized (getLock()) {
            String key;
            if (targetValue instanceof Instance)
                key=property.getURI()+"$"+sourceInstance.getURI()+"$"+((Instance)targetValue).getURI();
            else
                key=property.getURI()+"$"+sourceInstance.getURI()+"$"+targetValue.toString();
            PropertyInstance propertyInstance=(PropertyInstance)m_propertyInstances.get(key);
            if (propertyInstance==null) {
                propertyInstance=new PropertyInstanceProxy((PropertyProxy)property,(InstanceProxy)sourceInstance,targetValue);
                m_propertyInstances.put(key,propertyInstance);
            }
            return propertyInstance;
        }
    }
    /**
     * Creates a new URI unique wihtin this OI-model.
     *
     * @return                          an new URI unique within this OI-model
     */
    public String createNewURI() {
        return UniqueURIGenerator.getUniqueURI(getLogicalURI());
    }
    /**
     * Creates an URI by combining the logical URI with the given suffix.
     *
     * @param suffix                    the suffix
     * @return                          a new URI with given suffix (may not be unique)
     */
    public String getURI(String suffix) throws KAONException {
        suffix=suffix.replace(' ','-');
        try {
            suffix=URLEncoder.encode(suffix,"UTF-8");
        }
        catch (UnsupportedEncodingException shouldntHappen) {
            throw new KAONException(shouldntHappen);
        }
        String uriPrefix=getLogicalURI();
        if (!uriPrefix.endsWith("#"))
            uriPrefix+="#";
        return uriPrefix+suffix;
    }
    /**
     * Makes sure that all data of this ontology is saved. This mehtod is a no-op, since
     * this ontology type is always saved.
     */
    public void save() {
    }
    /**
     * Returns <code>true</code> if the OI-model has been changed since the last save.
     *
     * @return                          <code>true</code> if the OI-model has been changed since the last save
     */
    public boolean hasUnsavedChanges() {
        return false;
    }
    /**
     * Deletes this OI-model.
     */
    public void delete() throws KAONException {
        synchronized (getLock()) {
            m_oimodelSource.deleteOIModel();
            deleteNotify();
        }
    }
    /**
     * Notifies this OI-model that it was deleted.
     */
    public void deleteNotify() throws KAONException {
        synchronized (getLock()) {
            m_kaonConnection.notifyOIModelDeleted(this);
            Iterator iterator=m_kaonConnection.getOpenOIModels().iterator();
            while (iterator.hasNext()) {
                OIModelProxy someOIModel=(OIModelProxy)iterator.next();
                if (someOIModel.m_includedOIModels.contains(this))
                    someOIModel.removeIncludedOIModel(this);
            }
            m_eventManager.notifyDelete();
        }
    }
    /**
     * Refreshes the contents of this OI-model.
     */
    public void refresh() {
        synchronized (getLock()) {
            unloadObjects(m_concepts.values());
            unloadObjects(m_properties.values());
            unloadObjects(m_lexicalEntries.values());
            m_preloadingInvoked=false;
            m_eventManager.notifyRefresh();
        }
    }
    /**
     * Returns the value of the OI-model attribute with given key.
     *
     * @param key                       the key
     * @return                          the value (or <code>null</code> if there is no attribute with given key)
     */
    public String getAttribute(String key) throws KAONException {
        synchronized (getLock()) {
            return (String)getAttributes().get(key);
        }
    }
    /**
     * Sets the value of the OI-model attribute with given key.
     *
     * @param key                       the key
     * @param value                     the value (or <code>null</code> if the attribute should be deleted)
     */
    public void setAttribute(String key,String value) throws KAONException {
        synchronized (getLock()) {
            m_oimodelSource.setAttribute(key,value);
        }
    }
    /**
     * Returns the map of all key-value pairs of this OI-model.
     *
     * @return                          the map of key-value keys
     */
    public Map getAttributes() throws KAONException {
        synchronized (getLock()) {
            return m_oimodelSource.getAttributes();
        }
    }
    /**
     * Unloads a set of objects.
     *
     * @param collection                the collection of objects
     */
    protected void unloadObjects(Collection collection) {
        Iterator iterator=collection.iterator();
        while (iterator.hasNext()) {
            AbstractEntityProxy entityProxy=(AbstractEntityProxy)iterator.next();
            entityProxy.unload();
        }
    }
    /**
     * Loads the set of concepts specified by the array of entity IDs.
     *
     * @param conceptIDs                the array of concept entity IDs
     */
    public Set loadConcepts(EntityID[] conceptIDs) throws KAONException {
        Set set=new HashSet();
        for (int i=0;i<conceptIDs.length;i++)
            set.add(getConcept(conceptIDs[i]));
        return set;
    }
    /**
     * Loads the map of concepts specified by the array of relation entity IDs.
     *
     * @param relationConceptIDs        the array of concept relation entity IDs
     */
    public Map loadConcepts(RelationEntityID[] relationConceptIDs) throws KAONException {
        Map map=new HashMap();
        for (int i=0;i<relationConceptIDs.length;i++) {
            RelationEntityID relationEntityID=relationConceptIDs[i];
            ConceptProxy concept=getConcept(relationEntityID);
            OIModel sourceOIModel=getOIModel(relationEntityID.m_sourceOIModelID);
            map.put(concept,sourceOIModel);
        }
        return map;
    }
    /**
     * Loads the set of properties specified by the array of entity IDs.
     *
     * @param propertyIDs               the array of property entity IDs
     */
    public Set loadProperties(EntityID[] propertyIDs) throws KAONException {
        Set set=new HashSet();
        for (int i=0;i<propertyIDs.length;i++)
            set.add(getProperty(propertyIDs[i]));
        return set;
    }
    /**
     * Loads the map of properties specified by the array of relation entity IDs.
     *
     * @param relationPropertyIDs       the array of property relation entity IDs
     */
    public Map loadProperties(RelationEntityID[] relationPropertyIDs) throws KAONException {
        Map map=new HashMap();
        for (int i=0;i<relationPropertyIDs.length;i++) {
            RelationEntityID relationEntityID=relationPropertyIDs[i];
            PropertyProxy property=getProperty(relationEntityID);
            OIModel sourceOIModel=getOIModel(relationEntityID.m_sourceOIModelID);
            map.put(property,sourceOIModel);
        }
        return map;
    }
    /**
     * Loads the set of instances specified by the array of entity IDs.
     *
     * @param instanceIDs               the array of instance entity IDs
     */
    public Set loadInstances(EntityID[] instanceIDs) throws KAONException {
        Set set=new HashSet();
        for (int i=0;i<instanceIDs.length;i++)
            set.add(getInstance(instanceIDs[i]));
        return set;
    }
    /**
     * Returns the OI-model for given model ID.
     *
     * @param modelID                   the ID of the model
     */
    public OIModel getOIModel(int modelID) throws KAONException {
        return m_oimodelSource.getOIModel(modelID);
    }
    /**
     * Notifies the model that the changes will be performed.
     */
    public void startModifying() {
        suspendEvents();
        m_bookmark=m_eventManager.getBookmark();
    }
    /**
     * Notifies this OI-model about a change in this OI-model.
     *
     * @param changeEvent               the change event
     */
    public void notifyOIModelChanged(ChangeEvent changeEvent) throws KAONException {
        changeEvent.accept(m_includedOIModelEventCopier);
    }
    /**
     * Rolls back the model.
     */
    public void rollback() {
        if (m_bookmark!=null)
            m_eventManager.applyBookmark(m_bookmark);
        m_bookmark=null;
    }
    /**
     * Notifies the model that the changes are done.
     */
    public void endModifying() {
        m_includedOIModelEventCopier.fixCauses();
        m_includedOIModelEventCopier.clear();
        m_bookmark=null;
        resumeEvents();
    }
    /**
     * Returns the lock object for this entity.
     *
     * @return                              the lock object for the entity
     */
    protected Object getLock() {
        return m_kaonConnection;
    }

    /**
     * The class that copies the events from other OI-models.
     */
    protected class IncludedOIModelEventCopier extends ChangeEventCopier {
        public IncludedOIModelEventCopier() {
            super(OIModelProxy.this);
        }
        protected void consumeEvent(ChangeEvent changeEvent) throws KAONException {
            changeEvent.accept(m_eventProcessor);
            m_eventManager.notifyChangeEvent(changeEvent);
        }
    }
}
