package edu.unika.aifb.kaon.apiproxy;

import java.util.List;
import java.util.LinkedList;
import java.util.Map;
import java.util.HashMap;
import java.util.Random;
import java.util.Iterator;
import java.io.Serializable;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.change.*;
import edu.unika.aifb.kaon.api.util.*;

import edu.unika.aifb.kaon.apiproxy.source.*;

/**
 * Extension of the KAONConnection interface with some methods needed for this API implementation.
 */
public abstract class KAONConnectionProxy implements KAONConnection {
    /** The random number generator. */
    protected static final Random s_random=new Random();

    /** Identifier of this connection. */
    protected String m_connectionIdentifier;
    /** The list of available object loaders. */
    protected List m_availableObjectLoaders;
    /** The maximum number of object loaders. */
    protected int m_maximumObjectLoaders;
    /** The current number of object loaders. */
    protected int m_numberOfObjectLoaders;
    /** The thread group of all object loaders. */
    protected ThreadGroup m_objectLoadersGroup;

    /**
     * Creates an instance of this class.
     */
    public KAONConnectionProxy() {
        m_connectionIdentifier=System.currentTimeMillis()+"-"+s_random.nextInt();
        m_availableObjectLoaders=new LinkedList();
        m_maximumObjectLoaders=10;
        m_objectLoadersGroup=new ThreadGroup("Loaders_"+m_connectionIdentifier);
        m_objectLoadersGroup.setDaemon(true);
    }
    /**
     * Returns the capabilities of the model.
     *
     * @return                          the bit-mask defining the model's capaibilities
     */
    public synchronized int getCapabilities() {
        return OIModel.CAPABILITY_SUPPORTS_NOTIFICATIONS | OIModel.CAPABILITY_SUPPORTS_OPTIMIZED_LOADING;
    }
    /**
     * Returns the connection identifier.
     *
     * @return                                  the connection identifier
     */
    public synchronized String getConnectionIdentifier() {
        return m_connectionIdentifier;
    }
    /**
     * Closes this connection. All connections created by this connection are also closed.
     *
     * @throws KAONException                    thrown if there is an error
     */
    public synchronized void close() throws KAONException {
        if (!isOpen())
            throw new KAONException("KAON connection already closed.");
        m_objectLoadersGroup.interrupt();
        m_objectLoadersGroup=null;
        m_availableObjectLoaders=null;
    }
    /**
     * Returns <code>true</code> if this connection is open.
     *
     * @return                                  <code>true</code> if the connection is open
     */
    public synchronized boolean isOpen() {
        return m_objectLoadersGroup!=null;
    }
    /**
     * Called when the connection is garbage-collected.
     *
     * @throws Throwable                        thrown if there is an error
     */
    protected void finalize() throws Throwable {
        if (isOpen())
            close();
    }
    /**
     * Notifies this connection about an update.
     *
     * @param result                            contains information about the update
     * @throws KAONException                    thrown if there is an error
     */
    public synchronized void updateNotify(UpdateResultHolder result) throws KAONException {
        if (!m_connectionIdentifier.equals(result.m_initiatingConnectionIdentifier)) {
            final List originalList=new LinkedList();
            final Map oimodelsByLogicalURI=new HashMap();
            Iterator iterator=getOpenOIModels().iterator();
            while (iterator.hasNext()) {
                OIModel oimodel=(OIModel)iterator.next();
                oimodelsByLogicalURI.put(oimodel.getLogicalURI(),oimodel);
            }
            ChangeEventCopier eventCopier=new ChangeEventCopier() {
                protected boolean preprocessEvent(ChangeEvent event) throws KAONException {
                    super.preprocessEvent(event);
                    m_oimodel=(OIModel)oimodelsByLogicalURI.get(event.getOIModel().getLogicalURI());
                    return m_oimodel!=null;
                }
                protected void consumeEvent(ChangeEvent changeEvent) {
                    originalList.add(changeEvent);
                }
            };
            eventCopier.processChanges(result.m_updatedChangeList);
            processChangeNotification(result,originalList);
        }
    }
    /**
     * Applies the list of changes to the models in this connection.
     *
     * @param changeList                        list of changes to the models in the connection
     * @throws KAONException                    thrown if there is an error
     */
    public synchronized void applyChanges(List changeList) throws KAONException {
        // This is important! If load requests are pending, then it may happen that the load requests will be finished only
        // after apply is processed. In that case the state of the model can get out of sync. Therefore, applyChanges() is
        // an exclusive operation - no load request can be executed in parallel.
        while (m_availableObjectLoaders.size()!=m_numberOfObjectLoaders)
            try {
                wait();
            }
            catch (InterruptedException e) {
                throw new KAONException("Apply interrupted.",e);
            }
        UpdateResultHolder result=updateOIModel(changeList);
        processChangeNotification(result,changeList);
    }
    /**
     * Processes notification about a change.
     *
     * @param result                            contains information about the update
     * @param originalList                      the list of original events
     * @throws KAONException                    thrown if there is an error
     */
    protected void processChangeNotification(UpdateResultHolder result,List originalList) throws KAONException {
        // We can safely update the versions of entities in all open OI-models. This is due to the fact that
        // within a KAONConnection all models will be updated later.
        Iterator iterator=getOpenOIModels().iterator();
        while (iterator.hasNext()) {
            OIModelProxy oimodel=(OIModelProxy)iterator.next();
            for (int i=0;i<result.m_updatedConceptIDs.length;i++) {
                EntityID entityID=result.m_updatedConceptIDs[i];
                ConceptProxy concept=(ConceptProxy)oimodel.m_concepts.get(entityID.m_uri);
                if (concept!=null)
                    concept.updateVersion(entityID);
            }
            for (int i=0;i<result.m_updatedPropertyIDs.length;i++) {
                EntityID entityID=result.m_updatedPropertyIDs[i];
                PropertyProxy property=(PropertyProxy)oimodel.m_properties.get(entityID.m_uri);
                if (property!=null)
                    property.updateVersion(entityID);
            }
            for (int i=0;i<result.m_updatedInstanceIDs.length;i++) {
                EntityID entityID=result.m_updatedInstanceIDs[i];
                InstanceProxy instance=(InstanceProxy)oimodel.m_lexicalEntries.get(entityID.m_uri);
                if (instance!=null)
                    instance.updateVersion(entityID);
            }
        }
        // Now we apply the changes to the models.
        try {
            iterator=getOpenOIModels().iterator();
            while (iterator.hasNext()) {
                OIModelProxy oimodel=(OIModelProxy)iterator.next();
                oimodel.startModifying();
            }
            if (originalList!=result.m_updatedChangeList) {
                UpdateMerger updateMerger=new UpdateMerger();
                updateMerger.mergeEvents(originalList,result.m_updatedChangeList);
            }
            Iterator original=originalList.iterator();
            while (original.hasNext()) {
                ChangeEvent originalEvent=(ChangeEvent)original.next();
                OIModelProxy oimodel=(OIModelProxy)originalEvent.getOIModel();
                oimodel.notifyOIModelChanged(originalEvent);
                if (!oimodel.getAllIncludedByOIModels().isEmpty()) {
                    Iterator superOIModels=oimodel.getAllIncludedByOIModels().iterator();
                    while (superOIModels.hasNext()) {
                        OIModelProxy superOIModel=(OIModelProxy)superOIModels.next();
                        superOIModel.notifyOIModelChanged(originalEvent);
                    }
                }
            }
        }
        catch (KAONException e) {
            iterator=getOpenOIModels().iterator();
            while (iterator.hasNext()) {
                OIModelProxy oimodel=(OIModelProxy)iterator.next();
                oimodel.rollback();
            }
            throw e;
        }
        finally {
            iterator=getOpenOIModels().iterator();
            while (iterator.hasNext()) {
                OIModelProxy oimodel=(OIModelProxy)iterator.next();
                oimodel.endModifying();
            }
        }
    }
    /**
     * Executes a request to load object.
     *
     * @param oimodelSource                     the OI-model source
     * @param entityURIs                        the URIs to load
     * @param loadFlag                          the load flag
     * @return                                  loaded entities
     * @throws KAONException                    thrown if there is an error loading data
     */
    public synchronized EntityInfo[] loadEntities(OIModelSource oimodelSource,String[] entityURIs,int loadFlag) throws KAONException {
        if (m_availableObjectLoaders.isEmpty() && m_numberOfObjectLoaders<m_maximumObjectLoaders)
            m_availableObjectLoaders.add(new ObjectLoader());
        while (m_availableObjectLoaders.isEmpty())
            try {
                wait();
            }
            catch (InterruptedException e) {
                throw new KAONException("Thread interrupted.",e);
            }
        ObjectLoader objectLoader=(ObjectLoader)m_availableObjectLoaders.remove(0);
        try {
            return objectLoader.doLoading(oimodelSource,entityURIs,loadFlag);
        }
        finally {
            m_availableObjectLoaders.add(0,objectLoader);
            notifyAll();
        }
    }
    /**
     * Executes an update of the model.
     *
     * @param changeList                        list of changes to apply
     * @return                                  object containing information about the update
     * @throws KAONException                    thrown if there is a problem with fetching or updating information
     */
    protected abstract UpdateResultHolder updateOIModel(List changeList) throws KAONException;


    /**
     * Holder for the result of the update.
     */
    public static class UpdateResultHolder implements Serializable {
        public String m_initiatingConnectionIdentifier;
        public List m_updatedChangeList;
        public EntityID[] m_updatedConceptIDs;
        public EntityID[] m_updatedPropertyIDs;
        public EntityID[] m_updatedInstanceIDs;
    }

    /**
     * The object loader thread that services requests for object loading.
     */
    protected class ObjectLoader extends Thread {
        /** The OIModelSource that should serivce the request. */
        protected OIModelSource m_oimodelSource;
        /** The URIs to load. */
        protected String[] m_entityURIs;
        /** The load flag. */
        protected int m_loadFlag;
        /** The loaded entity infos. */
        protected EntityInfo[] m_entityInfos;
        /** The exception that has been thrown. */
        protected KAONException m_exception;

        /**
         * Creates a loader thread.
         */
        public ObjectLoader() {
            super(m_objectLoadersGroup,"Loader-"+m_numberOfObjectLoaders);
            m_numberOfObjectLoaders++;
            setDaemon(true);
            start();
        }
        /**
         * Called to perform loading. This method must be called with a lock held on the connection.
         *
         * @param oimodelSource                 the OI-model source
         * @param entityURIs                    the URIs to load
         * @param loadFlag                      the load flag
         * @return                              loaded entities
         * @throws KAONException                thrown if there is an error loading data
         */
        public EntityInfo[] doLoading(OIModelSource oimodelSource,String[] entityURIs,int loadFlag) throws KAONException {
            try {
                synchronized (this) {
                    m_oimodelSource=oimodelSource;
                    m_entityURIs=entityURIs;
                    m_loadFlag=loadFlag;
                    notify();
                }
                while (m_entityInfos==null && m_exception==null)
                    try {
                        KAONConnectionProxy.this.wait();
                    }
                    catch (InterruptedException e) {
                        throw new KAONException("Thread interrupted",e);
                    }
                if (m_exception!=null)
                    throw m_exception;
                else
                    return m_entityInfos;
            }
            finally {
                synchronized (this) {
                    m_oimodelSource=null;
                    m_entityURIs=null;
                    m_exception=null;
                    m_entityInfos=null;
                    notify();
                }
            }
        }
        /**
         * This method waits for the loading requests and performs the loading.
         */
         public void run() {
            try {
                while (true) {
                    OIModelSource oimodelSource=null;
                    String[] entityURIs=null;
                    EntityInfo[] entityInfos=null;
                    KAONException exception=null;
                    int loadFlag=0;
                    // here I wait until there is work to do
                    synchronized (this) {
                        while (m_oimodelSource==null)
                            wait();
                        oimodelSource=m_oimodelSource;
                        entityURIs=m_entityURIs;
                        entityInfos=m_entityInfos;
                        exception=m_exception;
                        loadFlag=m_loadFlag;
                    }
                    try {
                        entityInfos=oimodelSource.loadEntities(entityURIs,loadFlag);
                    }
                    catch (KAONException e) {
                        exception=e;
                    }
                    synchronized (KAONConnectionProxy.this) {
                        m_entityInfos=entityInfos;
                        m_exception=exception;
                        KAONConnectionProxy.this.notifyAll();
                    }
                    synchronized (this) {
                        while (m_entityInfos!=null || m_exception!=null)
                            wait();
                    }
                    // here I'm sure that the message has been read by doLoading()
                }
            }
            catch (InterruptedException exitNow) {
            }
         }
    }
}
