package edu.unika.aifb.kaon.apiproxy;

import java.util.Iterator;
import java.util.Set;
import java.util.HashSet;
import java.util.Map;
import java.util.HashMap;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;

import edu.unika.aifb.kaon.apiproxy.source.*;

/**
 * Implementation of the Instance interface that relies on the RDF API as the model.
 */
public class InstanceProxy extends AbstractEntityProxy implements Instance {
    /** Map of parent concepts - values are the models where the superconcept relationship has been declared. */
    transient protected Map m_parentConcepts;
    /** Set of property instances from this instance. */
    transient protected Set m_propertyInstancesFrom;
    /** Set of property values to this instance. */
    transient protected Set m_propertyInstancesTo;
    /** A strong reference to the spanning concept. */
    transient protected Concept m_spanningConcept;
    /** A strong reference to the spanning property. */
    transient protected Property m_spanningProperty;

    /**
     * Creates an instance of this class and attaches it to the OI-model.
     *
     * @param oimodel                   OI-model
     * @param uri                       URI of the concept
     */
    public InstanceProxy(OIModelProxy oimodel,String uri) {
        super(oimodel,uri);
    }
    /**
     * This constructor initializes only the URI and the version, and should be used exclusively for
     * creating dummy objects for passing them from server.
     *
     * @param uri                       URI of the concept
     * @param version                   the version of the object
     */
    public InstanceProxy(String uri,int version) {
        super(uri,version);
    }
    /**
     * Unloads this object.
     */
    public void unload() {
        synchronized (getLock()) {
            super.unload();
            m_parentConcepts=null;
            m_propertyInstancesFrom=null;
            m_propertyInstancesTo=null;
            m_spanningConcept=null;
            m_spanningProperty=null;
        }
    }
    /**
     * Returns the bit mask specifying which parts of the object have been loaded.
     *
     * @return                              the bit mask specifying which parts of the object have been loaded
     */
    public int getLoadedState() {
        synchronized (getLock()) {
            int loadedState=0;
            if (hasLoadedBasics())
                loadedState|=OIModel.LOAD_INSTANCE_BASICS;
            if (m_parentConcepts!=null)
                loadedState|=OIModel.LOAD_INSTANCE_PARENT_CONCEPTS;
            if (m_propertyInstancesFrom!=null)
                loadedState|=OIModel.LOAD_INSTANCE_FROM_PROPERTY_VALUES;
            if (m_propertyInstancesTo!=null)
                loadedState|=OIModel.LOAD_INSTANCE_TO_PROPERTY_VALUES;
            return loadedState;
        }
    }
    /**
     * Returns set of <code>Concepts</code> objects that specify which concepts is this instance instance of.
     *
     * @return                                  set of concepts that this instance is instance of (may be empty)
     */
    public Set getParentConcepts() throws KAONException {
        synchronized (getLock()) {
            if (m_parentConcepts==null)
                loadThisObject(OIModel.LOAD_INSTANCE_PARENT_CONCEPTS);
            return m_parentConcepts.keySet();
        }
    }
    /**
     * Returns set of all <code>Concepts</code> objects that this instance is instance of.
     *
     * @return                                  set of all concepts that this instance is instance of (may be empty)
     */
    public Set getAllParentConcepts() throws KAONException {
        synchronized (getLock()) {
            // if there are hierarcy cycles, this method will loop forever
            Set parentConcepts=getParentConcepts();
            Set result=new HashSet(parentConcepts);
            Iterator iterator=parentConcepts.iterator();
            while (iterator.hasNext()) {
                Concept concept=(Concept)iterator.next();
                result.addAll(concept.getAllSuperConcepts());
            }
            return result;
        }
    }
    /**
     * Tests if given concept is a direct parent of this instance.
     *
     * @param potentialParent           concept being tested
     * @return                          <code>true</code> if given concept is a direct parent of this instance
     */
    public boolean isDirectParent(Concept potentialParent) throws KAONException {
        synchronized (getLock()) {
            return getParentConcepts().contains(potentialParent);
        }
    }
    /**
     * Returns the OI-model containing the this concept - instance association.
     *
     * @param concept                   the concept of this instance
     * @return                          the OI-model in which the concept-instance association is declared
     */
    public OIModel getConceptInstanceOIModel(Concept concept) throws KAONException {
        synchronized (getLock()) {
            if (m_parentConcepts==null)
                loadThisObject(OIModel.LOAD_INSTANCE_PARENT_CONCEPTS);
            return (OIModel)m_parentConcepts.get(concept);
        }
    }
    /**
     * Returns <code>true</code> if the supplied concept has been declared a parent of this instance in this OI-model.
     *
     * @param concept                   the concept of this instance
     * @return                          <code>true</code> if supplied concept has been declared a parent of this instance in this OI-model
     */
    public boolean isConceptInstanceDeclaredLocally(Concept concept) throws KAONException {
        synchronized (getLock()) {
            return getConceptInstanceOIModel(concept)==m_oimodel;
        }
    }
    /**
     * Tests if given concept is a parent of this concept.
     *
     * @param potentialParent           concept being tested
     * @return                          <code>true</code> if given concept is a parent of this instance
     */
    public boolean isParent(Concept potentialParent) throws KAONException {
        synchronized (getLock()) {
            return getAllParentConcepts().contains(potentialParent);
        }
    }
    /**
     * Returns the set of <code>PropertyInstnace</code> objects that point out from this instance.
     *
     * @return                                  set of property instances pointing from this instance (may be empty)
     */
    public Set getFromPropertyInstances() throws KAONException {
        synchronized (getLock()) {
            if (m_propertyInstancesFrom==null)
                loadThisObject(OIModel.LOAD_INSTANCE_FROM_PROPERTY_VALUES);
            return m_propertyInstancesFrom;
        }
    }
    /**
     * Returns the map of sets of property values. The map is indexed by <code>Property</code> objects.
     *
     * @return                                  map of sets of property values
     */
    public Map getFromPropertyValues() throws KAONException {
        synchronized (getLock()) {
            Map map=new HashMap();
            Iterator iterator=getFromPropertyInstances().iterator();
            while (iterator.hasNext()) {
                PropertyInstance propertyInstance=(PropertyInstance)iterator.next();
                Set set=(Set)map.get(propertyInstance.getProperty());
                if (set==null) {
                    set=new HashSet();
                    map.put(propertyInstance.getProperty(),set);
                }
                set.add(propertyInstance.getTargetValue());
            }
            return map;
        }
    }
    /**
     * Returns the value of a given property for this instance. If there is no value for a property, <code>null</code> is
     * returned. If there are multiple values for a property, one is picked arbitrarily and returned. Returned value is either
     * an <code>Instance</code> or a <code>String</code>.
     *
     * @param property                          property
     * @return                                  value of the property or <code>null</code> if there is no value
     */
    public Object getFromPropertyValue(Property property) throws KAONException {
        synchronized (getLock()) {
            Set set=getFromPropertyValues(property);
            if (set.isEmpty())
                return null;
            else
                return set.iterator().next();
            }
    }
    /**
     * Returns the set of values of a given property. It there are no values, an empty set is returned. Elements of the set
     * may be <code>Instance</code> or <code>String</code> objects.
     *
     * @param property                          property
     * @return                                  set of values of the property
     */
    public Set getFromPropertyValues(Property property) throws KAONException {
        synchronized (getLock()) {
            Set result=new HashSet();
            Iterator iterator=getFromPropertyInstances().iterator();
            while (iterator.hasNext()) {
                PropertyInstance propertyInstance=(PropertyInstance)iterator.next();
                if (propertyInstance.getProperty().equals(property))
                    result.add(propertyInstance.getTargetValue());
            }
            return result;
        }
    }
    /**
     * Tests whether this instance points to given value using specified property.
     *
     * @param property                          property
     * @param targetValue                       the value (can be a string or an instance)
     * @return                                  <code>true</code> if this instance points to given value through given property (by including all inferred facts)
     */
    public boolean pointsToValue(Property property,Object targetValue) throws KAONException {
        synchronized (getLock()) {
            if (targetValue instanceof Instance)
                return m_oimodel.getOIModelSource().pointsToValue(m_uri,property.getURI(),((Instance)targetValue).getURI(),null);
            else
                return m_oimodel.getOIModelSource().pointsToValue(m_uri,property.getURI(),null,(String)targetValue);
        }
    }
    /**
     * Returns the values of a property for this instance, including the inferred ones.
     *
     * @param property                          property for which the value is requested
     * @return                                  set of property values for this given property
     */
    public Set getAllFromPropertyValues(Property property) throws KAONException {
        synchronized (getLock()) {
            Object[][] result=m_oimodel.getOIModelSource().getAllFromPropertyValues(m_uri,property.getURI());
            Set set=new HashSet();
            if (result.length==1) {
                Object[] valuesOfProperty=result[0];
                EntityID propertyID=(EntityID)valuesOfProperty[0];
                if (propertyID.m_uri.equals(property.getURI())) {
                    for (int j=1;j<valuesOfProperty.length;j++) {
                        Object value=valuesOfProperty[j];
                        if (value instanceof EntityID) {
                            InstanceProxy instance=m_oimodel.getInstance((EntityID)valuesOfProperty[j]);
                            set.add(instance);
                        }
                        else
                            set.add(value);
                    }
                }
            }
            return set;
        }
    }
    /**
     * Returns the values of all properties for this instance, including the inferred ones.
     *
     * @return                                  set of all property values for this instance
     */
    public Map getAllFromPropertyValues() throws KAONException {
        synchronized (getLock()) {
            Object[][] result=m_oimodel.getOIModelSource().getAllFromPropertyValues(m_uri,null);
            Map map=new HashMap();
            for (int i=0;i<result.length;i++) {
                Object[] valuesOfProperty=result[i];
                PropertyProxy property=m_oimodel.getProperty((EntityID)valuesOfProperty[0]);
                Set set=new HashSet();
                for (int j=1;j<valuesOfProperty.length;j++) {
                    Object value=valuesOfProperty[j];
                    if (value instanceof EntityID) {
                        InstanceProxy instance=m_oimodel.getInstance((EntityID)valuesOfProperty[j]);
                        set.add(instance);
                    }
                    else
                        set.add(value);
                }
                map.put(property,set);
            }
            return map;
        }
    }
    /**
     * Adds all elements to from source to destination that are not in the control set as well.
     *
     * @param source                            source set
     * @param destination                       destination set
     * @param control                           control set
     */
    protected void addElements(Set source,Set destination,Set control) {
        Iterator iterator=source.iterator();
        while (iterator.hasNext()) {
            Object element=iterator.next();
            if (!control.contains(element))
                destination.add(element);
        }
    }
    /**
     * Returns the set of <code>PropertyInstnace</code> objects that point to this instance.
     *
     * @return                                  set of property instances pointing to this instance (may be empty)
     */
    public Set getToPropertyInstances() throws KAONException {
        synchronized (getLock()) {
            if (m_propertyInstancesTo==null)
                loadThisObject(OIModel.LOAD_INSTANCE_TO_PROPERTY_VALUES);
            return m_propertyInstancesTo;
        }
    }
    /**
     * Returns the map of sets of property values pointing to the instance. The map is indexed by <code>Property</code> objects.
     *
     * @return                                  map of sets of property values
     */
    public Map getToPropertyValues() throws KAONException {
        synchronized (getLock()) {
            Map map=new HashMap();
            Iterator iterator=getToPropertyInstances().iterator();
            while (iterator.hasNext()) {
                PropertyInstance propertyInstance=(PropertyInstance)iterator.next();
                Set set=(Set)map.get(propertyInstance.getProperty());
                if (set==null) {
                    set=new HashSet();
                    map.put(propertyInstance.getProperty(),set);
                }
                set.add(propertyInstance.getSourceInstance());
            }
            return map;
        }
    }
    /**
     * Returns the set of <code>Instnace</code> objects that point to this instance via given property.
     *
     * @param property                          property of interest
     * @return                                  set of <code>Instance</code> objects pointing to this instance via given property
     */
    public Set getToPropertyValues(Property property) throws KAONException {
        synchronized (getLock()) {
            Set result=new HashSet();
            Iterator iterator=getToPropertyInstances().iterator();
            while (iterator.hasNext()) {
                PropertyInstance propertyInstance=(PropertyInstance)iterator.next();
                if (propertyInstance.getProperty().equals(property))
                    result.add(propertyInstance.getSourceInstance());
            }
            return result;
        }
    }
    /**
     * Returns the values of a property pointing to this instance, including the inferred ones.
     *
     * @param property                          property for which the value is requested
     * @return                                  set of all property values pointing to this instance for given property
     */
    public Set getAllToPropertyValues(Property property) throws KAONException {
            Object[][] result=m_oimodel.getOIModelSource().getAllToPropertyValues(m_uri,property.getURI());
            Set set=new HashSet();
            if (result.length==1) {
                Object[] valuesOfProperty=result[0];
                EntityID propertyID=(EntityID)valuesOfProperty[0];
                if (propertyID.m_uri.equals(property.getURI())) {
                    for (int j=1;j<valuesOfProperty.length;j++) {
                        Object value=valuesOfProperty[j];
                        if (value instanceof EntityID) {
                            InstanceProxy instance=m_oimodel.getInstance((EntityID)valuesOfProperty[j]);
                            set.add(instance);
                        }
                        else
                            set.add(value);
                    }
                }
            }
            return set;
    }
    /**
     * Returns the values of all properties pointing to this instance, including the inferred ones.
     *
     * @return                                  set of all property values pointing to this instance
     */
    public Map getAllToPropertyValues() throws KAONException {
        synchronized (getLock()) {
            Object[][] result=m_oimodel.getOIModelSource().getAllToPropertyValues(m_uri,null);
            Map map=new HashMap();
            for (int i=0;i<result.length;i++) {
                Object[] valuesOfProperty=result[i];
                PropertyProxy property=m_oimodel.getProperty((EntityID)valuesOfProperty[0]);
                Set set=new HashSet();
                for (int j=1;j<valuesOfProperty.length;j++) {
                    Object value=valuesOfProperty[j];
                    if (value instanceof EntityID) {
                        InstanceProxy instance=m_oimodel.getInstance((EntityID)valuesOfProperty[j]);
                        set.add(instance);
                    }
                    else
                        set.add(value);
                }
                map.put(property,set);
            }
            return map;
        }
    }
    /**
     * Returns the spanning concept for this instance.
     *
     * @return                                  spanning concept for this instance
     */
    public Concept getSpanningConcept() {
        synchronized (getLock()) {
            if (m_spanningConcept==null)
                m_spanningConcept=m_oimodel.getConcept(getURI());
            return m_spanningConcept;
        }
    }
    /**
     * Returns the spanning property for this instance.
     *
     * @return                                  spanning property for this instance
     */
    public Property getSpanningProperty() {
        synchronized (getLock()) {
            if (m_spanningProperty==null)
                m_spanningProperty=m_oimodel.getProperty(getURI());
            return m_spanningProperty;
        }
    }
    /**
     * Returns the instance associated with this object.
     *
     * @return                              instance associated with this object
     */
    public Instance getSpanningInstance() {
        return this;
    }
    /**
     * Makes sure that the basics of this object are loaded.
     */
    protected void makeSureBasicsLoaded() throws KAONException {
        if (!hasLoadedBasics())
            loadThisObject(OIModel.LOAD_INSTANCE_BASICS);
    }
    /**
     * Accepts an entity visitor.
     *
     * @param visitor                   visitor to apply
     */
    public void accept(EntityVisitor visitor) throws KAONException {
        visitor.visit(this);
    }
    /**
     * Sets the flag that determines whether this entity is in the model.
     *
     * @param isInOIModel               flag determining whether this entity is in the model
     * @param sourceOIModel             the source OI-model
     */
    void setIsInOIModel(boolean isInOIModel,OIModel sourceOIModel) throws KAONException {
        super.setIsInOIModel(isInOIModel,sourceOIModel);
        if (!isInOIModel) {
            if (m_parentConcepts!=null)
                m_parentConcepts.clear();
            else
                m_parentConcepts=new HashMap();
            if (m_propertyInstancesFrom!=null)
                m_propertyInstancesFrom.clear();
            else
                m_propertyInstancesFrom=new HashSet();
            if (m_propertyInstancesTo!=null)
                m_propertyInstancesTo.clear();
            else
                m_propertyInstancesTo=new HashSet();
        }
    }
    /**
     * Loads a map of properties and their values.
     *
     * @param propertyValues            an array of properties and their values
     * @param loadPropertiesFrom        set to <code>true</code> if properties from should be loaded
     * @return                          the set of property instances
     */
    Set loadPropertyValues(Object[][] propertyValues,boolean loadPropertiesFrom) throws KAONException {
        Set propertyInstances=new HashSet();
        for (int i=0;i<propertyValues.length;i++) {
            Property property=m_oimodel.getProperty((EntityID)propertyValues[i][0]);
            for (int j=1;j<propertyValues[i].length;j++) {
                Object value=propertyValues[i][j];
                Object valueObject;
                OIModel sourceOIModel;
                if (value instanceof RelationEntityID) {
                    RelationEntityID relationEntityID=(RelationEntityID)value;
                    valueObject=m_oimodel.getInstance(relationEntityID);
                    sourceOIModel=m_oimodel.getOIModel(relationEntityID.m_sourceOIModelID);
                }
                else {
                    PropertyValue propertyValue=(PropertyValue)value;
                    valueObject=propertyValue.m_textValue;
                    sourceOIModel=m_oimodel.getOIModel(propertyValue.m_sourceOIModelID);
                }
                PropertyInstanceProxy propertyInstance;
                if (loadPropertiesFrom)
                    propertyInstance=(PropertyInstanceProxy)m_oimodel.getPropertyInstance(property,this,valueObject);
                else
                    propertyInstance=(PropertyInstanceProxy)m_oimodel.getPropertyInstance(property,(Instance)valueObject,this);
                propertyInstance.setSourceOIModel(sourceOIModel);
                propertyInstances.add(propertyInstance);
            }
        }
        return propertyInstances;
    }
    /**
     * Loads this object.
     *
     * @param instanceInfo              the information about an instance
     */
    void loadObject(InstanceInfo instanceInfo) throws KAONException {
        checkVersions(instanceInfo.m_id);
        if (instanceInfo.m_modelID!=-1)
            loadBasics(instanceInfo);
        if (instanceInfo.m_instanceParentConceptIDs!=null && m_parentConcepts==null)
            m_parentConcepts=loadConcepts(instanceInfo.m_instanceParentConceptIDs);
        if (instanceInfo.m_propertyValuesFrom!=null && m_propertyInstancesFrom==null)
            m_propertyInstancesFrom=loadPropertyValues(instanceInfo.m_propertyValuesFrom,true);
        if (instanceInfo.m_propertyValuesTo!=null && m_propertyInstancesTo==null)
            m_propertyInstancesTo=loadPropertyValues(instanceInfo.m_propertyValuesTo,false);
    }
    /**
     * Adds the concept to this instance.
     *
     * @param concept                   the concept to add
     * @param sourceOIModel             the OI-model where concept-instance relationship has been declared
     */
    void addConcept(Concept concept,OIModel sourceOIModel) {
        if (m_parentConcepts!=null)
            m_parentConcepts.put(concept,sourceOIModel);
    }
    /**
     * Removes the concept to this instance.
     *
     * @param concept                   the concept to remove
     */
    void removeConcept(Concept concept) {
        if (m_parentConcepts!=null)
            m_parentConcepts.remove(concept);
    }
    /**
     * Adds a property value from this instance.
     *
     * @param propertyInstance          property instance
     */
    void addFromPropertyValue(PropertyInstance propertyInstance) {
        if (m_propertyInstancesFrom!=null)
            m_propertyInstancesFrom.add(propertyInstance);
    }
    /**
     * Removes a property value from this instance.
     *
     * @param propertyInstance          property instance
     */
    void removeFromPropertyValue(PropertyInstance propertyInstance) {
        if (m_propertyInstancesFrom!=null)
            m_propertyInstancesFrom.remove(propertyInstance);
    }
    /**
     * Adds a property value to this instance.
     *
     * @param propertyInstance          property instance
     */
    void addToPropertyValue(PropertyInstance propertyInstance) {
        if (m_propertyInstancesTo!=null)
            m_propertyInstancesTo.add(propertyInstance);
    }
    /**
     * Removes a property value to this instance.
     *
     * @param propertyInstance          property instance
     */
    void removeToPropertyValue(PropertyInstance propertyInstance) {
        if (m_propertyInstancesTo!=null)
            m_propertyInstancesTo.remove(propertyInstance);
    }
}
