package edu.unika.aifb.kaon.apiproxy;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.change.*;
import edu.unika.aifb.kaon.api.oimodel.*;

/**
 * Processes events received from the source and arranges the local model.
 */
public class EventProcessor implements ChangeVisitor {
    /** OI-model. */
    protected OIModelProxy m_oimodel;
    /** The current event. */
    protected ChangeEvent m_currentEvent;
    /** Visitor for adding. */
    protected EntityVisitor m_addVisitor;
    /** Visitor for removing. */
    protected EntityVisitor m_removeVisitor;

    /**
     * Creates an instance of this class.
     *
     * @param oimodel               the OI-model
     */
    public EventProcessor(OIModelProxy oimodel) {
        m_oimodel=oimodel;
        m_addVisitor=new AddEntityVisitor();
        m_removeVisitor=new RemoveEntityVisitor();
    }
    /**
     * Visits AddEntity change.
     */
    public void visit(AddEntity event) throws KAONException {
        m_currentEvent=event;
        try {
            event.getEntity().accept(m_addVisitor);
        }
        finally {
            m_currentEvent=null;
        }
    }
    /**
     * Visits RemoveEntity change.
     */
    public void visit(RemoveEntity event) throws KAONException {
        m_currentEvent=event;
        try {
            event.getEntity().accept(m_removeVisitor);
        }
        finally {
            m_currentEvent=null;
        }
    }
    /**
     * Visits an event for creation of subconcepts.
     */
    public void visit(AddSubConcept event) {
        ConceptProxy superConcept=(ConceptProxy)event.getSuperConcept();
        ConceptProxy subConcept=(ConceptProxy)event.getSubConcept();
        superConcept.addSubConcept(subConcept);
        subConcept.addSuperConcept(superConcept,event.getOIModel());
    }
    /**
     * Visits an event for removal of subconcepts.
     */
    public void visit(RemoveSubConcept event) {
        ConceptProxy superConcept=(ConceptProxy)event.getSuperConcept();
        ConceptProxy subConcept=(ConceptProxy)event.getSubConcept();
        superConcept.removeSubConcept(subConcept);
        subConcept.removeSuperConcept(superConcept);
    }
    /**
     * Visits an event for adding a domain to the property.
     */
    public void visit(AddPropertyDomain event) {
        PropertyProxy property=(PropertyProxy)event.getProperty();
        ConceptProxy concept=(ConceptProxy)event.getConcept();
        property.addDomainConcept(concept,event.getOIModel());
        concept.addPropertyFrom(property);
    }
    /**
     * Visits an event for removing a domain from the property.
     */
    public void visit(RemovePropertyDomain event) {
        PropertyProxy property=(PropertyProxy)event.getProperty();
        ConceptProxy concept=(ConceptProxy)event.getConcept();
        property.removeDomainConcept(concept);
        concept.removePropertyFrom(property);
    }
    /**
     * Visits an event for adding a range to the property.
     */
    public void visit(AddPropertyRange event) {
        PropertyProxy property=(PropertyProxy)event.getProperty();
        ConceptProxy concept=(ConceptProxy)event.getConcept();
        property.addRangeConcept(concept,event.getOIModel());
        concept.addPropertyTo(property);
    }
    /**
     * Visits an event for removing a range from the property.
     */
    public void visit(RemovePropertyRange event) {
        PropertyProxy property=(PropertyProxy)event.getProperty();
        ConceptProxy concept=(ConceptProxy)event.getConcept();
        property.removeRangeConcept(concept);
        concept.removePropertyTo(property);
    }
    /**
     * Visits an event for determinig whether property is attirubte.
     */
    public void visit(SetPropertyIsAttribute event) {
        PropertyProxy property=(PropertyProxy)event.getProperty();
        property.setIsAttribute(event.getIsAttribute());
    }
    /**
     * Visits an event for setting the minimum cardinality of a property for a concept.
     */
    public void visit(SetMinimumCardinality event) throws KAONException {
        PropertyProxy property=(PropertyProxy)event.getProperty();
        ConceptProxy concept=(ConceptProxy)event.getConcept();
        property.setMinimumCardinality(concept,event.getCardinality());
    }
    /**
     * Visits an event for setting the maximum cardinality of a property for a concept.
     */
    public void visit(SetMaximumCardinality event) throws KAONException {
        PropertyProxy property=(PropertyProxy)event.getProperty();
        ConceptProxy concept=(ConceptProxy)event.getConcept();
        property.setMaximumCardinality(concept,event.getCardinality());
    }
    /**
     * Visits an event for creation of subproperties.
     */
    public void visit(AddSubProperty event) {
        PropertyProxy superProperty=(PropertyProxy)event.getSuperProperty();
        PropertyProxy subProperty=(PropertyProxy)event.getSubProperty();
        superProperty.addSubProperty(subProperty);
        subProperty.addSuperProperty(superProperty,event.getOIModel());
    }
    /**
     * Visits an event for removal of subproperties.
     */
    public void visit(RemoveSubProperty event) {
        PropertyProxy superProperty=(PropertyProxy)event.getSuperProperty();
        PropertyProxy subProperty=(PropertyProxy)event.getSubProperty();
        superProperty.removeSubProperty(subProperty);
        subProperty.removeSuperProperty(superProperty);
    }
    /**
     * Visits an event for setting the inverse relationship between properties.
     */
    public void visit(SetInverseProperties event) {
        PropertyProxy property1=(PropertyProxy)event.getProperty1();
        PropertyProxy property2=(PropertyProxy)event.getProperty2();
        property1.setInverseProperty(property2,event.getOIModel());
        property2.setInverseProperty(property1,event.getOIModel());
    }
    /**
     * Visits an event for removing an inverse relationship between properties.
     */
    public void visit(SetNoInverseProperties event) {
        PropertyProxy property1=(PropertyProxy)event.getProperty1();
        PropertyProxy property2=(PropertyProxy)event.getProperty2();
        property1.setInverseProperty(null,null);
        property2.setInverseProperty(null,null);
    }
    /**
     * Visits an event for setting the symmetry flag of the property.
     */
    public void visit(SetPropertySymmetric event) throws KAONException {
        PropertyProxy property=(PropertyProxy)event.getProperty();
        property.setSymmetric(event.getSymmetric());
        event.setOldSymmetric(event.getOldSymmetric());
    }
    /**
     * Visits an event for setting the transitivity flag of the property.
     */
    public void visit(SetPropertyTransitive event) throws KAONException {
        PropertyProxy property=(PropertyProxy)event.getProperty();
        property.setTransitive(event.getTransitive());
        event.setOldTransitive(event.getOldTransitive());
    }
    /**
     * Visits an event for making an instance a subinstance of given concept.
     */
    public void visit(AddInstanceOf event) {
        ConceptProxy concept=(ConceptProxy)event.getConcept();
        InstanceProxy instance=(InstanceProxy)event.getInstance();
        concept.addInstance(instance);
        instance.addConcept(concept,event.getOIModel());
    }
    /**
     * Visits an event for making an instance not a subinstance of given concept.
     */
    public void visit(RemoveInstanceOf event) {
        ConceptProxy concept=(ConceptProxy)event.getConcept();
        InstanceProxy instance=(InstanceProxy)event.getInstance();
        concept.removeInstance(instance);
        instance.removeConcept(concept);
    }
    /**
     * Visits an event for adding a property instance.
     */
    public void visit(AddPropertyInstance event) {
        PropertyInstanceProxy propertyInstance=(PropertyInstanceProxy)event.getPropertyInstance();
        propertyInstance.setSourceOIModel(event.getOIModel());
        PropertyProxy property=(PropertyProxy)propertyInstance.getProperty();
        InstanceProxy sourceInstance=(InstanceProxy)propertyInstance.getSourceInstance();
        Object targetValue=propertyInstance.getTargetValue();
        property.addPropertyInstance(propertyInstance);
        sourceInstance.addFromPropertyValue(propertyInstance);
        if (targetValue instanceof InstanceProxy)
            ((InstanceProxy)targetValue).addToPropertyValue(propertyInstance);
    }
    /**
     * Visits an event for removing a property instance.
     */
    public void visit(RemovePropertyInstance event) {
        PropertyInstanceProxy propertyInstance=(PropertyInstanceProxy)event.getPropertyInstance();
        propertyInstance.setSourceOIModel(null);
        PropertyProxy property=(PropertyProxy)propertyInstance.getProperty();
        InstanceProxy sourceInstance=(InstanceProxy)propertyInstance.getSourceInstance();
        Object targetValue=propertyInstance.getTargetValue();
        property.removePropertyInstance(propertyInstance);
        sourceInstance.removeFromPropertyValue(propertyInstance);
        if (targetValue instanceof InstanceProxy)
            ((InstanceProxy)targetValue).removeToPropertyValue(propertyInstance);
    }
    /**
     * Visits an event for settomg the value of a property instance.
     */
    public void visit(SetPropertyInstanceValue event) {
        PropertyInstanceProxy propertyInstance=(PropertyInstanceProxy)event.getPropertyInstance();
        propertyInstance.setSourceOIModel(event.getOIModel());
        PropertyProxy property=(PropertyProxy)propertyInstance.getProperty();
        InstanceProxy sourceInstance=(InstanceProxy)propertyInstance.getSourceInstance();
        Object targetValue=propertyInstance.getTargetValue();
        PropertyInstance[] removedInstances=event.getRemovedInstances();
        for (int i=0;i<removedInstances.length;i++) {
            PropertyInstanceProxy removedInstance=(PropertyInstanceProxy)removedInstances[i];
            removedInstance.setSourceOIModel(null);
            Object removedTargetValue=removedInstance.getTargetValue();
            property.removePropertyInstance(removedInstance);
            sourceInstance.removeFromPropertyValue(removedInstance);
            if (removedTargetValue instanceof InstanceProxy)
                ((InstanceProxy)removedTargetValue).removeToPropertyValue(removedInstance);
        }
        property.addPropertyInstance(propertyInstance);
        sourceInstance.addFromPropertyValue(propertyInstance);
        if (targetValue instanceof InstanceProxy)
            ((InstanceProxy)targetValue).addToPropertyValue(propertyInstance);
    }
    /**
     * Visits an event for changing the value of a property instance.
     */
    public void visit(ChangePropertyInstanceValue event) throws KAONException {
        PropertyInstanceProxy propertyInstance=(PropertyInstanceProxy)event.getPropertyInstance();
        propertyInstance.setSourceOIModel(null);
        PropertyProxy property=(PropertyProxy)propertyInstance.getProperty();
        InstanceProxy sourceInstance=(InstanceProxy)propertyInstance.getSourceInstance();
        Object targetValue=propertyInstance.getTargetValue();
        property.removePropertyInstance(propertyInstance);
        sourceInstance.removeFromPropertyValue(propertyInstance);
        if (targetValue instanceof InstanceProxy)
            ((InstanceProxy)targetValue).removeToPropertyValue(propertyInstance);
        PropertyInstanceProxy newPropertyInstance=(PropertyInstanceProxy)m_oimodel.getPropertyInstance(propertyInstance.getProperty(),propertyInstance.getSourceInstance(),event.getNewTargetValue());
        newPropertyInstance.setSourceOIModel(event.getOIModel());
        Object newTargetValue=event.getNewTargetValue();
        property.addPropertyInstance(newPropertyInstance);
        sourceInstance.addFromPropertyValue(newPropertyInstance);
        if (newTargetValue instanceof InstanceProxy)
            ((InstanceProxy)newTargetValue).addToPropertyValue(newPropertyInstance);
    }
    /**
     * Visits an event for making a model included in another model.
     */
    public void visit(AddIncludedOIModel event) throws KAONException {
        m_oimodel.addIncludedOIModel(event.getIncludedOIModel());
    }
    /**
     * Visits an event for making a model not included in another model.
     */
    public void visit(RemoveIncludedOIModel event) throws KAONException {
        m_oimodel.removeIncludedOIModel(event.getIncludedOIModel());
    }

    /**
     * Visitor that adds elements to the ontology.
     */
    protected class AddEntityVisitor implements EntityVisitor {
        /**
         * Visits a concept.
         */
        public void visit(Concept entity) throws KAONException {
            ((ConceptProxy)entity).setIsInOIModel(true,m_currentEvent.getOIModel());
            visit(entity.getSpanningInstance());
        }
        /**
         * Visits a propety.
         */
        public void visit(Property entity) throws KAONException {
            ((PropertyProxy)entity).setIsInOIModel(true,m_currentEvent.getOIModel());
            visit(entity.getSpanningInstance());
        }
        /**
         * Visits an instance.
         */
        public void visit(Instance entity) throws KAONException {
            InstanceProxy instance=(InstanceProxy)entity;
            instance.setIsInOIModel(true,m_currentEvent.getOIModel());
            ConceptProxy root=(ConceptProxy)m_oimodel.getRootConcept();
            root.addInstance(instance);
            instance.addConcept(root,m_currentEvent.getOIModel());
        }
        /**
         * Visits a lexical entry.
         */
        public void visit(LexicalEntry entity) throws KAONException {
            visit((Instance)entity);
        }
    }

    /**
     * Visitor that removes elements to the ontology.
     */
    protected class RemoveEntityVisitor implements EntityVisitor {
        /**
         * Visits a concept.
         */
        public void visit(Concept entity) throws KAONException {
            ((ConceptProxy)entity).setIsInOIModel(false,null);
            visit(entity.getSpanningInstance());
        }
        /**
         * Visits a propety.
         */
        public void visit(Property entity) throws KAONException {
            ((PropertyProxy)entity).setIsInOIModel(false,null);
            visit(entity.getSpanningInstance());
        }
        /**
         * Visits an instance.
         */
        public void visit(Instance entity) throws KAONException {
            InstanceProxy instance=(InstanceProxy)entity;
            instance.setIsInOIModel(false,null);
            ConceptProxy root=(ConceptProxy)m_oimodel.getRootConcept();
            root.removeInstance(instance);
            instance.removeConcept(root);
        }
        /**
         * Visits a lexical entry.
         */
        public void visit(LexicalEntry entity) throws KAONException {
            visit((Instance)entity);
        }
    }
}
