package edu.unika.aifb.kaon.apiproxy;

import java.util.Iterator;
import java.util.Set;
import java.util.HashSet;
import java.util.Map;
import java.util.HashMap;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;

import edu.unika.aifb.kaon.apiproxy.source.*;

/**
 * Implementation of the Concept interface that relies on the RDF API as the model.
 */
public class ConceptProxy extends AbstractEntityProxy implements Concept {
    /** Map of direct superconcepts - values are the models where the superconcept relationship has been declared. */
    transient protected Map m_superConcepts;
    /** Set of direct subconcepts. */
    transient protected Set m_subConcepts;
    /** Set of property instances from this concept. */
    transient protected Set m_propertiesFrom;
    /** Set of property instances to this concept. */
    transient protected Set m_propertiesTo;
    /** Set of all direct instances of this concept. */
    transient protected Set m_instances;
    /** A strong reference to the spanning instance. */
    transient protected Instance m_spanningInstance;

    /**
     * Creates an instance of this class and attaches it to the OI-model.
     *
     * @param oimodel                   OI-model
     * @param uri                       URI of the concept
     */
    public ConceptProxy(OIModelProxy oimodel,String uri) {
        super(oimodel,uri);
    }
    /**
     * This constructor initializes only the URI and the version, and should be used exclusively for
     * creating dummy objects for passing them from server.
     *
     * @param uri                       URI of the concept
     * @param version                   the version of the object
     */
    public ConceptProxy(String uri,int version) {
        super(uri,version);
    }
    /**
     * Unloads this object.
     */
    public void unload() {
        synchronized (getLock()) {
            super.unload();
            m_superConcepts=null;
            m_subConcepts=null;
            m_propertiesFrom=null;
            m_propertiesTo=null;
            m_instances=null;
            m_spanningInstance=null;
        }
    }
    /**
     * Returns the bit mask specifying which parts of the object have been loaded.
     *
     * @return                              the bit mask specifying which parts of the object have been loaded
     */
    public int getLoadedState() {
        synchronized (getLock()) {
            int loadedState=0;
            if (hasLoadedBasics())
                loadedState|=OIModel.LOAD_CONCEPT_BASICS;
            if (m_superConcepts!=null)
                loadedState|=OIModel.LOAD_SUPER_CONCEPTS;
            if (m_subConcepts!=null)
                loadedState|=OIModel.LOAD_SUB_CONCEPTS;
            if (m_propertiesFrom!=null)
                loadedState|=OIModel.LOAD_PROPERTIES_FROM;
            if (m_propertiesTo!=null)
                loadedState|=OIModel.LOAD_PROPERTIES_TO;
            if (m_instances!=null)
                loadedState|=OIModel.LOAD_CONCEPT_INSTANCES;
            return loadedState;
        }
    }
    /**
     * Returns all properties starting from this concept.
     *
     * @return                          set of properties starting from this concept (may be empty)
     */
    public Set getPropertiesFromConcept() throws KAONException {
        synchronized (getLock()) {
            if (m_propertiesFrom==null)
                loadThisObject(OIModel.LOAD_CONCEPT_BASICS | OIModel.LOAD_PROPERTIES_FROM);
            return m_propertiesFrom;
        }
    }
    /**
     * Returns all properties (including inherited ones) starting from this concept.
     *
     * @return                          set of all properties starting from this concept (may be empty)
     */
    public Set getAllPropertiesFromConcept() throws KAONException {
        synchronized (getLock()) {
            // if there are hierarcy cycles, this method will loop forever
            Set result=new HashSet(getPropertiesFromConcept());
            Iterator superConcepts=getSuperConcepts().iterator();
            while (superConcepts.hasNext()) {
                Concept concept=(Concept)superConcepts.next();
                result.addAll(concept.getAllPropertiesFromConcept());
            }
            return result;
        }
    }
    /**
     * Returns all properties pointing to this concept.
     *
     * @return                          set of properties poining to this concept (may be empty)
     */
    public Set getPropertiesToConcept() throws KAONException {
        synchronized (getLock()) {
            if (m_propertiesTo==null)
                loadThisObject(OIModel.LOAD_CONCEPT_BASICS | OIModel.LOAD_PROPERTIES_TO);
            return m_propertiesTo;
        }
    }
    /**
     * Returns all properties (including inherited ones) pointing to this concept.
     *
     * @return                          set of all properties pointing to this concept (may be empty)
     */
    public Set getAllPropertiesToConcept() throws KAONException {
        synchronized (getLock()) {
            // if there are hierarcy cycles, this method will loop forever
            Set result=new HashSet(getPropertiesToConcept());
            Iterator superConcepts=getSuperConcepts().iterator();
            while (superConcepts.hasNext()) {
                Concept concept=(Concept)superConcepts.next();
                result.addAll(concept.getAllPropertiesToConcept());
            }
            return result;
        }
    }
    /**
     * Returns direct subconcepts of this concept.
     *
     * @return                          subconcepts of this concept
     */
    public Set getSubConcepts() throws KAONException {
        synchronized (getLock()) {
            if (m_subConcepts==null)
                loadThisObject(OIModel.LOAD_CONCEPT_BASICS | OIModel.LOAD_SUB_CONCEPTS);
            return m_subConcepts;
        }
    }
    /**
     * Returns all subconcepts (inherited as well) of this concept.
     *
     * @return                          subconcepts of this concept
     */
    public Set getAllSubConcepts() throws KAONException {
        synchronized (getLock()) {
            // if there are hierarcy cycles, this method will loop forever
            Set subConcepts=getSubConcepts();
            Set result=new HashSet(subConcepts);
            Iterator iterator=subConcepts.iterator();
            while (iterator.hasNext()) {
                Concept concept=(Concept)iterator.next();
                result.addAll(concept.getAllSubConcepts());
            }
            return result;
        }
    }
    /**
     * Tests if this concept is a direct subconcept of given concept. If concepts are the same, this method returns <code>false</code>.
     *
     * @param potentialSuperConcept     concept being tested
     * @return                          <code>true</code> if this concept is a direct subconcept
     */
    public boolean isDirectSubConceptOf(Concept potentialSuperConcept) throws KAONException {
        synchronized (getLock()) {
            return getSuperConcepts().contains(potentialSuperConcept);
        }
    }
    /**
     * Returns the OI-model containing the concept - superconcept association.
     *
     * @param superConcept              the superconcept of this concept
     * @return                          the OI-model in which the super-subconcept association is declared
     */
    public OIModel getSuperSubConceptOIModel(Concept superConcept) throws KAONException {
        synchronized (getLock()) {
            if (m_superConcepts==null)
                loadThisObject(OIModel.LOAD_CONCEPT_BASICS | OIModel.LOAD_SUPER_CONCEPTS);
            return (OIModel)m_superConcepts.get(superConcept);
        }
    }
    /**
     * Returns <code>true</code> if this concept has been declared a subconcept of given concept in this OI-model.
     *
     * @param superConcept              the superconcept of this concept
     * @return                          <code>true</code> if this concept has been declared a subconcept of given concept in this OI-model
     */
    public boolean isSuperSubConceptDeclaredLocally(Concept superConcept) throws KAONException {
        synchronized (getLock()) {
            return getSuperSubConceptOIModel(superConcept)==m_oimodel;
        }
    }
    /**
     * Tests if this concept is a subconcept of given concept. If concepts are the same, this method returns <code>false</code>.
     *
     * @param potentialSuperConcept     concept being tested
     * @return                          <code>true</code> if this concept is a subconcept
     */
    public boolean isSubConceptOf(Concept potentialSuperConcept) throws KAONException {
        synchronized (getLock()) {
            return getAllSuperConcepts().contains(potentialSuperConcept);
        }
    }
    /**
     * Returns direct superconcepts of this concept.
     *
     * @return                          superconcepts of this concept
     */
    public Set getSuperConcepts() throws KAONException {
        synchronized (getLock()) {
            if (m_superConcepts==null)
                loadThisObject(OIModel.LOAD_CONCEPT_BASICS | OIModel.LOAD_SUPER_CONCEPTS);
            return m_superConcepts.keySet();
        }
    }
    /**
     * Returns all superconcepts (inherited as well) of this concept.
     *
     * @return                          superconcepts of this concept
     */
    public Set getAllSuperConcepts() throws KAONException {
        synchronized (getLock()) {
            // if there are hierarcy cycles, this method will loop forever
            Set superConcepts=getSuperConcepts();
            Set result=new HashSet(superConcepts);
            Iterator iterator=superConcepts.iterator();
            while (iterator.hasNext()) {
                Concept concept=(Concept)iterator.next();
                result.addAll(concept.getAllSuperConcepts());
            }
            return result;
        }
    }
    /**
     * Returns the set of direct instances of this concept.
     *
     * @return                          set of instances of this concept
     */
    public Set getInstances() throws KAONException {
        synchronized (getLock()) {
            if (m_instances==null)
                loadThisObject(OIModel.LOAD_CONCEPT_BASICS | OIModel.LOAD_CONCEPT_INSTANCES);
            return m_instances;
        }
    }
    /**
     * Returns the set of all instances (even of subconcepts) of a given concept in the pool.
     *
     * @return                          set of all instances of this concept
     */
    public Set getAllInstances() throws KAONException {
        synchronized (getLock()) {
            // if there are hierarcy cycles, this method will loop forever
            Set result=new HashSet(getInstances());
            Set subConcepts=getSubConcepts();
            Iterator iterator=subConcepts.iterator();
            while (iterator.hasNext()) {
                Concept subConcept=(Concept)iterator.next();
                result.addAll(subConcept.getAllInstances());
            }
            return result;
        }
    }
    /**
     * Returns the instance associated with this object.
     *
     * @return                              instance associated with this object
     */
    public Instance getSpanningInstance() {
        synchronized (getLock()) {
            if (m_spanningInstance==null)
                m_spanningInstance=m_oimodel.getInstance(m_uri);
            return m_spanningInstance;
        }
    }
    /**
     * Makes sure that the basics of this object are loaded.
     */
    protected void makeSureBasicsLoaded() throws KAONException {
        if (!hasLoadedBasics())
            loadThisObject(OIModel.LOAD_CONCEPT_BASICS);
    }
    /**
     * Accepts an entity visitor.
     *
     * @param visitor                   visitor to apply
     */
    public void accept(EntityVisitor visitor) throws KAONException {
        visitor.visit(this);
    }
    /**
     * Sets the flag that determines whether this entity is in the model.
     *
     * @param isInOIModel               flag determining whether this entity is in the model
     * @param sourceOIModel             the source OI-model
     */
    void setIsInOIModel(boolean isInOIModel,OIModel sourceOIModel) throws KAONException {
        super.setIsInOIModel(isInOIModel,sourceOIModel);
        if (!isInOIModel) {
            if (m_superConcepts!=null)
                m_superConcepts.clear();
            else
                m_superConcepts=new HashMap();
            if (m_subConcepts!=null)
                m_subConcepts.clear();
            else
                m_subConcepts=new HashSet();
            if (m_propertiesFrom!=null)
                m_propertiesFrom.clear();
            else
                m_propertiesFrom=new HashSet();
            if (m_propertiesTo!=null)
                m_propertiesTo.clear();
            else
                m_propertiesTo=new HashSet();
            if (m_instances!=null)
                m_instances.clear();
            else
                m_instances=new HashSet();
        }
    }
    /**
     * Loads this object.
     *
     * @param conceptInfo               the information about a concept
     */
    void loadObject(ConceptInfo conceptInfo) throws KAONException {
        checkVersions(conceptInfo.m_id);
        if (conceptInfo.m_modelID!=-1)
            loadBasics(conceptInfo);
        if (conceptInfo.m_superConceptIDs!=null && m_superConcepts==null)
            m_superConcepts=loadConcepts(conceptInfo.m_superConceptIDs);
        if (conceptInfo.m_subConceptIDs!=null && m_subConcepts==null)
            m_subConcepts=loadConcepts(conceptInfo.m_subConceptIDs);
        if (conceptInfo.m_propertiesFromIDs!=null && m_propertiesFrom==null)
            m_propertiesFrom=loadProperties(conceptInfo.m_propertiesFromIDs);
        if (conceptInfo.m_propertiesToIDs!=null && m_propertiesTo==null)
            m_propertiesTo=loadProperties(conceptInfo.m_propertiesToIDs);
        if (conceptInfo.m_instanceIDs!=null && m_instances==null)
            m_instances=loadInstances(conceptInfo.m_instanceIDs);
    }
    /**
     * Adds the superconcept to this concept.
     *
     * @param superConcept              the superconcept to add
     * @param sourceOIModel             the OI-model where this relationship has been declared
     */
    void addSuperConcept(Concept superConcept,OIModel sourceOIModel) {
        if (m_superConcepts!=null)
            m_superConcepts.put(superConcept,sourceOIModel);
    }
    /**
     * Removes the superconcept from this concept.
     *
     * @param superConcept              the superconcept to remove
     */
    void removeSuperConcept(Concept superConcept) {
        if (m_superConcepts!=null)
            m_superConcepts.remove(superConcept);
    }
    /**
     * Adds the subconcept to this concept.
     *
     * @param subConcept                the subconcept to add
     */
    void addSubConcept(Concept subConcept) {
        if (m_subConcepts!=null)
            m_subConcepts.add(subConcept);
    }
    /**
     * Removes the subconcept from this concept.
     *
     * @param subConcept                the sunconcept to remove
     */
    void removeSubConcept(Concept subConcept) {
        if (m_subConcepts!=null)
            m_subConcepts.remove(subConcept);
    }
    /**
     * Adds the property from this concept.
     *
     * @param property                  the property to add
     */
    void addPropertyFrom(Property property) {
        if (m_propertiesFrom!=null)
            m_propertiesFrom.add(property);
    }
    /**
     * Removes the property from this concept.
     *
     * @param property                  the property to remove
     */
    void removePropertyFrom(Property property) {
        if (m_propertiesFrom!=null)
            m_propertiesFrom.remove(property);
    }
    /**
     * Adds the property to this concept.
     *
     * @param property                  the property to add
     */
    void addPropertyTo(Property property) {
        if (m_propertiesTo!=null)
            m_propertiesTo.add(property);
    }
    /**
     * Removes the property to this concept.
     *
     * @param property                  the property to remove
     */
    void removePropertyTo(Property property) {
        if (m_propertiesTo!=null)
            m_propertiesTo.remove(property);
    }
    /**
     * Adds the instance to this concept.
     *
     * @param instance                  the instance to add
     */
    void addInstance(Instance instance) {
        if (m_instances!=null)
            m_instances.add(instance);
    }
    /**
     * Removes the instance from this concept.
     *
     * @param instance                  the instance to remove
     */
    void removeInstance(Instance instance) {
        if (m_instances!=null)
            m_instances.remove(instance);
    }
}
