package edu.unika.aifb.kaon.apiproxy;

import java.util.Set;
import java.util.HashSet;
import java.util.Map;
import java.util.Iterator;
import java.util.Collections;
import java.io.Serializable;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.vocabulary.*;

import edu.unika.aifb.kaon.apiproxy.source.*;

/**
 * Abstract entity in the OI-model.
 */
public abstract class AbstractEntityProxy implements Entity,Serializable {
    /** Signals that the current version of the entity is unknown. */
    protected static final int VERSION_UNKNOWN=-2;
    /** The servial version. */
    public static final long serialVersionUID=2413616562511050725L;
    /** OI-model of this entity. */
    transient protected OIModelProxy m_oimodel;
    /** The source OI-model of this entity. */
    transient protected OIModel m_sourceOIModel;
    /** The URI of this entity. */
    protected String m_uri;
    /**
     * The version of this entity. This field also encodes whether the object was loaded at all.
     * If this field is negative or zero, this means that the object hasn't been loaded.
     **/
    protected volatile int m_version;
    /** Flag specifying whether this entity is in OI-model. */
    transient protected boolean m_isInOIModel;

    /**
     * Creates an instance of this class.
     *
     * @param oimodel                   OI-model where this entity lives
     * @param uri                       URI of the concept
     */
    public AbstractEntityProxy(OIModelProxy oimodel,String uri) {
        m_oimodel=oimodel;
        m_uri=uri;
        unload();
    }
    /**
     * This constructor initializes only the URI and the version, and should be used exclusively for
     * creating dummy objects for passing them from server.
     *
     * @param uri                       URI of the concept
     * @param version                   the version of the object
     */
    public AbstractEntityProxy(String uri,int version) {
        m_uri=uri;
        if (version!=VERSION_UNKNOWN)
            m_version=version;
    }
    /**
     * Unloads this object.
     */
    public void unload() {
        synchronized (getLock()) {
            m_version=0;
            m_sourceOIModel=null;
            m_isInOIModel=false;
        }
    }
    /**
     * Returns the OI-model of this object.
     *
     * @return                          OI-model of this object
     */
    public OIModel getOIModel() {
        return m_oimodel;
    }
    /**
     * Returns <code>true</code> if this object is in the OI-model.
     *
     * @return                          <code>true</code> if this object in in the OI-model
     */
    public boolean isInOIModel() throws KAONException {
        synchronized (getLock()) {
            makeSureBasicsLoaded();
            return m_isInOIModel;
        }
    }
    /**
     * Returns <code>true</code> if this entity has been declared in the OI-model as returned
     * by <code>getOIModel()</code> call. Note that this may return <code>false</code> if this
     * entity was declared in an included OI-model.
     *
     * @return                          <code>true</code> if this entity case declared in the OI-model as returned by <code>getOIModel()</code> call
     */
    public boolean isDeclaredLocally() throws KAONException {
        synchronized (getLock()) {
            return getSourceOIModel()==m_oimodel;
        }
    }
    /**
     * Returns the OI-model where this entity was declared originally.
     *
     * @return                          the OI-model there this entity was declated originally
     */
    public OIModel getSourceOIModel() throws KAONException {
        synchronized (getLock()) {
            makeSureBasicsLoaded();
            return m_sourceOIModel;
        }
    }
    /**
     * Returns the URI of this entity.
     *
     * @return                          URI of this entity
     */
    public String getURI() {
        return m_uri;
    }
    /**
     * Returns the version of this entity.
     *
     * @return                          the version of this entity
     */
    public int getVersion() {
        // we read the version first into a local variable since m_version is volatile (thus we avoid the need for synchronization)
        int version=m_version;
        return version<0 ? -version : version;
    }
    /**
     * Returns whether this object's basics were loaded at all.
     *
     * @return                          <code>true</code> if the object's basics were loaded
     */
    public boolean hasLoadedBasics() {
        return m_version>0;
    }
    /**
     * Makes sure that the basics of this object are loaded.
     */
    protected abstract void makeSureBasicsLoaded() throws KAONException;
    /**
     * Returns all lexical entries associated with this entity.
     *
     * @return                              set of lexical entries associated with this entity
     */
    public Set getLexicalEntries() throws KAONException {
        synchronized (getLock()) {
            // This is a hack! Its purpose ist to compensate for the case when kaon:en or similar elements
            // are asked for a label. Since many objects point to these instances, getToPropertyValues()
            // takes a very long time. This can be remedied by the query language.
            if (getURI().startsWith(KAONConnection.LEXICAL_OIMODEL_URI))
                return Collections.EMPTY_SET;
            else {
                Property references=m_oimodel.getProperty(KAONVocabularyAdaptor.INSTANCE.getReferences());
                return getSpanningInstance().getToPropertyValues(references);
            }
        }
    }
    /**
     * Retruns all lexical entries of specified type associated with this entity.
     *
     * @param typeURI                       URI of the lexical entry type
     * @return                              set of lexical entries of given type
     */
    public Set getLexicalEntries(String typeURI) throws KAONException {
        synchronized (getLock()) {
            // This is a hack! Its purpose ist to compensate for the case when kaon:en or similar elements
            // are asked for a label. Since many objects point to these instances, getToPropertyValues()
            // takes a very long time. This can be remedied by the query language.
            if (getURI().startsWith(KAONConnection.LEXICAL_OIMODEL_URI))
                return Collections.EMPTY_SET;
            else {
                Property references=m_oimodel.getProperty(KAONVocabularyAdaptor.INSTANCE.getReferences());
                Set set=new HashSet();
                Iterator iterator=getSpanningInstance().getToPropertyInstances().iterator();
                while (iterator.hasNext()) {
                    PropertyInstance propertyInstance=(PropertyInstance)iterator.next();
                    if (propertyInstance.getProperty().equals(references)) {
                        LexicalEntry lexicalEntry=(LexicalEntry)propertyInstance.getSourceInstance();
                        if (lexicalEntry.getTypeURI().equals(typeURI))
                            set.add(lexicalEntry);
                    }
                }
                return set;
            }
        }
    }
    /**
     * Retruns all lexical entries of specified type associated with this entity.
     *
     * @param typeURI                       URI of the lexical entry type
     * @param languageURI                   URI of the language
     * @return                              set of lexical entries of given type
     */
    public Set getLexicalEntries(String typeURI,String languageURI) throws KAONException {
        synchronized (getLock()) {
            // This is a hack! Its purpose ist to compensate for the case when kaon:en or similar elements
            // are asked for a label. Since many objects point to these instances, getToPropertyValues()
            // takes a very long time. This can be remedied by the query language.
            if (getURI().startsWith(KAONConnection.LEXICAL_OIMODEL_URI))
                return Collections.EMPTY_SET;
            else {
                Property references=m_oimodel.getProperty(KAONVocabularyAdaptor.INSTANCE.getReferences());
                Set set=new HashSet();
                Iterator iterator=getSpanningInstance().getToPropertyInstances().iterator();
                while (iterator.hasNext()) {
                    PropertyInstance propertyInstance=(PropertyInstance)iterator.next();
                    if (propertyInstance.getProperty().equals(references)) {
                        LexicalEntry lexicalEntry=(LexicalEntry)propertyInstance.getSourceInstance();
                        if (lexicalEntry.getTypeURI().equals(typeURI) && lexicalEntry.getInLanguage().equals(languageURI))
                            set.add(lexicalEntry);
                    }
                }
                return set;
            }
        }
    }
    /**
     * Retruns a lexical entry of specified type and in given language associated with this entity. If there are more
     * lexical entries that match the criteria, one is picked and returned. If there are no lexical entries, <code>null</code>
     * is returned.
     *
     * @param typeURI                       URI of the lexical entry type
     * @param languageURI                   URI of the language
     * @return                              lexical entries of given type and language, or <code>null</code> if no entry is found
     */
    public LexicalEntry getLexicalEntry(String typeURI,String languageURI) throws KAONException {
        synchronized (getLock()) {
            // This is a hack! Its purpose ist to compensate for the case when kaon:en or similar elements
            // are asked for a label. Since many objects point to these instances, getToPropertyValues()
            // takes a very long time. This can be remedied by the query language.
            if (getURI().startsWith(KAONConnection.LEXICAL_OIMODEL_URI))
                return null;
            else {
                Property references=m_oimodel.getProperty(KAONVocabularyAdaptor.INSTANCE.getReferences());
                Iterator iterator=getSpanningInstance().getToPropertyInstances().iterator();
                while (iterator.hasNext()) {
                    PropertyInstance propertyInstance=(PropertyInstance)iterator.next();
                    if (propertyInstance.getProperty().equals(references)) {
                        LexicalEntry lexicalEntry=(LexicalEntry)propertyInstance.getSourceInstance();
                        if (lexicalEntry.getTypeURI().equals(typeURI) && languageURI.equals(lexicalEntry.getInLanguage()))
                            return lexicalEntry;
                    }
                }
                return null;
            }
        }
    }
    /**
     * Returns the value of the lexical entry of given type in given language with given attribute name. If no appropriate
     * information is found, <code>null</code> is returned.
     *
     * @param typeURI                       URI of the lexical entry type
     * @param languageURI                   URI of the language
     * @param attributeURI                  URI name of the attribute that is requested
     * @return                              value of given attribute
     */
    public String getLexicalAttribute(String typeURI,String languageURI,String attributeURI) throws KAONException {
        synchronized (getLock()) {
            LexicalEntry lexicalEntry=getLexicalEntry(typeURI,languageURI);
            if (lexicalEntry==null)
                return null;
            else {
                Object value=lexicalEntry.getAttribute(attributeURI);
                if (value instanceof String)
                    return (String)value;
                else
                    return null;
            }
        }
    }
    /**
     * Returns the label in given language.
     *
     * @param languageURI                   URI of the language
     * @return                              label in given language
     */
    public String getLabel(String languageURI) throws KAONException {
        synchronized (getLock()) {
            return getLexicalAttribute(KAONVocabularyAdaptor.INSTANCE.getKAONLabel(),languageURI,KAONVocabularyAdaptor.INSTANCE.getValue());
        }
    }
    /**
     * Loads this object. If the object is already loaded, then the version number is only compared.
     *
     * @param loadFlag                      the flag specifying which data should be loaded
     */
    protected void loadThisObject(int loadFlag) throws KAONException {
        m_oimodel.loadObjects(Collections.singleton(this),loadFlag);
    }
    /**
     * Sets the flag that determines whether this entity is in the model.
     *
     * @param isInOIModel               flag determining whether this entity is in the model
     * @param sourceOIModel             the source OI-model
     */
    void setIsInOIModel(boolean isInOIModel,OIModel sourceOIModel) throws KAONException {
        m_isInOIModel=isInOIModel;
        if (m_isInOIModel)
            m_sourceOIModel=sourceOIModel;
        else
            m_sourceOIModel=null;
    }
    /**
     * Checks the version of this object. If this object doesn't have a version, then then version is recorded.
     * Otherwise, if the versions do not match, then an exception is thrown.
     *
     * @param entityID                  the ID of the entity
     */
    void checkVersions(EntityID entityID) throws KAONException {
        if (entityID.m_version!=VERSION_UNKNOWN) {
            // if version is 0 this means the object is not loaded and hasn't been touched
            if (m_version==0)
                m_version=-entityID.m_version;
            else if (getVersion()!=entityID.m_version)
                throw new StaleDataException(this);
        }
    }
    /**
     * Updates the version of this object.
     *
     * @param entityID                  the ID of the entity
     */
    void updateVersion(EntityID entityID) {
        if (entityID.m_version!=VERSION_UNKNOWN) {
            if (hasLoadedBasics())
                m_version=entityID.m_version;
            else
                m_version=-entityID.m_version;
        }
    }
    /**
     * Loads the basics of this object.
     *
     * @param entityInfo                the information about object
     */
    void loadBasics(EntityInfo entityInfo) throws KAONException {
        if (!hasLoadedBasics()) {
            if (m_version<0 && entityInfo.m_id.m_version!=VERSION_UNKNOWN)
                m_version=entityInfo.m_id.m_version;
            m_isInOIModel=entityInfo.m_id.m_version!=0 && Math.abs(entityInfo.m_id.m_version)!=Integer.MAX_VALUE;
            m_sourceOIModel=m_oimodel.getOIModel(entityInfo.m_modelID);
        }
    }
    /**
     * Notifies this object that a load request for it has been shceduled, but no data arrived. This either
     * means that the object didn't exist, or that it was deleted. If a local version exists for the object,
     * then this means the object was deleted, and an execption is thrown. Otherwise the object is initialized as if
     * it is not in the OI-model.
     */
    void loadRequestedButNoDataReceived() throws KAONException {
        if (m_version!=0 && getVersion()!=Integer.MAX_VALUE)
            throw new StaleDataException(this);
        if (hasLoadedBasics())
            m_version=Integer.MAX_VALUE;
        else
            m_version=-Integer.MAX_VALUE;
        setIsInOIModel(false,null);
    }
    /**
     * Loads the set of concepts specified by the array of entity IDs.
     *
     * @param conceptIDs                the array of concept entity IDs
     */
    protected Set loadConcepts(EntityID[] conceptIDs) throws KAONException {
        return m_oimodel.loadConcepts(conceptIDs);
    }
    /**
     * Loads the set of properties specified by the array of entity IDs.
     *
     * @param propertyIDs               the array of property entity IDs
     */
    protected Set loadProperties(EntityID[] propertyIDs) throws KAONException {
        return m_oimodel.loadProperties(propertyIDs);
    }
    /**
     * Loads the set of instances specified by the array of entity IDs.
     *
     * @param instanceIDs               the array of instance entity IDs
     */
    protected Set loadInstances(EntityID[] instanceIDs) throws KAONException {
        return m_oimodel.loadInstances(instanceIDs);
    }
    /**
     * Loads the map of concepts specified by the array of entity IDs.
     *
     * @param relationConceptIDs                the array of concept relation entity IDs
     */
    protected Map loadConcepts(RelationEntityID[] relationConceptIDs) throws KAONException {
        return m_oimodel.loadConcepts(relationConceptIDs);
    }
    /**
     * Loads the map of properties specified by the array of entity IDs.
     *
     * @param relationPropertyIDs               the array of property relation entity IDs
     */
    protected Map loadProperties(RelationEntityID[] relationPropertyIDs) throws KAONException {
        return m_oimodel.loadProperties(relationPropertyIDs);
    }
    /**
     * Returns the lock object for this entity.
     *
     * @return                              the lock object for the entity
     */
    protected Object getLock() {
        return m_oimodel.m_kaonConnection;
    }
}
