package edu.unika.aifb.kaon.apionrdf.importer;

import java.util.Set;
import java.util.HashSet;
import java.util.Iterator;
import org.xml.sax.SAXException;

import edu.unika.aifb.rdf.api.model.*;

/**
 * Model consumer that converts RDFS ontologies into KAON ontologies during ontology parsing.
 */
public class RDFSImporter extends AbstractImporter {
    /** Set to <code>true</code> if rdfs:comment should be interpreted as a label. */
    protected boolean m_rdfsCommentAsLabel;
    /** Set of classes that have been seen but that don't have a parent. */
    protected Set m_classesWithoutParent;
    /** Set of classes whose parent has been seen but that haven't been parsed yet. */
    protected Set m_notSeenClassesWithParent;
    /** Set of properties what have been seen but that don't have a domain. */
    protected Set m_propertiesWithoutDomain;
    /** Set of properties whose domain has been seen but that haven't been parsed yet. */
    protected Set m_nonSeenPropertiesWithDomain;
    /** Set of properties what have been seen but that don't have a range. */
    protected Set m_propertiesWithoutRange;
    /** Set of properties whose range has been seen but that haven't been parsed yet. */
    protected Set m_nonSeenPropertiesWithRange;

    /**
     * Creates an instance of this class.
     *
     * @param rdfsCommentAsLabel    if <code>true</code>, RDFS comments are treated as labels
     */
    public RDFSImporter(boolean rdfsCommentAsLabel) {
        m_rdfsCommentAsLabel=rdfsCommentAsLabel;
    }
    /**
     * Called when model parsing is started.
     *
     * @param physicalURI           physical URI of the model
     * @throws SAXException         thrown if there is an error
     */
    public void startModel(String physicalURI) throws SAXException {
        super.startModel(physicalURI);
        m_classesWithoutParent=new HashSet();
        m_notSeenClassesWithParent=new HashSet();
        m_propertiesWithoutDomain=new HashSet();
        m_nonSeenPropertiesWithDomain=new HashSet();
        m_propertiesWithoutRange=new HashSet();
        m_nonSeenPropertiesWithRange=new HashSet();
    }
    /**
     * Called when model parsing is finished.
     *
     * @throws SAXException         thrown if there is an error
     */
    public void endModel() throws SAXException {
        super.endModel();
        try {
            Iterator iterator=m_classesWithoutParent.iterator();
            while (iterator.hasNext()) {
                String className=(String)iterator.next();
                createStatement(className,RDFSNS+"subClassOf",KAONROOT);
            }
            iterator=m_propertiesWithoutDomain.iterator();
            while (iterator.hasNext()) {
                String propertyName=(String)iterator.next();
                createStatement(propertyName,RDFSNS+"domain",KAONROOT);
            }
            iterator=m_propertiesWithoutRange.iterator();
            while (iterator.hasNext()) {
                String propertyName=(String)iterator.next();
                createStatement(propertyName,RDFSNS+"range",KAONROOT);
            }
        }
        catch (ModelException e) {
            throw new SAXException("RDF error.",e);
        }
    }
    /**
     * Called when a statement with resource value is added to the model.
     *
     * @param subject               URI of the subject resource
     * @param predicate             URI of the predicate resource
     * @param object                URI of the object resource
     * @throws SAXException         thrown if there is an error
     */
    public void statementWithResourceValue(String subject,String predicate,String object) throws SAXException {
        if (isClassDeclaration(subject,predicate,object)) {
            if (!m_notSeenClassesWithParent.remove(subject))
                m_classesWithoutParent.add(subject);
        }
        else if (isSubclassDeclaration(subject,predicate,object)) {
            if (!m_classesWithoutParent.remove(subject))
                m_notSeenClassesWithParent.add(subject);
        }
        else if (isPropertyDeclaration(subject,predicate,object)) {
            if (!m_nonSeenPropertiesWithDomain.remove(subject))
                m_propertiesWithoutDomain.add(subject);
            if (!m_nonSeenPropertiesWithRange.remove(subject))
                m_propertiesWithoutRange.add(subject);
        }
        else if (isPropertyDomainDeclaration(subject,predicate,object)) {
            if (!m_propertiesWithoutDomain.remove(subject))
                m_nonSeenPropertiesWithDomain.add(subject);
        }
        else if (isPropertyRangeDeclaration(subject,predicate,object)) {
            if (!m_propertiesWithoutRange.remove(subject))
                m_nonSeenPropertiesWithRange.add(subject);
        }
        super.statementWithResourceValue(subject,predicate,object);
    }
    /**
     * Called when a statement with literal value is added to the model.
     *
     * @param subject               URI of the subject resource
     * @param predicate             URI of the predicate resource
     * @param object                literal object value
     * @param language              the language
     * @param dataType              the data type
     * @throws SAXException         thrown if there is an error
     */
    public void statementWithLiteralValue(String subject,String predicate,String object,String language,String dataType) throws SAXException {
        try {
            if (predicate.equals(RDFSNS+"comment")) {
                if (m_rdfsCommentAsLabel)
                    createLexicalEntry(KAONNS+"Label",language,subject,predicate,object);
                else
                    createLexicalEntry(KAONNS+"Documentation",language,subject,predicate,object);
            }
            else if (predicate.equals(RDFSNS+"label"))
                createLexicalEntry(KAONNS+"Label",language,subject,predicate,object);
            else
                super.statementWithLiteralValue(subject,predicate,object,language,dataType);
        }
        catch (ModelException e) {
            throw new SAXException("RDF exception",e);
        }
    }
    /**
     * Returns <code>true<code> if this is a class declaration.
     *
     * @param subject               URI of the subject resource
     * @param predicate             URI of the predicate resource
     * @param object                URI of the object resource
     * @return                      <code>true</code> if this is a class declaration
     */
    protected boolean isClassDeclaration(String subject,String predicate,String object) {
        return predicate.equals(RDFTYPE) && object.equals(RDFSNS+"Class");
    }
    /**
     * Returns <code>true<code> if this is a subclass declaration.
     *
     * @param subject               URI of the subject resource
     * @param predicate             URI of the predicate resource
     * @param object                URI of the object resource
     * @return                      <code>true</code> if this is a subclass declaration
     */
    protected boolean isSubclassDeclaration(String subject,String predicate,String object) {
        return predicate.equals(RDFSNS+"subClassOf");
    }
    /**
     * Returns <code>true</code> if this is a property declaration.
     *
     * @param subject               URI of the subject resource
     * @param predicate             URI of the predicate resource
     * @param object                URI of the object resource
     * @return                      <code>true</code> if this is a property declaration
     */
    protected boolean isPropertyDeclaration(String subject,String predicate,String object) {
        return predicate.equals(RDFTYPE) && object.equals(RDFNS+"Property");
    }
    /**
     * Returns <code>true<code> if this is a property domain declaration.
     *
     * @param subject               URI of the subject resource
     * @param predicate             URI of the predicate resource
     * @param object                URI of the object resource
     * @return                      <code>true</code> if this is a subclass declaration
     */
    protected boolean isPropertyDomainDeclaration(String subject,String predicate,String object) {
        return predicate.equals(RDFSNS+"domain");
    }
    /**
     * Returns <code>true<code> if this is a property range declaration.
     *
     * @param subject               URI of the subject resource
     * @param predicate             URI of the predicate resource
     * @param object                URI of the object resource
     * @return                      <code>true</code> if this is a subclass declaration
     */
    protected boolean isPropertyRangeDeclaration(String subject,String predicate,String object) {
        return predicate.equals(RDFSNS+"range");
    }
}
