package edu.unika.aifb.kaon.apionrdf.importer;

import java.util.Set;
import java.util.HashSet;
import org.xml.sax.SAXException;

import edu.unika.aifb.rdf.api.model.*;

/**
 * Model consumer that converts Protege ontologies into KAON ontologies during ontology parsing.
 */
public class ProtegeImporter extends AbstractImporter {
    protected static final String PROTEGE_RDFSNS="http://www.w3.org/TR/1999/PR-rdf-schema-19990303#";
    protected static final String PROTEGENS="http://protege.stanford.edu/system#";

    protected static final String PROTEGEINVERSE=PROTEGENS+"inverseProperty";
    protected static final String KAONINVERSE=KAONNS+"inverse";

    protected static final String PROTEGEROOT=PROTEGE_RDFSNS+"Resource";
    protected static final String PROTEGEMETAROOT=PROTEGE_RDFSNS+"Class";
    protected static final String KAONROOT="http://kaon.semanticweb.org/2001/11/kaon-lexical#Root";

    /** The set of properties to be skipped when parsing the ontology. */
    protected static final Set s_skipSet=new HashSet();
    static {
        s_skipSet.add(PROTEGENS+"role");
        s_skipSet.add(PROTEGENS+"maxCardinality");
        s_skipSet.add(PROTEGENS+"minCardinality");
        s_skipSet.add(PROTEGENS+"maxValue");
        s_skipSet.add(PROTEGENS+"minValue");
        s_skipSet.add(PROTEGENS+"allowedValues");
    }

    /** Set to <code>true</code> if rdfs:comment should be interpreted as a label. */
    protected boolean m_rdfsCommentAsLabel;

    /**
     * Creates an instance of this class.
     *
     * @param rdfsCommentAsLabel    if <code>true</code>, RDFS comments are treated as labels
     */
    public ProtegeImporter(boolean rdfsCommentAsLabel) {
        m_rdfsCommentAsLabel=rdfsCommentAsLabel;
    }
    /**
     * Called when model parsing is started.
     *
     * @param physicalURI           physical URI of the model
     * @throws SAXException         thrown if there is an error
     */
    public void startModel(String physicalURI) throws SAXException {
        try {
            super.startModel(physicalURI);
            addProtegeTypeDeclarations();
        }
        catch (ModelException e) {
            throw new SAXException("RDF exception",e);
        }
    }
    /**
     * Called when a statement with resource value is added to the model.
     *
     * @param subject               URI of the subject resource
     * @param predicate             URI of the predicate resource
     * @param object                URI of the object resource
     * @throws SAXException         thrown if there is an error
     */
    public void statementWithResourceValue(String subject,String predicate,String object) throws SAXException {
        try {
            if (predicate.equals(PROTEGE_RDFSNS+"subClassOf")) {
                if (object.equals(PROTEGEROOT))
                    object=KAONROOT;
                else if (object.equals(PROTEGEMETAROOT))
                    object=KAONROOT;
            }
            else if (predicate.equals(PROTEGEINVERSE))
                predicate=KAONINVERSE;
            predicate=replaceNameSpace(predicate,PROTEGE_RDFSNS,RDFSNS);
            object=replaceNameSpace(object,PROTEGE_RDFSNS,RDFSNS);
            if (!s_skipSet.contains(predicate))
                createStatement(subject,predicate,object);
        }
        catch (ModelException e) {
            throw new SAXException("RDF exception.",e);
        }
    }
    /**
     * Called when a statement with literal value is added to the model.
     *
     * @param subject               URI of the subject resource
     * @param predicate             URI of the predicate resource
     * @param object                literal object value
     * @param language              the language
     * @param dataType              the data type
     * @throws SAXException         thrown if there is an error
     */
    public void statementWithLiteralValue(String subject,String predicate,String object,String language,String dataType) throws SAXException {
        try {
            predicate=replaceNameSpace(predicate,PROTEGE_RDFSNS,RDFSNS);
            object=replaceNameSpace(object,PROTEGE_RDFSNS,RDFSNS);
            if (!s_skipSet.contains(predicate)) {
                if (predicate.equals(RDFSNS+"comment")) {
                    if (m_rdfsCommentAsLabel)
                        createLexicalEntry(KAONNS+"Label",language,subject,predicate,object);
                    else
                        createLexicalEntry(KAONNS+"Documentation",language,subject,predicate,object);
                }
                else if (predicate.equals(RDFSNS+"label"))
                    createLexicalEntry(KAONNS+"Label",language,subject,predicate,object);
                else if (predicate.equals(PROTEGENS+"range")) {
                    if (object.equals("symbol"))
                        makeSubProperty(subject,m_convertedNameSpace+"SymbolProperty");
                    else if (object.equals("boolean"))
                        makeSubProperty(subject,m_convertedNameSpace+"BooleanProperty");
                    else if (object.equals("float"))
                        makeSubProperty(subject,m_convertedNameSpace+"FloatProperty");
                    else if (object.equals("cls"))
                        makeSubProperty(subject,m_convertedNameSpace+"ClsProperty");
                    createStatement(subject,predicate,RDFSNS+"Literal");
                }
                else
                    createLiteral(subject,predicate,object);
            }
        }
        catch (ModelException e) {
            throw new SAXException("RDF exception",e);
        }
    }
    /**
     * Adds protege-specific type declarations.
     *
     * @throws ModelException       thrown if there is an error
     */
    protected void addProtegeTypeDeclarations() throws ModelException {
        createStatement(m_convertedNameSpace+"ProtegeProperty",RDFTYPE,RDFPROPERTY);
        createStatement(m_convertedNameSpace+"BooleanProperty",RDFTYPE,RDFPROPERTY);
        createStatement(m_convertedNameSpace+"BooleanProperty",RDFSNS+"subPropertyOf",m_convertedNameSpace+"ProtegeProperty");
        createStatement(m_convertedNameSpace+"FloatProperty",RDFTYPE,RDFPROPERTY);
        createStatement(m_convertedNameSpace+"FloatProperty",RDFSNS+"subPropertyOf",m_convertedNameSpace+"ProtegeProperty");
        createStatement(m_convertedNameSpace+"SymbolProperty",RDFTYPE,RDFPROPERTY);
        createStatement(m_convertedNameSpace+"SymbolProperty",RDFSNS+"subPropertyOf",m_convertedNameSpace+"ProtegeProperty");
    }
    /**
     * Makes supplied property a subproperty of given property.
     *
     * @param property              URI of the property
     * @param superProperty         URI of the superProperty
     * @throws ModelException       thrown if there is an error
     */
    protected void makeSubProperty(String property,String superProperty) throws ModelException {
        createStatement(property,RDFSNS+"subPropertyOf",superProperty);
    }
    /**
     * Replaces the namespace of the URI.
     *
     * @param uri                   the URI whose namespace is to be replaced
     * @param toReplace             the namespace to replace
     * @param replaceWith           the namespace to replace with
     * @return                      the new URI
     */
    protected String replaceNameSpace(String uri,String toReplace,String replaceWith) {
        if (uri.startsWith(toReplace))
            return replaceWith+uri.substring(toReplace.length());
        else
            return uri;
    }
}
