package edu.unika.aifb.kaon.apionrdf.importer;

import org.xml.sax.SAXException;

import edu.unika.aifb.rdf.api.util.*;
import edu.unika.aifb.rdf.api.model.*;

/**
 * An abstract importer containing common methods.
 */
public abstract class AbstractImporter extends ModelConsumer {
    protected static final String RDFNS="http://www.w3.org/1999/02/22-rdf-syntax-ns#";
    protected static final String KAON_ROOT_URI="http://kaon.semanticweb.org/2001/11/kaon-root";
    protected static final String KAON_LEXICAL_URI="http://kaon.semanticweb.org/2001/11/kaon-lexical";
    protected static final String KAONNS=KAON_LEXICAL_URI+"#";
    protected static final String RDFSNS="http://www.w3.org/2000/01/rdf-schema#";

    protected static final String RDFTYPE=RDFNS+"type";
    protected static final String RDFPROPERTY=RDFNS+"Property";

    protected static final String KAONROOT="http://kaon.semanticweb.org/2001/11/kaon-lexical#Root";

    /** The name space of the converted ontology. */
    protected String m_convertedNameSpace;
    /** The counter for the labels. */
    protected int m_labelCounter;

    /**
     * Creates an instance of this class.
     */
    public AbstractImporter() {
    }
    /**
     * Called when model parsing is started.
     *
     * @param physicalURI           physical URI of the model
     * @throws SAXException         thrown if there is an error
     */
    public void startModel(String physicalURI) throws SAXException {
        super.startModel(physicalURI);
        m_convertedNameSpace=physicalURI+"#";
        m_includedModels.add(new String[] { KAON_ROOT_URI,null });
        m_includedModels.add(new String[] { KAON_LEXICAL_URI,null });
    }
    /**
     * Receives the logical URI of the model.
     *
     * @param logicalURI            logcal URI of the model
     * @throws SAXException         thrown if there is an error
     */
    public void logicalURI(String logicalURI) throws SAXException {
        super.logicalURI(logicalURI);
        m_convertedNameSpace=logicalURI+"#";
    }
    /**
     * Creates a lexical entry (in lexical layer) for an entity.
     *
     * @param lexicalEntryType      the type of the leixcal entry
     * @param language              the language of the leixcal entry
     * @param subject               URI of the subject resource
     * @param predicate             URI of the predicate resource
     * @param object                URI of the object resource
     * @throws ModelException       thrown if there is an error
     */
    protected void createLexicalEntry(String lexicalEntryType,String language,String subject,String predicate,String object) throws ModelException {
        // create documentation with the reference to the subject
        String newURI=m_convertedNameSpace+(m_labelCounter++);
        createStatement(newURI,RDFTYPE,lexicalEntryType);
        createStatement(newURI,KAONNS+"references",subject);
        createLiteral(newURI,KAONNS+"value",object);
        if (language==null)
            language="en";
        createStatement(newURI,KAONNS+"inLanguage",KAONNS+language);
    }
    /**
     * Adds a statement with resource value to the model.
     *
     * @param subject               URI of the subject resource
     * @param predicate             URI of the predicate resource
     * @param object                URI of the object resource
     * @throws ModelException       thrown if there is an error
     */
    protected void createStatement(String subject,String predicate,String object) throws ModelException {
        NodeFactory nodeFactory=m_model.getNodeFactory();
        Resource subjectResource=nodeFactory.createResource(subject);
        Resource predicateResource=nodeFactory.createResource(predicate);
        Resource objectResource=nodeFactory.createResource(object);
        Statement statement=nodeFactory.createStatement(subjectResource,predicateResource,objectResource);
        m_model.add(statement);
    }
    /**
     * Adds a statement with literal value to the model.
     *
     * @param subject               URI of the subject resource
     * @param predicate             URI of the predicate resource
     * @param object                URI of the object resource
     * @throws ModelException       thrown if there is an error
     */
    protected void createLiteral(String subject,String predicate,String object) throws ModelException {
        NodeFactory nodeFactory=m_model.getNodeFactory();
        Resource subjectResource=nodeFactory.createResource(subject);
        Resource predicateResource=nodeFactory.createResource(predicate);
        Literal objectLiteral=nodeFactory.createLiteral(object);
        Statement statement=nodeFactory.createStatement(subjectResource,predicateResource,objectLiteral);
        m_model.add(statement);
    }
    /**
     * Replaces the namespace of the URI.
     *
     * @param uri                   the URI whose namespace is to be replaced
     * @param toReplace             the namespace to replace
     * @param replaceWith           the namespace to replace with
     * @return                      the new URI
     */
    protected String replaceNameSpace(String uri,String toReplace,String replaceWith) {
        if (uri.startsWith(toReplace))
            return replaceWith+uri.substring(toReplace.length());
        else
            return uri;
    }
}
