package edu.unika.aifb.kaon.apionrdf.exporter;

import java.io.Writer;
import java.io.PrintWriter;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Stack;
import java.util.Iterator;

/**
 * This is a utility class for wiritng XML files.
 */
public class XMLWriter {
    protected static final int IN_TAG_BODY=0;
    protected static final int IN_TAG_BODY_START_LINE=1;
    protected static final int IN_TAG_ATTRIBUTES=2;
    protected static final int IN_TAG_ATTRIBUTES_AFTER_WS=3;

    /** The writer receiving the output. */
    protected PrintWriter m_output;
    /** The encoding to use. */
    protected String m_encoding;
    /** The namespaces for attribute values. */
    protected Namespaces m_namespacesForAttributeValues;
    /** The namespaces for element names. */
    protected Namespaces m_namespacesForElements;
    /** Current indentation level. */
    protected int m_indentation;
    /** The base URI of the document. */
    protected URI m_baseURI;
    /** The names of open tags. */
    protected Stack m_openTagNames;
    /** The stack of indentation of open tags. */
    protected Stack m_openTagIndentation;
    /** The stack specifying whether the close tag should match the opening tag in position. */
    protected Stack m_closeTagIndentation;
    /** Determines the current position of the XML stream. */
    protected int m_currentStreamPosition;

    /**
     * Creates an instance of this class.
     *
     * @param output                        the writer receiving the output (the called should close the writer)
     * @param encoding                      the encoding of the writer
     * @param namespacesForAttributeValues  namespaces object for abbreviating namespaces in attribute values
     * @param namespacesForElements         namespaces object for abbreviating namespaces in element names
     */
    public XMLWriter(Writer output,String encoding,Namespaces namespacesForAttributeValues,Namespaces namespacesForElements) {
        m_output=new PrintWriter(output);
        m_encoding=encoding;
        m_namespacesForAttributeValues=namespacesForAttributeValues;
        m_namespacesForElements=namespacesForElements;
        m_baseURI=null;
        m_openTagNames=new Stack();
        m_openTagIndentation=new Stack();
        m_closeTagIndentation=new Stack();
        m_currentStreamPosition=IN_TAG_BODY_START_LINE;
    }
    /**
     * Returns the namespaces object used for abbreviating attribute values values.
     *
     * @return                              the namespaces for abbreviating attribute values
     */
    public Namespaces getNamespacesForAttributeValues() {
        return m_namespacesForAttributeValues;
    }
    /**
     * Returns the namespaces object used for abbreviating element names.
     *
     * @return                              the namespaces for abbreviating element names
     */
    public Namespaces getNamespacesForElements() {
        return m_namespacesForElements;
    }
    /**
     * Sets the base URI of the document.
     *
     * @param baseURI                       the base URI of the document
     */
    public void setBaseURI(String baseURI) {
        if (baseURI==null)
            m_baseURI=null;
        else
            try {
                m_baseURI=new URI(baseURI);
            }
            catch (URISyntaxException ignored) {
            }
    }
    /**
     * Starts the XML document.
     */
    public void printXMLDeclaration() {
        if (m_currentStreamPosition!=IN_TAG_BODY_START_LINE)
            throw new IllegalStateException("Cannot print XML declaration at this position.");
        m_output.print("<?xml version=\"1.0\" encoding=\"");
        m_output.print(m_encoding);
        m_output.println("\"?>");
    }
    /**
     * Prints the document type and entity declarations from the Namespaces object.
     *
     * @param rootElementName               the name of the root element
     */
    public void printDocumentDeclaration(String rootElementName) {
        if (m_currentStreamPosition!=IN_TAG_BODY_START_LINE)
            throw new IllegalStateException("Cannot print document declaration at this position.");
        m_output.print("<!DOCTYPE ");
        m_output.print(rootElementName);
        boolean isSquareBracketOpen=false;
        Iterator prefixes=m_namespacesForAttributeValues.prefixes();
        while (prefixes.hasNext()) {
            String prefix=(String)prefixes.next();
            String namespace=m_namespacesForAttributeValues.getNamespaceForPrefix(prefix);
            boolean printPrefix=true;
            if (m_baseURI!=null)
                try {
                    URI namespaceURI=new URI(namespace);
                    URI relativizedURI=m_baseURI.relativize(namespaceURI);
                    if (!relativizedURI.equals(namespaceURI))
                        printPrefix=false;
                }
                catch (URISyntaxException ignored) {
                }
            if (printPrefix) {
                if (!isSquareBracketOpen) {
                    isSquareBracketOpen=true;
                    m_output.println(" [");
                }
                m_output.print("    <!ENTITY ");
                m_output.print(prefix);
                m_output.print(" '");
                m_output.print(namespace);
                m_output.println("'>");
            }
        }
        if (isSquareBracketOpen)
            m_output.print("]");
        m_output.println(">");
    }
    /**
     * Prints all namespace declarations.
     */
    public void printNamespaceDeclarations() {
        if (m_currentStreamPosition!=IN_TAG_ATTRIBUTES && m_currentStreamPosition!=IN_TAG_ATTRIBUTES_AFTER_WS)
            throw new IllegalStateException("Cannot print namespaces unless in tag attributes.");
        if (m_namespacesForElements.getDefaultNamespace()!=null) {
            m_output.println();
            indent();
            printAttributeRaw("xmlns",m_namespacesForElements.getDefaultNamespace());
        }
        Iterator prefixes=m_namespacesForElements.prefixes();
        while (prefixes.hasNext()) {
            String prefix=(String)prefixes.next();
            String namespace=m_namespacesForElements.getNamespaceForPrefix(prefix);
            m_output.println();
            indent();
            printAttributeRaw("xmlns:"+prefix,namespace);
        }
    }
    /**
     * Prints the xml:base, if one has been set on this writer.
     */
    public void printXMLBase() {
        if (m_baseURI!=null)
            printAttributeRaw("xml:base",m_baseURI.toString());
    }
    /**
     * Attempts to relativize given URI against the base URI.
     *
     * @param uri                           the URI to be relativized
     * @return                              relativized URI (or original URI if it cannot be relativized)
     */
    protected String relativize(String uri) {
        if (m_baseURI!=null)
            try {
                return m_baseURI.relativize(new URI(uri)).toString();
            }
            catch (URISyntaxException error) {
            }
        return uri;
    }
    /**
     * Opens the tag with given name at the current position. The closing &gt; is not printed, so that
     * attributes can be printed.
     *
     * @param tagName                       the name of the tag
     */
    public void openTagInline(String tagName) {
        ensureTagBody();
        m_output.print('<');
        m_output.print(m_namespacesForElements.abbreviateAsNamespace(tagName));
        m_currentStreamPosition=IN_TAG_ATTRIBUTES;
        m_openTagNames.push(tagName);
        m_openTagIndentation.push(new Integer(-1));
        m_closeTagIndentation.push(Boolean.FALSE);
    }
    /**
     * Opens the tag with given name with indented contents.
     * The closing &gt; is not printed, so that attributes can be printed.
     *
     * @param tagName                       the name of the tag
     * @param matchCloseTagPosition         <code>true</code> if the position of the closing tag should match the position of the opening tag
     */
    public void openTagIndent(String tagName,boolean matchCloseTagPosition) {
        ensureTagBodyNewLine();
        indent();
        m_output.print('<');
        m_output.print(m_namespacesForElements.abbreviateAsNamespace(tagName));
        m_currentStreamPosition=IN_TAG_ATTRIBUTES;
        m_openTagNames.push(tagName);
        m_openTagIndentation.push(new Integer(m_indentation));
        m_closeTagIndentation.push(matchCloseTagPosition ? Boolean.TRUE : Boolean.FALSE);
        increaseIndent();
    }
    /**
     * Closes the last opened tag.
     */
    public void closeLastTag() {
        String tagName=(String)m_openTagNames.pop();
        int indentation=((Integer)m_openTagIndentation.pop()).intValue();
        if (indentation!=-1)
            m_indentation=indentation;
        boolean matchCloseTagPosition=((Boolean)m_closeTagIndentation.pop()).booleanValue();
        if (m_currentStreamPosition==IN_TAG_ATTRIBUTES || m_currentStreamPosition==IN_TAG_ATTRIBUTES_AFTER_WS)
            m_output.print("/>");
        else {
            if (matchCloseTagPosition) {
                ensureTagBodyNewLine();
                indent();
            }
            m_output.print("</");
            m_output.print(m_namespacesForElements.abbreviateAsNamespace(tagName));
            m_output.print('>');
        }
        m_currentStreamPosition=IN_TAG_BODY;
    }
    /**
     * Prints the attribute with given value. Value is NOT abbreviated.
     *
     * @param attributeName                 the name of the attribute
     * @param attributeValue                the value of the attribute
     */
    public void printAttributeRaw(String attributeName,String attributeValue) {
        if (m_currentStreamPosition!=IN_TAG_ATTRIBUTES && m_currentStreamPosition!=IN_TAG_ATTRIBUTES_AFTER_WS)
            throw new IllegalStateException("Attributes cannot be written at this position.");
        if (m_currentStreamPosition!=IN_TAG_ATTRIBUTES_AFTER_WS)
            m_output.print(' ');
        m_output.print(m_namespacesForElements.abbreviateAsNamespaceNoDefault(attributeName));
        m_output.print('=');
        char quote=getQuoteType(attributeValue);
        m_output.print(quote);
        printXMLEncodedValue(attributeValue,quote);
        m_output.print(quote);
    }
    /**
     * Prints the attribute with given URI value.
     *
     * @param attributeName                 the name of the attribute
     * @param attributeValueURI             the URI value of the attribute
     */
    public void printAttributeURI(String attributeName,String attributeValueURI) {
        if (m_currentStreamPosition!=IN_TAG_ATTRIBUTES && m_currentStreamPosition!=IN_TAG_ATTRIBUTES_AFTER_WS)
            throw new IllegalStateException("Attributes cannot be written at this position.");
        if (m_currentStreamPosition!=IN_TAG_ATTRIBUTES_AFTER_WS)
            m_output.print(' ');
        m_output.print(m_namespacesForElements.abbreviateAsNamespaceNoDefault(attributeName));
        m_output.print('=');
        String relativizedURI=relativize(attributeValueURI);
        String abbreviationPrefix=m_namespacesForAttributeValues.getAbbreviationPrefix(relativizedURI);
        if (abbreviationPrefix==null) {
            char quote=getQuoteType(relativizedURI);
            m_output.print(quote);
            printXMLEncodedValue(relativizedURI,quote);
            m_output.print(quote);
        }
        else {
            String valueWithoutAmpersand=abbreviationPrefix+";"+Namespaces.guessLocalName(relativizedURI);
            char quote=getQuoteType(valueWithoutAmpersand);
            m_output.print(quote);
            m_output.print('&');
            printXMLEncodedValue(valueWithoutAmpersand,quote);
            m_output.print(quote);
        }
    }
    /**
     * Returns the type of quote used for the value.
     *
     * @param value                         the value to pring
     * @return                              the type of quote used to encode the value
     */
    protected char getQuoteType(String value) {
        return value.indexOf('"')==-1 ? '"' : (value.indexOf('\'')==-1 ? '\'' : '"');
    }
    /**
     * Prints the value encoded for XML.
     *
     * @param value                         the value
     * @param quote                         the quote
     */
    protected void printXMLEncodedValue(String value,char quote) {
        for (int i=0;i<value.length();i++) {
            char c=value.charAt(i);
            if (c==quote)
                m_output.print(quote=='"' ? "&quot;"  : "&apos;");
            else if (c=='<')
                m_output.print("&lt;");
            else if (c=='&')
                m_output.print("&amp;");
            else
                m_output.print(c);
        }
    }
    /**
     * Makes sure that the current stream position is in tab body.
     */
    public void ensureTagBody() {
        if (m_currentStreamPosition==IN_TAG_ATTRIBUTES || m_currentStreamPosition==IN_TAG_ATTRIBUTES_AFTER_WS) {
            m_output.print('>');
            m_currentStreamPosition=IN_TAG_BODY;
        }
    }
    /**
     * Ensures that the current position is at the new line in the tag body.
     */
    public void ensureTagBodyNewLine() {
        ensureTagBody();
        if (m_currentStreamPosition==IN_TAG_BODY) {
            m_output.println();
            m_currentStreamPosition=IN_TAG_BODY_START_LINE;
        }
    }
    /**
     * Prints given data.
     *
     * @param data                          data to be printed
     */
    public void printData(String data) {
        ensureTagBody();
        boolean hasBreaks=false;
        boolean whiteSpaceOnly=true;
        for (int i=0;i<data.length();i++) {
            char c=data.charAt(i);
            if (c=='\n')
                hasBreaks=true;
            if (!Character.isWhitespace(c))
                whiteSpaceOnly=false;
        }
        if (whiteSpaceOnly || hasBreaks) {
            int start=0;
            int i=data.indexOf("]]>",start);
            m_output.print("<![CDATA[");
            while (i>=0 && start<data.length()) {
                m_output.print(data.substring(start,i));
                m_output.print("]]>]]&#x3e;<![CDATA[");
                start=i+3;
                i=data.indexOf("]]>",start);
            }
            m_output.print(data.substring(start));
            m_output.print("]]>");
        }
        else
            printXMLEncodedValue(data,(char)0);
    }
    /**
     * Indents output appropriately.
     *
     * @param spaces                        the number of spaces for indentation
     */
    public void indent(int spaces) {
        for (int i=0;i<spaces;i++)
            m_output.print(' ');
        if (m_currentStreamPosition==IN_TAG_ATTRIBUTES)
            m_currentStreamPosition=IN_TAG_ATTRIBUTES_AFTER_WS;
    }
    /**
     * Indents output appropriately.
     */
    public void indent() {
        indent(m_indentation);
    }
    /**
     * Increases the indent to the next level.
     */
    public void increaseIndent() {
        m_indentation+=4;
    }
    /**
     * Decreases the indent to the next level.
     */
    public void decreaseIndent() {
        m_indentation-=4;
    }
    /**
     * Inserts a new line.
     */
    public void println() {
        m_output.println();
        if (m_currentStreamPosition==IN_TAG_ATTRIBUTES)
            m_currentStreamPosition=IN_TAG_ATTRIBUTES_AFTER_WS;
        if (m_currentStreamPosition==IN_TAG_BODY)
            m_currentStreamPosition=IN_TAG_BODY_START_LINE;
    }
    /**
     * Flushes the output.
     */
    public void flush() {
        m_output.flush();
    }
}

