package edu.unika.aifb.kaon.apionrdf.exporter;

import java.util.Set;
import java.util.LinkedHashSet;
import java.util.Iterator;
import java.io.IOException;
import java.io.Writer;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.vocabulary.*;

/**
 * An exporter for wiritng OWL XML ontologies.
 */
public class OWLXMLExporter extends AbstractExporter {
    /** The XML writer. */
    protected XMLWriter m_xmlWriter;

    /**
     * Creates an instance of this class.
     */
    public OWLXMLExporter() {
    }
    /**
     * Serializes given RDF model.
     *
     * @param oimodel                   the OI-model to serialize
     * @param physicalURI               the physical URI of the exported file
     * @param writer                    the writer where serialization is performed
     * @param encoding                  tne encoding
     * @throws IOException              thrown if there is an error
     * @throws KAONException            thrown if there is an error accessing the OI-model
     * @throws InterruptedException     thrown if the process is interrupted
     */
    public void export(OIModel oimodel,String physicalURI,Writer writer,String encoding) throws IOException,KAONException,InterruptedException {
        Namespaces namespacesForAttributeValues=new Namespaces();
        namespacesForAttributeValues.ensureNamespacePrefixExists(Namespaces.XSD_NS);
        Namespaces namespacesForElements=new Namespaces();
        namespacesForElements.ensureNamespacePrefixExists(Namespaces.OWLX_NS);
        namespacesForElements.registerPrefix("kaon",KAONVocabularyAdaptor.KAON);
        m_xmlWriter=new XMLWriter(writer,encoding,namespacesForAttributeValues,namespacesForElements);
        m_progressListener.processorProgress(PHASE_LOAD_OBJECTS,0,3);
        Set concepts=oimodel.getConcepts();
        collectNamespaces(concepts);
        checkInterrupted();
        m_progressListener.processorProgress(PHASE_LOAD_OBJECTS,1,3);
        Set properties=oimodel.getProperties();
        collectNamespaces(properties);
        checkInterrupted();
        m_progressListener.processorProgress(PHASE_LOAD_OBJECTS,2,3);
        Set instances=oimodel.getInstances();
        collectNamespaces(instances);
        checkInterrupted();
        m_progressListener.processorProgress(PHASE_LOAD_OBJECTS,3,3);
        m_xmlWriter.setBaseURI(oimodel.getLogicalURI());
        m_xmlWriter.printXMLDeclaration();
        m_xmlWriter.printDocumentDeclaration("owlx:Ontology");
        m_xmlWriter.println();
        m_xmlWriter.openTagIndent(Namespaces.OWLX_NS+"Ontology",true);
        m_xmlWriter.printAttributeRaw(Namespaces.OWLX_NS+"name",oimodel.getLogicalURI());
        m_xmlWriter.println();
        m_xmlWriter.indent();
        m_xmlWriter.printXMLBase();
        m_xmlWriter.printNamespaceDeclarations();
        m_xmlWriter.ensureTagBodyNewLine();
        m_xmlWriter.decreaseIndent();
        writeInclusions(oimodel);
        String version=oimodel.getAttribute("OIModel.version");
        if (version!=null) {
            m_xmlWriter.openTagIndent(Namespaces.OWLX_NS+"VersionInfo",false);
            m_xmlWriter.printData(version);
            m_xmlWriter.closeLastTag();
        }
        String evolutionLogPhysicalURI=oimodel.getAttribute("evolutionLog.physicalURI");
        if (evolutionLogPhysicalURI!=null) {
            m_xmlWriter.openTagIndent(Namespaces.OWLX_NS+"Annotation",true);
            m_xmlWriter.openTagIndent(KAONVocabularyAdaptor.KAON+"EvolutionLogPhysicalURI",false);
            m_xmlWriter.printData(evolutionLogPhysicalURI);
            m_xmlWriter.closeLastTag();
            m_xmlWriter.closeLastTag();
        }
        m_xmlWriter.ensureTagBodyNewLine();
        m_xmlWriter.println();
        processElements(concepts,oimodel,OIModel.LOAD_CONCEPT_BASICS | OIModel.LOAD_SUPER_CONCEPTS | OIModel.LOAD_PROPERTIES_FROM | OIModel.LOAD_LEXICON,new SingleObjectProcessor() {
            protected void processObject(Entity entity) throws KAONException,IOException {
                writeConcept((Concept)entity);
            }
        },PHASE_WRITE_CONCEPTS);
        concepts=null; // to free some memory
        processElements(properties,oimodel,OIModel.LOAD_PROPERTY_BASICS | OIModel.LOAD_SUPER_PROPERTIES | OIModel.LOAD_PROPERTY_DOMAINS | OIModel.LOAD_PROPERTY_RANGES | OIModel.LOAD_LEXICON,new SingleObjectProcessor() {
            protected void processObject(Entity entity) throws KAONException,IOException {
                writeProperty((Property)entity);
            }
        },PHASE_WRITE_PROPERTIES);
        properties=null; // to free some memory
        processElements(instances,oimodel,OIModel.LOAD_INSTANCE_BASICS | OIModel.LOAD_INSTANCE_PARENT_CONCEPTS | OIModel.LOAD_INSTANCE_FROM_PROPERTY_VALUES | OIModel.LOAD_LEXICON,new SingleObjectProcessor() {
            protected void processObject(Entity entity) throws KAONException,IOException {
                writeInstance((Instance)entity);
            }
        },PHASE_WRITE_INSTANCES);
        instances=null; // to free some memory
        m_xmlWriter.ensureTagBodyNewLine();
        m_xmlWriter.println();
        m_xmlWriter.closeLastTag();
        m_xmlWriter.ensureTagBodyNewLine();
        m_xmlWriter.flush();
    }
    /**
     * Writes the inclusion statements.
     *
     * @param oimodel                   the OI-model
     * @throws IOException              thrown if there is an I/O error
     * @throws KAONException            thrown if there is an error accessing the OI-model
     */
    protected void writeInclusions(OIModel oimodel) throws IOException,KAONException {
        Set includedOIModels=oimodel.getIncludedOIModels();
        if (!includedOIModels.isEmpty()) {
            Iterator iterator=includedOIModels.iterator();
            while (iterator.hasNext()) {
                OIModel includedOIModel=(OIModel)iterator.next();
                String importedURI=includedOIModel.getLogicalURI();
                if (!KAONConnection.ROOT_OIMODEL_URI.equals(importedURI) && !KAONConnection.LEXICAL_OIMODEL_URI.equals(importedURI)) {
                    m_xmlWriter.openTagIndent(Namespaces.OWLX_NS+"Imports",true);
                    m_xmlWriter.printAttributeURI(Namespaces.OWLX_NS+"ontology",importedURI);
                    m_xmlWriter.closeLastTag();
                }
            }
        }
    }
    /**
     * Processes the names of entities from given set.
     *
     * @param set                       the set
     * @throws KAONException            thrown if there is an error
     */
    protected void collectNamespaces(Set set) throws KAONException {
        Iterator iterator=set.iterator();
        while (iterator.hasNext()) {
            Entity entity=(Entity)iterator.next();
            String uri=entity.getURI();
            if (!uri.startsWith(KAONVocabularyAdaptor.KAON))
                m_xmlWriter.getNamespacesForAttributeValues().ensureNamespacePrefixExists(uri);
        }
    }
    /**
     * Writes the concept.
     *
     * @param concept                   the concept to write
     * @throws KAONException            thrown if there is an error
     * @throws IOException              thrown if there is an error
     */
    protected void writeConcept(Concept concept) throws KAONException,IOException {
        boolean conceptDeclaration=concept.isDeclaredLocally();
        Set superConcepts=new LinkedHashSet();
        Iterator iterator=concept.getSuperConcepts().iterator();
        while (iterator.hasNext()) {
            Concept superConcept=(Concept)iterator.next();
            if (!KAONVocabularyAdaptor.INSTANCE.getRoot().equals(superConcept.getURI()) && concept.isSuperSubConceptDeclaredLocally(superConcept))
                superConcepts.add(superConcept);
        }
        if (conceptDeclaration) {
            m_xmlWriter.openTagIndent(Namespaces.OWLX_NS+"Class",true);
            m_xmlWriter.printAttributeURI(Namespaces.OWLX_NS+"name",concept.getURI());
            m_xmlWriter.printAttributeRaw(Namespaces.OWLX_NS+"complete","false");
            writeLexicalEntries(concept);
            iterator=superConcepts.iterator();
            while (iterator.hasNext()) {
                Concept superConcept=(Concept)iterator.next();
                printConceptRef(superConcept);
            }
            iterator=concept.getPropertiesFromConcept().iterator();
            while (iterator.hasNext()) {
                Property property=(Property)iterator.next();
                int minCardinality=property.getMinimumCardinality(concept);
                if (minCardinality>0) {
                    m_xmlWriter.openTagIndent(Namespaces.OWLX_NS+"ObjectRestriction",true);
                    m_xmlWriter.printAttributeURI(Namespaces.OWLX_NS+"property",concept.getURI());
                    m_xmlWriter.openTagIndent(Namespaces.OWLX_NS+"minCardinality",true);
                    m_xmlWriter.printAttributeRaw(Namespaces.OWLX_NS+"value",String.valueOf(minCardinality));
                    m_xmlWriter.closeLastTag();
                    m_xmlWriter.closeLastTag();
                }
                int maxCardinality=property.getMaximumCardinality(concept);
                if (maxCardinality!=Integer.MAX_VALUE) {
                    m_xmlWriter.openTagIndent(Namespaces.OWLX_NS+"ObjectRestriction",true);
                    m_xmlWriter.printAttributeURI(Namespaces.OWLX_NS+"property",concept.getURI());
                    m_xmlWriter.openTagIndent(Namespaces.OWLX_NS+"maxCardinality",true);
                    m_xmlWriter.printAttributeRaw(Namespaces.OWLX_NS+"value",String.valueOf(maxCardinality));
                    m_xmlWriter.closeLastTag();
                    m_xmlWriter.closeLastTag();
                }
            }
            m_xmlWriter.closeLastTag();
        }
        else {
            iterator=superConcepts.iterator();
            while (iterator.hasNext()) {
                Concept superConcept=(Concept)iterator.next();
                m_xmlWriter.openTagIndent(Namespaces.OWLX_NS+"SubClassOf",true);
                m_xmlWriter.openTagIndent(Namespaces.OWLX_NS+"sub",true);
                printConceptRef(concept);
                m_xmlWriter.closeLastTag();
                m_xmlWriter.openTagIndent(Namespaces.OWLX_NS+"super",true);
                printConceptRef(superConcept);
                m_xmlWriter.closeLastTag();
                m_xmlWriter.closeLastTag();
            }
            iterator=concept.getPropertiesFromConcept().iterator();
            while (iterator.hasNext()) {
                Property property=(Property)iterator.next();
                int minCardinality=property.getMinimumCardinality(concept);
                if (minCardinality>0) {
                    m_xmlWriter.openTagIndent(Namespaces.OWLX_NS+"SubClassOf",true);
                    m_xmlWriter.openTagIndent(Namespaces.OWLX_NS+"sub",true);
                    printConceptRef(concept);
                    m_xmlWriter.closeLastTag();
                    m_xmlWriter.openTagIndent(Namespaces.OWLX_NS+"super",true);
                    m_xmlWriter.openTagIndent(Namespaces.OWLX_NS+"ObjectRestriction",true);
                    m_xmlWriter.printAttributeURI(Namespaces.OWLX_NS+"property",concept.getURI());
                    m_xmlWriter.openTagIndent(Namespaces.OWLX_NS+"minCardinality",true);
                    m_xmlWriter.printAttributeRaw(Namespaces.OWLX_NS+"value",String.valueOf(minCardinality));
                    m_xmlWriter.closeLastTag();
                    m_xmlWriter.closeLastTag();
                    m_xmlWriter.closeLastTag();
                    m_xmlWriter.closeLastTag();
                }
                int maxCardinality=property.getMaximumCardinality(concept);
                if (maxCardinality!=Integer.MAX_VALUE) {
                    m_xmlWriter.openTagIndent(Namespaces.OWLX_NS+"SubClassOf",true);
                    m_xmlWriter.openTagIndent(Namespaces.OWLX_NS+"sub",true);
                    printConceptRef(concept);
                    m_xmlWriter.closeLastTag();
                    m_xmlWriter.openTagIndent(Namespaces.OWLX_NS+"super",true);
                    m_xmlWriter.openTagIndent(Namespaces.OWLX_NS+"ObjectRestriction",true);
                    m_xmlWriter.printAttributeURI(Namespaces.OWLX_NS+"property",concept.getURI());
                    m_xmlWriter.openTagIndent(Namespaces.OWLX_NS+"maxCardinality",true);
                    m_xmlWriter.printAttributeRaw(Namespaces.OWLX_NS+"value",String.valueOf(minCardinality));
                    m_xmlWriter.closeLastTag();
                    m_xmlWriter.closeLastTag();
                    m_xmlWriter.closeLastTag();
                    m_xmlWriter.closeLastTag();
                }
            }
        }
    }
    /**
     * Writes the property.
     *
     * @param property                  the property to write
     * @throws KAONException            thrown if there is an error
     * @throws IOException              thrown if there is an error
     */
    protected void writeProperty(Property property) throws KAONException,IOException {
        boolean propertyDeclaration=property.isDeclaredLocally();
        Set superProperties=new LinkedHashSet();
        Iterator iterator=property.getSuperProperties().iterator();
        while (iterator.hasNext()) {
            Property superProperty=(Property)iterator.next();
            if (property.isSuperSubPropertyDeclaredLocally(superProperty))
                superProperties.add(superProperty);
        }
        Set domainConcepts=new LinkedHashSet();
        iterator=property.getDomainConcepts().iterator();
        while (iterator.hasNext()) {
            Concept concept=(Concept)iterator.next();
            if (property.isDomainConceptDeclaredLocally(concept))
                domainConcepts.add(concept);
        }
        Set rangeConcepts=new LinkedHashSet();
        iterator=property.getRangeConcepts().iterator();
        while (iterator.hasNext()) {
            Concept concept=(Concept)iterator.next();
            if (property.isRangeConceptDeclaredLocally(concept))
                rangeConcepts.add(concept);
        }
        if (propertyDeclaration || !domainConcepts.isEmpty() || !rangeConcepts.isEmpty()) {
            if (property.isAttribute())
                m_xmlWriter.openTagIndent(Namespaces.OWLX_NS+"DatatypeProperty",true);
            else
                m_xmlWriter.openTagIndent(Namespaces.OWLX_NS+"ObjectProperty",true);
            m_xmlWriter.printAttributeURI(Namespaces.OWLX_NS+"name",property.getURI());
            if (propertyDeclaration) {
                if (property.getInverseProperty()!=null)
                    m_xmlWriter.printAttributeURI(Namespaces.OWLX_NS+"inverseOf",property.getInverseProperty().getURI());
                if (property.isSymmetric())
                    m_xmlWriter.printAttributeRaw(Namespaces.OWLX_NS+"symmetric","true");
                if (property.isTransitive())
                    m_xmlWriter.printAttributeRaw(Namespaces.OWLX_NS+"transitive","true");
                writeLexicalEntries(property);
            }
            iterator=superProperties.iterator();
            while (iterator.hasNext()) {
                Property superProperty=(Property)iterator.next();
                m_xmlWriter.openTagIndent(Namespaces.OWLX_NS+"superProperty",true);
                m_xmlWriter.printAttributeURI(Namespaces.OWLX_NS+"name",superProperty.getURI());
                m_xmlWriter.closeLastTag();
            }
            iterator=domainConcepts.iterator();
            while (iterator.hasNext()) {
                Concept domainConcept=(Concept)iterator.next();
                m_xmlWriter.openTagIndent(Namespaces.OWLX_NS+"domain",true);
                m_xmlWriter.printAttributeURI(Namespaces.OWLX_NS+"class",domainConcept.getURI());
                m_xmlWriter.closeLastTag();
            }
            iterator=rangeConcepts.iterator();
            while (iterator.hasNext()) {
                Concept domainConcept=(Concept)iterator.next();
                m_xmlWriter.openTagIndent(Namespaces.OWLX_NS+"range",true);
                m_xmlWriter.printAttributeURI(Namespaces.OWLX_NS+"class",domainConcept.getURI());
                m_xmlWriter.closeLastTag();
            }
            if (property.isAttribute()) {
                m_xmlWriter.openTagIndent(Namespaces.OWLX_NS+"range",true);
                m_xmlWriter.printAttributeURI(Namespaces.OWLX_NS+"datatype",Namespaces.XSD_NS+"string");
                m_xmlWriter.closeLastTag();
            }
            m_xmlWriter.closeLastTag();
        }
        else if (!superProperties.isEmpty()) {
            m_xmlWriter.openTagIndent(Namespaces.OWLX_NS+"SubPropertyOf",true);
            m_xmlWriter.printAttributeURI(Namespaces.OWLX_NS+"sub",property.getURI());
            iterator=superProperties.iterator();
            while (iterator.hasNext()) {
                Property superProperty=(Property)iterator.next();
                if (property.isAttribute())
                    printDataPropertyRef(superProperty);
                else
                    printObjectPropertyRef(superProperty);
            }
            m_xmlWriter.closeLastTag();
        }
    }
    /**
     * Writes the instance.
     *
     * @param instance                  the instance to write
     * @throws KAONException            thrown if there is an error
     * @throws IOException              thrown if there is an error
     */
    protected void writeInstance(Instance instance) throws KAONException,IOException {
        boolean instanceDeclaration=!instance.getSpanningConcept().isInOIModel() && !instance.getSpanningProperty().isInOIModel() && instance.isDeclaredLocally();
        Set parentConcepts=new LinkedHashSet();
        Iterator iterator=instance.getParentConcepts().iterator();
        while (iterator.hasNext()) {
            Concept parentConcept=(Concept)iterator.next();
            if (!KAONVocabularyAdaptor.INSTANCE.getRoot().equals(parentConcept.getURI())) {
                if (parentConcept.getURI().startsWith(KAONVocabularyAdaptor.KAON))
                    return;
                if (instance.isConceptInstanceDeclaredLocally(parentConcept))
                    parentConcepts.add(parentConcept);
            }
        }
        Set propertyInstances=new LinkedHashSet();
        iterator=instance.getFromPropertyInstances().iterator();
        while (iterator.hasNext()) {
            PropertyInstance propertyInstance=(PropertyInstance)iterator.next();
            if (propertyInstance.isDeclaredLocally())
                propertyInstances.add(propertyInstance);
        }
        if (instanceDeclaration || !parentConcepts.isEmpty() || !propertyInstances.isEmpty()) {
            m_xmlWriter.openTagIndent(Namespaces.OWLX_NS+"Individual",true);
            m_xmlWriter.printAttributeURI(Namespaces.OWLX_NS+"name",instance.getURI());
            if (instanceDeclaration)
                writeLexicalEntries(instance);
            iterator=parentConcepts.iterator();
            while (iterator.hasNext()) {
                Concept parentConcept=(Concept)iterator.next();
                m_xmlWriter.openTagIndent(Namespaces.OWLX_NS+"type",true);
                m_xmlWriter.printAttributeURI(Namespaces.OWLX_NS+"name",parentConcept.getURI());
                m_xmlWriter.closeLastTag();
            }
            iterator=propertyInstances.iterator();
            while (iterator.hasNext()) {
                PropertyInstance propertyInstance=(PropertyInstance)iterator.next();
                Object targetValue=propertyInstance.getTargetValue();
                if (targetValue instanceof Instance) {
                    m_xmlWriter.openTagIndent(Namespaces.OWLX_NS+"ObjectPropertyValue",true);
                    m_xmlWriter.printAttributeURI(Namespaces.OWLX_NS+"property",propertyInstance.getProperty().getURI());
                    printInstancePropertyRef((Instance)targetValue);
                    m_xmlWriter.closeLastTag();
                }
                else {
                    m_xmlWriter.openTagIndent(Namespaces.OWLX_NS+"DataPropertyValue",true);
                    m_xmlWriter.printAttributeURI(Namespaces.OWLX_NS+"property",propertyInstance.getProperty().getURI());
                    printDataValue(targetValue);
                    m_xmlWriter.closeLastTag();
                }
            }
            m_xmlWriter.closeLastTag();
        }
    }
    /**
     * Processes the lexical entries attached to the entity.
     *
     * @param entity                    the entity
     * @throws KAONException            thrown if there is an error
     * @throws IOException              thrown if there is an error
     */
    protected void writeLexicalEntries(Entity entity) throws KAONException,IOException {
        Set lexicalEntries=entity.getLexicalEntries();
        if (!lexicalEntries.isEmpty()) {
            m_xmlWriter.openTagIndent(Namespaces.OWLX_NS+"Annotation",true);
            Iterator iterator=lexicalEntries.iterator();
            while (iterator.hasNext()) {
                LexicalEntry lexicalEntry=(LexicalEntry)iterator.next();
                String typeURI=lexicalEntry.getTypeURI();
                String languageCode=lexicalEntry.getInLanguage();
                if (languageCode!=null)
                    languageCode=languageCode.substring(KAONVocabularyAdaptor.KAON.length());
                if (KAONVocabularyAdaptor.INSTANCE.getKAONLabel().equals(typeURI))
                    m_xmlWriter.openTagIndent(Namespaces.OWLX_NS+"Label",false);
                else if (KAONVocabularyAdaptor.INSTANCE.getDocumentation().equals(typeURI))
                    m_xmlWriter.openTagIndent(Namespaces.OWLX_NS+"Documentation",false);
                else
                    m_xmlWriter.openTagIndent(typeURI,true);
                if (languageCode!=null)
                    m_xmlWriter.printAttributeRaw("xml:lang",languageCode);
                m_xmlWriter.printData(lexicalEntry.getValue());
                m_xmlWriter.closeLastTag();
            }
            m_xmlWriter.closeLastTag();
        }
    }
    /**
     * Prints the reference to a concept.
     *
     * @param concept                   the concpet to which the reference should be printed
     * @throws KAONException            thrown if there is an error
     */
    protected void printConceptRef(Concept concept) throws KAONException {
        m_xmlWriter.openTagIndent(Namespaces.OWLX_NS+"Class",true);
        m_xmlWriter.printAttributeURI(Namespaces.OWLX_NS+"name",concept.getURI());
        m_xmlWriter.closeLastTag();
    }
    /**
     * Prints the reference to an object property.
     *
     * @param property                  the object property to which the reference should be printed
     * @throws KAONException            thrown if there is an error
     */
    protected void printObjectPropertyRef(Property property) throws KAONException {
        m_xmlWriter.openTagIndent(Namespaces.OWLX_NS+"ObjectProperty",true);
        m_xmlWriter.printAttributeURI(Namespaces.OWLX_NS+"name",property.getURI());
        m_xmlWriter.closeLastTag();
    }
    /**
     * Prints the reference to a data property.
     *
     * @param property                  the data property to which the reference should be printed
     * @throws KAONException            thrown if there is an error
     */
    protected void printDataPropertyRef(Property property) throws KAONException {
        m_xmlWriter.openTagIndent(Namespaces.OWLX_NS+"DatatypeProperty",true);
        m_xmlWriter.printAttributeURI(Namespaces.OWLX_NS+"name",property.getURI());
        m_xmlWriter.closeLastTag();
    }
    /**
     * Prints the reference to an instance.
     *
     * @param instance                  the instance to which the reference should be printed
     * @throws KAONException            thrown if there is an error
     */
    protected void printInstancePropertyRef(Instance instance) throws KAONException {
        m_xmlWriter.openTagIndent(Namespaces.OWLX_NS+"Individual",true);
        m_xmlWriter.printAttributeURI(Namespaces.OWLX_NS+"name",instance.getURI());
        m_xmlWriter.closeLastTag();
    }
    /**
     * Prints the data value.
     *
     * @param value                     the data value that should be printed
     * @throws KAONException            thrown if there is an error
     */
    protected void printDataValue(Object value) throws KAONException {
        m_xmlWriter.openTagIndent(Namespaces.OWLX_NS+"DataValue",false);
        m_xmlWriter.printAttributeURI(Namespaces.OWLX_NS+"datatype",Namespaces.XSD_NS+"string");
        m_xmlWriter.printData(value.toString());
        m_xmlWriter.closeLastTag();
    }

    /**
     * The processor for single objects.
     */
    protected abstract class SingleObjectProcessor implements ObjectProcessor {
        public void processLoadedObjects(Set objects) throws KAONException,InterruptedException {
            try {
                Iterator iterator=objects.iterator();
                while (iterator.hasNext()) {
                    Entity entity=(Entity)iterator.next();
                    processObject(entity);
                    checkInterrupted();
                }
                checkInterrupted();
            }
            catch (IOException e) {
                throw new KAONException("I/O error",e);
            }
        }
        protected abstract void processObject(Entity entity) throws KAONException,IOException;
    }
}
