package edu.unika.aifb.kaon.apionrdf.exporter;

import java.io.IOException;
import java.io.Writer;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.Set;

import edu.unika.aifb.kaon.api.KAONException;
import edu.unika.aifb.kaon.api.oimodel.Concept;
import edu.unika.aifb.kaon.api.oimodel.Entity;
import edu.unika.aifb.kaon.api.oimodel.Instance;
import edu.unika.aifb.kaon.api.oimodel.KAONConnection;
import edu.unika.aifb.kaon.api.oimodel.LexicalEntry;
import edu.unika.aifb.kaon.api.oimodel.OIModel;
import edu.unika.aifb.kaon.api.oimodel.Property;
import edu.unika.aifb.kaon.api.oimodel.PropertyInstance;
import edu.unika.aifb.kaon.api.vocabulary.KAONVocabularyAdaptor;

/**
 * An exporter for wiritng OWL RDF ontologies.
 */
public class OWLRDFExporter extends AbstractExporter {

    /** The XML writer. */
    protected XMLWriter m_xmlWriter;

    /**
     * Creates an instance of this class.
     */
    public OWLRDFExporter() {
    }

    /**
     * Serializes given RDF model.
     *
     * @param oimodel
     *            the OI-model to serialize
     * @param physicalURI
     *            the physical URI of the exported file
     * @param writer
     *            the writer where serialization is performed
     * @param encoding
     *            tne encoding
     * @throws IOException
     *             thrown if there is an error
     * @throws KAONException
     *             thrown if there is an error accessing the OI-model
     * @throws InterruptedException
     *             thrown if the process is interrupted
     */
    public void export(OIModel oimodel, String physicalURI, Writer writer, String encoding) throws IOException, KAONException, InterruptedException {
        Namespaces namespacesForAttributeValues = new Namespaces();
        namespacesForAttributeValues.ensureNamespacePrefixExists(Namespaces.XSD_NS);
        namespacesForAttributeValues.ensureNamespacePrefixExists(Namespaces.OWL_NS);
        Namespaces namespacesForElements = new Namespaces();
        namespacesForElements.ensureNamespacePrefixExists(Namespaces.OWL_NS);
        namespacesForElements.ensureNamespacePrefixExists(Namespaces.RDF_NS);
        namespacesForElements.ensureNamespacePrefixExists(Namespaces.RDFS_NS);
        namespacesForElements.registerPrefix("kaon",KAONVocabularyAdaptor.KAON);
        namespacesForElements.setDefaultNamespace(oimodel.getLogicalURI() + "#");
        m_xmlWriter = new XMLWriter(writer, encoding, namespacesForAttributeValues, namespacesForElements);
        m_progressListener.processorProgress(PHASE_LOAD_OBJECTS, 0, 3);
        Set concepts = oimodel.getConcepts();
        collectNamespaces(concepts);
        checkInterrupted();
        m_progressListener.processorProgress(PHASE_LOAD_OBJECTS, 1, 3);
        Set properties = oimodel.getProperties();
        collectNamespaces(properties);
        checkInterrupted();
        m_progressListener.processorProgress(PHASE_LOAD_OBJECTS, 2, 3);
        Set instances = oimodel.getInstances();
        collectNamespaces(instances);
        checkInterrupted();
        m_progressListener.processorProgress(PHASE_LOAD_OBJECTS, 3, 3);
        m_xmlWriter.setBaseURI(oimodel.getLogicalURI());
        m_xmlWriter.printXMLDeclaration();
        m_xmlWriter.printDocumentDeclaration("rdf:RDF");
        m_xmlWriter.println();
        m_xmlWriter.openTagIndent(Namespaces.RDF_NS + "RDF", true);
        m_xmlWriter.println();
        m_xmlWriter.indent();
        m_xmlWriter.printXMLBase();
        m_xmlWriter.printNamespaceDeclarations();
        m_xmlWriter.ensureTagBodyNewLine();
        m_xmlWriter.decreaseIndent();
        m_xmlWriter.println();
        m_xmlWriter.openTagIndent(Namespaces.OWL_NS + "Ontology", true);
        m_xmlWriter.printAttributeRaw(Namespaces.RDF_NS + "about", oimodel.getLogicalURI());
        writeInclusions(oimodel);
        String version=oimodel.getAttribute("OIModel.version");
        if (version!=null)
            printOntologyAnnotationValue(Namespaces.OWL_NS + "versionInfo",version);
        String evolutionLogPhysicalURI=oimodel.getAttribute("evolutionLog.physicalURI");
        if (evolutionLogPhysicalURI!=null)
            printOntologyAnnotationValue(KAONVocabularyAdaptor.KAON + "evolutionLogPhysicalURI", evolutionLogPhysicalURI);
        m_xmlWriter.closeLastTag(); //Ontology
        m_xmlWriter.ensureTagBodyNewLine();
        m_xmlWriter.println();
        processElements(concepts, oimodel, OIModel.LOAD_CONCEPT_BASICS | OIModel.LOAD_SUPER_CONCEPTS | OIModel.LOAD_PROPERTIES_FROM | OIModel.LOAD_LEXICON, new SingleObjectProcessor() {

            protected void processObject(Entity entity) throws KAONException, IOException {
                writeConcept((Concept) entity);
            }
        }, PHASE_WRITE_CONCEPTS);
        concepts = null; // to free some memory
        processElements(properties, oimodel, OIModel.LOAD_PROPERTY_BASICS | OIModel.LOAD_SUPER_PROPERTIES | OIModel.LOAD_PROPERTY_DOMAINS | OIModel.LOAD_PROPERTY_RANGES
                | OIModel.LOAD_LEXICON, new SingleObjectProcessor() {

            protected void processObject(Entity entity) throws KAONException, IOException {
                writeProperty((Property) entity);
            }
        }, PHASE_WRITE_PROPERTIES);
        properties = null; // to free some memory
        processElements(instances, oimodel, OIModel.LOAD_INSTANCE_BASICS | OIModel.LOAD_INSTANCE_PARENT_CONCEPTS | OIModel.LOAD_INSTANCE_FROM_PROPERTY_VALUES
                | OIModel.LOAD_LEXICON, new SingleObjectProcessor() {

            protected void processObject(Entity entity) throws KAONException, IOException {
                writeInstance((Instance) entity);
            }
        }, PHASE_WRITE_INSTANCES);
        instances = null; // to free some memory
        m_xmlWriter.ensureTagBodyNewLine();
        m_xmlWriter.println();
        m_xmlWriter.ensureTagBodyNewLine();
        m_xmlWriter.closeLastTag(); //rdf:RDF
        m_xmlWriter.ensureTagBodyNewLine();
        m_xmlWriter.flush();
    }

    protected void printOntologyAnnotationValue(String annotationURI,String annotationValue) throws IOException {
        m_xmlWriter.openTagIndent(annotationURI, false);
        m_xmlWriter.printData(annotationValue);
        m_xmlWriter.closeLastTag();
    }

    /**
     * Writes the inclusion statements.
     *
     * @param oimodel
     *            the OI-model
     * @throws IOException
     *             thrown if there is an I/O error
     * @throws KAONException
     *             thrown if there is an error accessing the OI-model
     */
    protected void writeInclusions(OIModel oimodel) throws IOException, KAONException {
        Set includedOIModels = oimodel.getIncludedOIModels();
        if (!includedOIModels.isEmpty()) {
            Iterator iterator = includedOIModels.iterator();
            while (iterator.hasNext()) {
                OIModel includedOIModel = (OIModel) iterator.next();
                String importedURI = includedOIModel.getLogicalURI();
                if (!KAONConnection.ROOT_OIMODEL_URI.equals(importedURI) /*
                                                                          * &&
                                                                          * !KAONConnection.LEXICAL_OIMODEL_URI.equals(importedURI)
                                                                          */) {
                    m_xmlWriter.openTagIndent(Namespaces.OWL_NS + "imports", true);
                    m_xmlWriter.printAttributeURI(Namespaces.RDF_NS + "resource", importedURI);
                    m_xmlWriter.closeLastTag();
                }
            }
        }
    }

    /**
     * Processes the names of entities from given set.
     *
     * @param set
     *            the set
     * @throws KAONException
     *             thrown if there is an error
     */
    protected void collectNamespaces(Set set) throws KAONException {
        Iterator iterator = set.iterator();
        while (iterator.hasNext()) {
            Entity entity = (Entity) iterator.next();
            String uri = entity.getURI();
            if (!uri.startsWith(KAONVocabularyAdaptor.KAON)) m_xmlWriter.getNamespacesForAttributeValues().ensureNamespacePrefixExists(uri);
        }
    }

    /**
     * Writes the concept.
     *
     * @param concept
     *            the concept to write
     * @throws KAONException
     *             thrown if there is an error
     * @throws IOException
     *             thrown if there is an error
     */
    protected void writeConcept(Concept concept) throws KAONException, IOException {
        boolean conceptDeclaration = concept.isDeclaredLocally();
        Set superConcepts = new LinkedHashSet();
        Iterator iterator = concept.getSuperConcepts().iterator();
        while (iterator.hasNext()) {
            Concept superConcept = (Concept) iterator.next();
            if (!KAONVocabularyAdaptor.INSTANCE.getRoot().equals(superConcept.getURI()) && concept.isSuperSubConceptDeclaredLocally(superConcept)) {
                superConcepts.add(superConcept);
            }
        }

        if (conceptDeclaration || !superConcepts.isEmpty()) {

            m_xmlWriter.openTagIndent(Namespaces.OWL_NS + "Class", true);
            if (conceptDeclaration) {
                m_xmlWriter.printAttributeRaw(Namespaces.RDF_NS + "ID", getIDPart(concept.getURI()));
            } else {
                m_xmlWriter.printAttributeURI(Namespaces.RDF_NS + "about", concept.getURI());
            }
            if (conceptDeclaration) {
                writeLexicalEntries(concept);
            }
            iterator = superConcepts.iterator();
            while (iterator.hasNext()) {
                Concept superConcept = (Concept) iterator.next();
                printSuperConceptRef(superConcept);
            }
            /* FIXME Add property cardinalities */
            if (conceptDeclaration) {
                writePropertyCardinalities(concept);
            }
            m_xmlWriter.closeLastTag();
        }
    }

    /**
     * @param concept
     */
    private void writePropertyCardinalities(Concept concept) throws KAONException {
        Set propertiesFromConcept = concept.getPropertiesFromConcept();
        for (Iterator i = propertiesFromConcept.iterator(); i.hasNext();) {
            Property property = (Property) i.next();
            int minCardinality = property.getMinimumCardinality(concept);
            int maxCardinality = property.getMaximumCardinality(concept);
            if (minCardinality > 0) {
                m_xmlWriter.openTagIndent(Namespaces.RDFS_NS + "subClassOf", true);
                m_xmlWriter.openTagIndent(Namespaces.OWL_NS + "Restriction", true);
                m_xmlWriter.openTagIndent(Namespaces.OWL_NS + "minCardinality", true);
                m_xmlWriter.printAttributeURI(Namespaces.RDF_NS + "datatype", Namespaces.XSD_NS + "nonNegativeInteger");
                m_xmlWriter.printData("" + minCardinality);
                m_xmlWriter.closeLastTag();
                m_xmlWriter.openTagIndent(Namespaces.OWL_NS + "onProperty", true);
                if (property.isAttribute()) {
                    m_xmlWriter.openTagIndent(Namespaces.OWL_NS + "DatatypeProperty", true);
                } else {
                    m_xmlWriter.openTagIndent(Namespaces.OWL_NS + "ObjectProperty", true);
                }
                m_xmlWriter.printAttributeURI(Namespaces.RDF_NS + "about", property.getURI());
                m_xmlWriter.closeLastTag();
                m_xmlWriter.closeLastTag(); //onProperty
                m_xmlWriter.closeLastTag(); //Restriction
                m_xmlWriter.closeLastTag(); //subClassOf
            }
            if (maxCardinality < Integer.MAX_VALUE) {
                m_xmlWriter.openTagIndent(Namespaces.RDFS_NS + "subClassOf", true);
                m_xmlWriter.openTagIndent(Namespaces.OWL_NS + "Restriction", true);
                m_xmlWriter.openTagIndent(Namespaces.OWL_NS + "maxCardinality", true);
                m_xmlWriter.printAttributeURI(Namespaces.RDF_NS + "datatype", Namespaces.XSD_NS + "nonNegativeInteger");
                m_xmlWriter.printData("" + maxCardinality);
                m_xmlWriter.closeLastTag();
                m_xmlWriter.openTagIndent(Namespaces.OWL_NS + "onProperty", true);
                if (property.isAttribute()) {
                    m_xmlWriter.openTagIndent(Namespaces.OWL_NS + "DatatypeProperty", true);
                } else {
                    m_xmlWriter.openTagIndent(Namespaces.OWL_NS + "ObjectProperty", true);
                }
                m_xmlWriter.printAttributeURI(Namespaces.RDF_NS + "about", property.getURI());
                m_xmlWriter.closeLastTag();
                m_xmlWriter.closeLastTag(); //onProperty
                m_xmlWriter.closeLastTag(); //Restriction
                m_xmlWriter.closeLastTag(); //subClassOf
            }
        }

    }

    /**
     * Writes the property.
     *
     * @param property
     *            the property to write
     * @throws KAONException
     *             thrown if there is an error
     * @throws IOException
     *             thrown if there is an error
     */
    protected void writeProperty(Property property) throws KAONException, IOException {
        boolean propertyDeclaration = property.isDeclaredLocally();
        Set superProperties = new LinkedHashSet();
        Iterator iterator = property.getSuperProperties().iterator();
        while (iterator.hasNext()) {
            Property superProperty = (Property) iterator.next();
            if (property.isSuperSubPropertyDeclaredLocally(superProperty)) superProperties.add(superProperty);
        }
        Set domainConcepts = new LinkedHashSet();
        iterator = property.getDomainConcepts().iterator();
        while (iterator.hasNext()) {
            Concept concept = (Concept) iterator.next();
            if (property.isDomainConceptDeclaredLocally(concept)) domainConcepts.add(concept);
        }
        Set rangeConcepts = new LinkedHashSet();
        iterator = property.getRangeConcepts().iterator();
        while (iterator.hasNext()) {
            Concept concept = (Concept) iterator.next();
            if (property.isRangeConceptDeclaredLocally(concept)) rangeConcepts.add(concept);
        }

        if (propertyDeclaration || !domainConcepts.isEmpty() || !rangeConcepts.isEmpty() || !superProperties.isEmpty()) {
            if (property.isAttribute()) {
                m_xmlWriter.openTagIndent(Namespaces.OWL_NS + "DatatypeProperty", true);
            } else {
                m_xmlWriter.openTagIndent(Namespaces.OWL_NS + "ObjectProperty", true);
            }
            if (propertyDeclaration) {
                m_xmlWriter.printAttributeRaw(Namespaces.RDF_NS + "ID", getIDPart(property.getURI()));
            } else {
                m_xmlWriter.printAttributeURI(Namespaces.RDF_NS + "about", property.getURI());
            }
            if (propertyDeclaration) {
                writeLexicalEntries(property);
                if (property.getInverseProperty() != null) {
                    m_xmlWriter.openTagIndent(Namespaces.OWL_NS + "inverseOf", true);
                    m_xmlWriter.printAttributeURI(Namespaces.RDF_NS + "resource", property.getInverseProperty().getURI());
                    m_xmlWriter.closeLastTag();
                }
                if (property.isSymmetric()) {
                    m_xmlWriter.openTagIndent(Namespaces.RDF_NS + "type", true);
                    m_xmlWriter.printAttributeURI(Namespaces.RDF_NS + "resource", Namespaces.OWL_NS + "SymmetricProperty");
                    m_xmlWriter.closeLastTag();
                }
                if (property.isTransitive()) {
                    m_xmlWriter.openTagIndent(Namespaces.RDF_NS + "type", true);
                    m_xmlWriter.printAttributeURI(Namespaces.RDF_NS + "resource", Namespaces.OWL_NS + "TransitiveProperty");
                    m_xmlWriter.closeLastTag();
                }
                if (property.isAttribute()) {
                    m_xmlWriter.openTagIndent(Namespaces.RDFS_NS + "range", true);
                    m_xmlWriter.printAttributeURI(Namespaces.RDF_NS + "resource", Namespaces.XSD_NS + "string");
                    m_xmlWriter.closeLastTag();
                }
            }
            iterator = superProperties.iterator();
            while (iterator.hasNext()) {
                Property superProperty = (Property) iterator.next();
                m_xmlWriter.openTagIndent(Namespaces.RDFS_NS + "subPropertyOf", true);
                m_xmlWriter.printAttributeURI(Namespaces.RDF_NS + "resource", superProperty.getURI());
                m_xmlWriter.closeLastTag();
            }
            iterator = domainConcepts.iterator();
            while (iterator.hasNext()) {
                Concept domainConcept = (Concept) iterator.next();
                m_xmlWriter.openTagIndent(Namespaces.RDFS_NS + "domain", true);
                m_xmlWriter.printAttributeURI(Namespaces.RDF_NS + "resource", domainConcept.getURI());
                m_xmlWriter.closeLastTag();
            }
            iterator = rangeConcepts.iterator();
            while (iterator.hasNext()) {
                Concept domainConcept = (Concept) iterator.next();
                m_xmlWriter.openTagIndent(Namespaces.RDFS_NS + "range", true);
                m_xmlWriter.printAttributeURI(Namespaces.RDF_NS + "resource", domainConcept.getURI());
                m_xmlWriter.closeLastTag();
            }
            m_xmlWriter.closeLastTag();
        }
    }

    /**
     * Writes the instance.
     *
     * @param instance
     *            the instance to write
     * @throws KAONException
     *             thrown if there is an error
     * @throws IOException
     *             thrown if there is an error
     */
    protected void writeInstance(Instance instance) throws KAONException, IOException {
        boolean instanceDeclaration = !instance.getSpanningConcept().isInOIModel() && !instance.getSpanningProperty().isInOIModel() && instance.isDeclaredLocally();
        Set parentConcepts = new LinkedHashSet();
        Iterator iterator = instance.getParentConcepts().iterator();
        while (iterator.hasNext()) {
            Concept parentConcept = (Concept) iterator.next();
            if (!KAONVocabularyAdaptor.INSTANCE.getRoot().equals(parentConcept.getURI())) {
                if (parentConcept.getURI().startsWith(KAONVocabularyAdaptor.KAON)) return;
                if (instance.isConceptInstanceDeclaredLocally(parentConcept)) parentConcepts.add(parentConcept);
            }
        }
        Set propertyInstances = new LinkedHashSet();
        iterator = instance.getFromPropertyInstances().iterator();
        while (iterator.hasNext()) {
            PropertyInstance propertyInstance = (PropertyInstance) iterator.next();
            if (propertyInstance.isDeclaredLocally()) propertyInstances.add(propertyInstance);
        }
        if (instanceDeclaration || !parentConcepts.isEmpty() || !propertyInstances.isEmpty()) {
            m_xmlWriter.openTagIndent(Namespaces.RDF_NS + "Description", true);
            if (instanceDeclaration) {
                m_xmlWriter.printAttributeRaw(Namespaces.RDF_NS + "ID", getIDPart(instance.getURI()));
                writeLexicalEntries(instance);
            } else {
                m_xmlWriter.printAttributeURI(Namespaces.RDF_NS + "about", instance.getURI());
            }
            iterator = parentConcepts.iterator();
            while (iterator.hasNext()) {
                Concept parentConcept = (Concept) iterator.next();
                m_xmlWriter.openTagIndent(Namespaces.RDF_NS + "type", true);
                m_xmlWriter.printAttributeURI(Namespaces.RDF_NS + "resource", parentConcept.getURI());
                m_xmlWriter.closeLastTag();
            }
            iterator = propertyInstances.iterator();
            while (iterator.hasNext()) {
                PropertyInstance propertyInstance = (PropertyInstance) iterator.next();
                Object targetValue = propertyInstance.getTargetValue();
                m_xmlWriter.openTagIndent(propertyInstance.getProperty().getURI(), true);
                if (targetValue instanceof Instance) {
                    m_xmlWriter.printAttributeURI(Namespaces.RDF_NS + "resource", ((Instance) targetValue).getURI());
                } else {
                    m_xmlWriter.printAttributeURI(Namespaces.RDF_NS + "datatype", Namespaces.XSD_NS + "string");
                    m_xmlWriter.printData(targetValue.toString());
                }
                m_xmlWriter.closeLastTag();
            }
            m_xmlWriter.closeLastTag();
        }
    }

    /**
     * Processes the lexical entries attached to the entity.
     *
     * @param entity
     *            the entity
     * @throws KAONException
     *             thrown if there is an error
     * @throws IOException
     *             thrown if there is an error
     */
    protected void writeLexicalEntries(Entity entity) throws KAONException, IOException {
        Set lexicalEntries = entity.getLexicalEntries();
        if (!lexicalEntries.isEmpty()) {
            Iterator iterator = lexicalEntries.iterator();
            while (iterator.hasNext()) {
                LexicalEntry lexicalEntry = (LexicalEntry) iterator.next();
                String typeURI = lexicalEntry.getTypeURI();
                String languageCode = lexicalEntry.getInLanguage();
                if (languageCode != null) languageCode = languageCode.substring(KAONVocabularyAdaptor.KAON.length());
                if (KAONVocabularyAdaptor.INSTANCE.getKAONLabel().equals(typeURI))
                    m_xmlWriter.openTagIndent(Namespaces.RDFS_NS + "label", false);
                else if (KAONVocabularyAdaptor.INSTANCE.getDocumentation().equals(typeURI))
                    m_xmlWriter.openTagIndent(Namespaces.RDFS_NS + "comment", false);
                else
                    m_xmlWriter.openTagIndent(typeURI, false);
                if (languageCode != null) m_xmlWriter.printAttributeRaw("xml:lang", languageCode);
                m_xmlWriter.printData(lexicalEntry.getValue());
                m_xmlWriter.closeLastTag();
            }
        }
    }

    /**
     * Prints a rdfs:subClassOf reference to a superconcept.
     *
     * @param concept
     *            the superconcept to which the subclassof reference should be
     *            printed
     * @throws KAONException
     *             thrown if there is an error
     */
    protected void printSuperConceptRef(Concept concept) throws KAONException {
        m_xmlWriter.openTagIndent(Namespaces.RDFS_NS + "subClassOf", true);
        m_xmlWriter.printAttributeURI(Namespaces.RDF_NS + "resource", concept.getURI());
        m_xmlWriter.closeLastTag();
    }

    /**
     * Returns the ID part of an URI (the part after the last #)
     *
     * @param uriString
     *            the original URI
     * @return the part after the #
     */
    private String getIDPart(String uriString) {
        return uriString.subSequence(uriString.lastIndexOf('#') + 1, uriString.length()).toString();
    }

    /**
     * The processor for single objects.
     */
    protected abstract class SingleObjectProcessor implements ObjectProcessor {

        public void processLoadedObjects(Set objects) throws KAONException, InterruptedException {
            try {
                Iterator iterator = objects.iterator();
                while (iterator.hasNext()) {
                    Entity entity = (Entity) iterator.next();
                    processObject(entity);
                    checkInterrupted();
                }
                checkInterrupted();
            } catch (IOException e) {
                throw new KAONException("I/O error", e);
            }
        }

        protected abstract void processObject(Entity entity) throws KAONException, IOException;
    }
}
