package edu.unika.aifb.kaon.apionrdf.exporter;

import java.util.Map;
import java.util.TreeMap;
import java.util.Iterator;
import java.util.Collections;

/**
 * This interface contains some well-known namespaces.
 */
public class Namespaces {
    /** The namespace for OWL ontologies. */
    public static final String OWL_NS="http://www.w3.org/2002/07/owl#";
    /** The namespace for OWL XML syntax. */
    public static final String OWLX_NS="http://www.w3.org/2003/05/owl-xml#";
    /** The namespace for XSD datatypes. */
    public static final String XSD_NS="http://www.w3.org/2001/XMLSchema#";
    /** The namespace for RDF elements. */
    public static final String RDF_NS="http://www.w3.org/1999/02/22-rdf-syntax-ns#";
    /** The namespace for RDFS elements. */
    public static final String RDFS_NS="http://www.w3.org/2000/01/rdf-schema#";
    /** The map of well-known namespaces and prefixes. */
    protected static final Map s_wellKnownNamespaces=new TreeMap();
    static {
        s_wellKnownNamespaces.put("owl",OWL_NS);
        s_wellKnownNamespaces.put("owlx",OWLX_NS);
        s_wellKnownNamespaces.put("xsd",XSD_NS);
        s_wellKnownNamespaces.put("rdf",RDF_NS);
        s_wellKnownNamespaces.put("rdfs",RDFS_NS);
    }

    /** The global static instance. */
    public static final Namespaces INSTANCE=new Namespaces();
    static {
        INSTANCE.registerStandardPrefixes();
    }

    /** The default namespace, of <code>null</code> if none is set. */
    protected String m_defaultNamespace;
    /** The map of prefixes to the corresponding URI. */
    protected final Map m_namespaceByPrefix;
    /** The map of URIs to prefixes. */
    protected final Map m_prefixByNamespace;
    /** The index of the next automatic prefix. */
    protected int m_nextAutomaticPrefix;

    /**
     * Creates an instance of this class not containing any mappings.
     */
    public Namespaces() {
        m_namespaceByPrefix=new TreeMap();
        m_prefixByNamespace=new TreeMap();
        m_nextAutomaticPrefix=0;
    }
    /**
     * Registers started prefixes to this object.
     */
    public synchronized void registerStandardPrefixes() {
        Iterator entries=s_wellKnownNamespaces.entrySet().iterator();
        while (entries.hasNext()) {
            Map.Entry entry=(Map.Entry)entries.next();
            registerPrefix((String)entry.getKey(),(String)entry.getValue());
        }
     }
    /**
     * Sets the default namespace. It is used only in abbreviateAsNamespace method.
     *
     * @param defaultNamespace                     the default namespace
     */
    public synchronized void setDefaultNamespace(String defaultNamespace) {
        m_defaultNamespace=defaultNamespace;
    }
    /**
     * Returns the default namespace.
     *
     * @return                                  the default namespace
     */
    public synchronized String getDefaultNamespace() {
        return m_defaultNamespace;
    }
    /**
     * Deregisters a prefix.
     *
     * @param prefix                            the prefix of the URI
     */
    public synchronized void unregisterPrefix(String prefix) {
        String namespace=(String)m_namespaceByPrefix.remove(prefix);
        m_prefixByNamespace.remove(namespace);
    }
    /**
     * Registers a prefix for the URI.
     *
     * @param prefix                            the prefix of the URI
     * @param namespace                         the namespace URI
     */
    public synchronized void registerPrefix(String prefix,String namespace) {
        m_namespaceByPrefix.put(prefix,namespace);
        m_prefixByNamespace.put(namespace,prefix);
    }
    /**
     * Returns the namespace URI for the given prefix.
     *
     * @param prefix                            the prefix
     * @return                                  the namespace URI for the prefix (or <code>null</code> if the namespace for the prefix is not registered)
     */
    public synchronized String getNamespaceForPrefix(String prefix) {
        return (String)m_namespaceByPrefix.get(prefix);
    }
    /**
     * Returns the prefix for the given namespace URI.
     *
     * @param namespace                         the namespace URI
     * @return                                  the prefix for the namespace URI (or <code>null</code> if the prefix for the namespace is not registered)
     */
    public synchronized String getPrefixForNamespace(String namespace) {
        return (String)m_prefixByNamespace.get(namespace);
    }
    /**
     * Returns the prefix used to abbreviate the URI.
     *
     * @param uri                                 the URI
     * @return                                     the prefix, or <code>null</code> if the URI cannot be abbreviated
     */
    public synchronized String getAbbreviationPrefix(String uri) {
        String namespace=guessNamespace(uri);
        if (namespace.length()==0)
            return null;
        return getPrefixForNamespace(namespace);
    }
    /**
     * Abbreviates given URI into the form prefix:local_name if possible.
     *
     * @param uri                               the URI
     * @return                                  the abbreviated form, or the original URI if abbreviation is not possible
     */
    public synchronized String abbreviateAsNamespace(String uri) {
        int namespaceEnd=guessNamespaceEnd(uri);
        if (namespaceEnd<0)
            return uri;
        String namespace=uri.substring(0,namespaceEnd+1);
        String localName=uri.substring(namespaceEnd+1);
        if (namespace.equals(m_defaultNamespace))
            return localName;
        String prefix=getPrefixForNamespace(namespace);
        if (prefix==null)
            return uri;
        return prefix+":"+localName;
    }
    /**
     * Abbreviates given URI into the form prefix:local_name if possible.
     *
     * @param uri                               the URI
     * @return                                  the abbreviated form, or the original URI if abbreviation is not possible
     */
    public synchronized String abbreviateAsNamespaceNoDefault(String uri) {
        int namespaceEnd=guessNamespaceEnd(uri);
        if (namespaceEnd<0)
            return uri;
        String namespace=uri.substring(0,namespaceEnd+1);
        String localName=uri.substring(namespaceEnd+1);
        String prefix=getPrefixForNamespace(namespace);
        if (prefix==null)
            return uri;
        return prefix+":"+localName;
    }
    /**
     * Abbreviates given URI into the form &prefix;local_name if possible.
     *
     * @param uri                               the URI
     * @return                                  the abbreviated form, or the original URI if abbreviation is not possible
     */
    public synchronized String abbreviateAsEntity(String uri) {
        int namespaceEnd=guessNamespaceEnd(uri);
        if (namespaceEnd<0)
            return uri;
        String prefix=getPrefixForNamespace(uri.substring(0,namespaceEnd+1));
        if (prefix==null)
            return uri;
        return "&"+prefix+";"+uri.substring(namespaceEnd+1);
    }
    /**
     * Attempts to expand given string (either of the form prefix:local_name or of the form &prefix;local_name) into an URI.
     *
     * @param string                            the string
     * @return                                  the expanded URI
     */
    public synchronized String expandString(String string) {
        if (string.length()>0 && string.charAt(0)=='&') {
            int lastSemicolonPosition=string.lastIndexOf(';');
            if (lastSemicolonPosition>=0) {
                String prefix=string.substring(1,lastSemicolonPosition);
                String namespace=getNamespaceForPrefix(prefix);
                if (namespace!=null)
                    return namespace+string.substring(lastSemicolonPosition+1);
            }
        }
        int lastColonPosition=string.lastIndexOf(':');
        if (lastColonPosition>=0) {
            String prefix=string.substring(0,lastColonPosition);
            String namespace=getNamespaceForPrefix(prefix);
            if (namespace!=null)
                return namespace+string.substring(lastColonPosition+1);
        }
        return string;
    }
    /**
     * Returns the iterator of all prefixes.
     *
     * @return                                  all prefixes
     */
    public synchronized Iterator prefixes() {
        return Collections.unmodifiableSet(m_namespaceByPrefix.keySet()).iterator();
    }
    /**
     * Makes sure that a prefix for given uri exists. If a prefix for this URI does not exist,
     * a new prefix is generated.
     *
     * @param uri                                 the URI
     */
    public synchronized void ensureNamespacePrefixExists(String uri) {
        String namespace=guessNamespace(uri);
        if (namespace.length()!=0) {
            String prefix=getPrefixForNamespace(namespace);
            if (prefix==null) {
                Iterator entries=s_wellKnownNamespaces.entrySet().iterator();
                while (entries.hasNext()) {
                    Map.Entry entry=(Map.Entry)entries.next();
                    if (entry.getValue().equals(namespace) && getNamespaceForPrefix((String)entry.getKey())==null) {
                        prefix=(String)entry.getKey();
                        break;
                    }
                }
                if (prefix==null)
                    do {
                        prefix=getNextNamespacePrefix();
                    } while (getNamespaceForPrefix(prefix)!=null);
                registerPrefix(prefix,namespace);
            }
        }
    }
    /**
     * Returns the next new namespace prefix.
     *
     * @return                                  the next new namespace prefix
     */
    protected String getNextNamespacePrefix() {
        StringBuffer buffer=new StringBuffer();
        int index=m_nextAutomaticPrefix++;
        do {
            buffer.append((char)('a'+(index % 26)));
            index=index/26;
        } while (index!=0);
        return buffer.toString();
    }
    /**
     * Returns the index of the last characted of the namespace.
     *
     * @param uri                               the URI of the namespace
     * @return                                  the index of the last characted of the namespace
     */
    public static int guessNamespaceEnd(String uri) {
        for (int i=uri.length()-1;i>=0;i--) {
            char c=uri.charAt(i);
            if (c=='#' || c==':' || c=='/')
                return i;
        }
        return -1;
    }
    /**
     * Guesses a namespace prefix of a URI.
     *
     * @param uri                               the URI for which the namespace prefix is guessed
     * @return                                  the namespace prefix or empty string if the prefix cannot be guessed
     */
    public static String guessNamespace(String uri) {
        int index=guessNamespaceEnd(uri);
        return index>=0 ? uri.substring(0,index+1) : "";
    }
    /**
     * Guesses the local name of a URI.
     *
     * @param uri                               the URI for which the local name is guessed
     * @return                                  the local name or the whole URI if the local name cannot be guessed
     */
    public static String guessLocalName(String uri) {
        return uri.substring(guessNamespaceEnd(uri)+1);
    }
}
